"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const path = require("path");
const apigateway = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                apiGETECF0BD67: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "GET",
                        ResourceId: {
                            "Fn::GetAtt": [
                                "apiC8550315",
                                "RootResourceId"
                            ]
                        },
                        RestApiId: {
                            Ref: "apiC8550315"
                        },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                },
                apiC8550315: {
                    Type: "AWS::ApiGateway::RestApi",
                    Properties: {
                        Name: "api"
                    }
                },
                deployment33381975: {
                    Type: "AWS::ApiGateway::Deployment",
                    Properties: {
                        RestApiId: {
                            Ref: "apiC8550315"
                        }
                    },
                    DependsOn: [
                        "apiGETECF0BD67"
                    ]
                }
            }
        });
        test.done();
    },
    '"retainDeployments" can be used to control the deletion policy of the resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api, retainDeployments: true });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                apiGETECF0BD67: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "GET",
                        ResourceId: {
                            "Fn::GetAtt": [
                                "apiC8550315",
                                "RootResourceId"
                            ]
                        },
                        RestApiId: {
                            Ref: "apiC8550315"
                        },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                },
                apiC8550315: {
                    Type: "AWS::ApiGateway::RestApi",
                    Properties: {
                        Name: "api"
                    }
                },
                deployment33381975: {
                    Type: "AWS::ApiGateway::Deployment",
                    Properties: {
                        RestApiId: {
                            Ref: "apiC8550315"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                    DependsOn: [
                        "apiGETECF0BD67"
                    ]
                }
            }
        });
        test.done();
    },
    '"description" can be set on the deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Deployment(stack, 'deployment', { api, description: 'this is my deployment' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            Description: 'this is my deployment'
        }));
        test.done();
    },
    '"addToLogicalId" will "salt" the logical ID of the deployment resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        const deployment = new apigateway.Deployment(stack, 'deployment', { api });
        api.root.addMethod('GET');
        // default logical ID (with no "salt")
        test.ok(synthesize().Resources.deployment33381975);
        // adding some salt
        deployment.addToLogicalId({ foo: 123 }); // add some data to the logical ID
        // the logical ID changed
        const template = synthesize();
        test.ok(!template.Resources.deployment33381975, 'old resource id deleted');
        test.ok(template.Resources.deployment33381975427670fa9e4148dc851927485bdf36a5, 'new resource is created');
        // tokens supported, and are resolved upon synthesis
        const value = 'hello hello';
        deployment.addToLogicalId({ foo: core_1.Lazy.stringValue({ produce: () => value }) });
        const template2 = synthesize();
        test.ok(template2.Resources.deployment33381975a12dfe81474913364dc31c06e37f9449);
        test.done();
        function synthesize() {
            return assert_1.SynthUtils.synthesize(stack).template;
        }
    },
    '"addDependency" can be used to add a resource as a dependency'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigateway.RestApi(stack, 'api', { deploy: false, cloudWatchRole: false });
        const deployment = new apigateway.Deployment(stack, 'deployment', { api });
        api.root.addMethod('GET');
        const dep = new core_1.CfnResource(stack, 'MyResource', { type: 'foo' });
        // WHEN
        deployment.node.addDependency(dep);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            DependsOn: [
                "apiGETECF0BD67",
                "MyResource"
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'integration change invalidates deployment'(test) {
        // GIVEN
        const stack1 = new core_1.Stack();
        const stack2 = new core_1.Stack();
        const handler1 = new lambda.Function(stack1, 'handler1', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        });
        const handler2 = new lambda.Function(stack2, 'handler2', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        });
        // WHEN
        const api1 = new apigateway.RestApi(stack1, 'myapi', {
            defaultIntegration: new apigateway.LambdaIntegration(handler1)
        });
        const api2 = new apigateway.RestApi(stack2, 'myapi', {
            defaultIntegration: new apigateway.LambdaIntegration(handler2)
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        // THEN
        assert_1.expect(stack1).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            DeploymentId: { Ref: 'myapiDeploymentB7EF8EB7e0b8372768854261d2d1218739e0a307' }
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            DeploymentId: { Ref: 'myapiDeploymentB7EF8EB77c517352b0f7ab73c333e36585c8f1f3' }
        }));
        test.done();
    },
    'deployment resource depends on all restapi methods defined'(test) {
        const stack = new core_1.Stack();
        const restapi = new apigateway.RestApi(stack, 'myapi', {
            deploy: false
        });
        restapi.root.addMethod('GET');
        const deployment = new apigateway.Deployment(stack, 'mydeployment', {
            api: restapi
        });
        const stage = new apigateway.Stage(stack, 'mystage', {
            deployment
        });
        restapi.deploymentStage = stage;
        restapi.root.addMethod('POST');
        const resource = restapi.root.addResource('myresource');
        resource.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Deployment', {
            DependsOn: [
                'myapiGET9B7CD29E',
                'myapimyresourceGET732851A5',
                'myapiPOST23417BD2'
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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