"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'my-api');
        const handler = new lambda.Function(stack, 'Handler', {
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'boom',
            code: lambda.Code.fromInline('foo')
        });
        // WHEN
        const integ = new apigateway.LambdaIntegration(handler);
        api.root.addMethod('GET', integ);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: "POST",
                Type: "AWS_PROXY",
                Uri: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            {
                                Ref: "AWS::Partition"
                            },
                            ":apigateway:",
                            {
                                Ref: "AWS::Region"
                            },
                            ":lambda:path/2015-03-31/functions/",
                            {
                                "Fn::GetAtt": [
                                    "Handler886CB40B",
                                    "Arn"
                                ]
                            },
                            "/invocations"
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    '"allowTestInvoke" can be used to disallow calling the API from the test UI'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Handler', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler'
        });
        const api = new apigateway.RestApi(stack, 'api');
        // WHEN
        const integ = new apigateway.LambdaIntegration(fn, { allowTestInvoke: false });
        api.root.addMethod('GET', integ);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            SourceArn: {
                "Fn::Join": [
                    "",
                    [
                        "arn:", { Ref: "AWS::Partition" }, ":execute-api:", { Ref: "AWS::Region" }, ":", { Ref: "AWS::AccountId" }, ":",
                        { Ref: "apiC8550315" }, "/", { Ref: "apiDeploymentStageprod896C8101" }, "/GET/"
                    ]
                ]
            }
        }));
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::Lambda::Permission', {
            SourceArn: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        { Ref: "AWS::Partition" },
                        ":execute-api:",
                        { Ref: "AWS::Region" },
                        ":",
                        { Ref: "AWS::AccountId" },
                        ":",
                        { Ref: "apiC8550315" },
                        "/test-invoke-stage/GET/"
                    ]
                ]
            }
        })));
        test.done();
    },
    '"proxy" can be used to disable proxy mode'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Handler', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler'
        });
        const api = new apigateway.RestApi(stack, 'api');
        // WHEN
        const integ = new apigateway.LambdaIntegration(fn, { proxy: false });
        api.root.addMethod('GET', integ);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Type: 'AWS'
            }
        }));
        test.done();
    },
    'when "ANY" is used, lambda permission will include "*" for method'(test) {
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api');
        const handler = new lambda.Function(stack, 'MyFunc', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`loo`)
        });
        const target = new apigateway.LambdaIntegration(handler);
        api.root.addMethod('ANY', target);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            SourceArn: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        { Ref: "AWS::Partition" },
                        ":execute-api:",
                        { Ref: "AWS::Region" },
                        ":",
                        { Ref: "AWS::AccountId" },
                        ":",
                        { Ref: "testapiD6451F70" },
                        "/test-invoke-stage/*/"
                    ]
                ]
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            SourceArn: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":execute-api:",
                        {
                            Ref: "AWS::Region"
                        },
                        ":",
                        {
                            Ref: "AWS::AccountId"
                        },
                        ":",
                        {
                            Ref: "testapiD6451F70"
                        },
                        "/",
                        { Ref: "testapiDeploymentStageprod5C9E92A4" },
                        "/*/"
                    ]
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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