"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigw = require("../lib");
const DUMMY_AUTHORIZER = {
    authorizerId: 'dummyauthorizer',
    authorizationType: apigw.AuthorizationType.CUSTOM
};
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: "POST",
            AuthorizationType: "NONE",
            Integration: {
                Type: "MOCK"
            }
        }));
        test.done();
    },
    'method options can be specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                operationName: 'MyOperation',
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            OperationName: "MyOperation"
        }));
        test.done();
    },
    'integration can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key' })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: "POST",
                Type: "AWS",
                Uri: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:", { Ref: "AWS::Partition" }, ":apigateway:",
                            { Ref: "AWS::Region" }, ":s3:path/bucket/key"
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    'integration with a custom http method can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key', integrationHttpMethod: 'GET' })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: "GET"
            }
        }));
        test.done();
    },
    'use default integration from api'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const defaultIntegration = new apigw.Integration({ type: apigw.IntegrationType.HTTP_PROXY, uri: 'https://amazon.com' });
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultIntegration
        });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                Type: "HTTP_PROXY",
                Uri: 'https://amazon.com'
            }
        }));
        test.done();
    },
    '"methodArn" returns the ARN execute-api ARN for this method in the current stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    { Ref: "AWS::Partition" },
                    ":execute-api:",
                    { Ref: "AWS::Region" },
                    ":",
                    { Ref: "AWS::AccountId" },
                    ":",
                    { Ref: "testapiD6451F70" },
                    "/",
                    { Ref: "testapiDeploymentStageprod5C9E92A4" },
                    "/POST/"
                ]
            ]
        });
        test.done();
    },
    '"testMethodArn" returns the ARN of the "test-invoke-stage" stage (console UI)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.testMethodArn), {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    { Ref: "AWS::Partition" },
                    ":execute-api:",
                    { Ref: "AWS::Region" },
                    ":",
                    { Ref: "AWS::AccountId" },
                    ":",
                    { Ref: "testapiD6451F70" },
                    "/test-invoke-stage/POST/"
                ]
            ]
        });
        test.done();
    },
    '"methodArn" fails if the API does not have a deployment stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const method = new apigw.Method(stack, 'my-method', { httpMethod: 'POST', resource: api.root });
        // WHEN + THEN
        test.throws(() => method.methodArn, /Unable to determine ARN for method "my-method" since there is no stage associated with this API./);
        test.done();
    },
    'integration "credentialsRole" can be used to assume a role when calling backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsRole: role
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { "Fn::GetAtt": ["MyRoleF48FFE04", "Arn"] }
            }
        }));
        test.done();
    },
    'integration "credentialsPassthrough" can be used to passthrough user credentials to backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::*:user/*"]] }
            }
        }));
        test.done();
    },
    'integration "credentialsRole" and "credentialsPassthrough" are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true,
                credentialsRole: role
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'credentialsPassthrough' and 'credentialsRole' are mutually exclusive/);
        test.done();
    },
    'integration connectionType VpcLink requires vpcLink to be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.VPC_LINK,
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'connectionType' of VPC_LINK requires 'vpcLink' prop to be set/);
        test.done();
    },
    'connectionType of INTERNET and vpcLink are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const vpc = new ec2.Vpc(stack, 'VPC');
        const nlb = new elbv2.NetworkLoadBalancer(stack, 'NLB', {
            vpc
        });
        const link = new apigw.VpcLink(stack, 'link', {
            targets: [nlb]
        });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.INTERNET,
                vpcLink: link
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /cannot set 'vpcLink' where 'connectionType' is INTERNET/);
        test.done();
    },
    'methodResponse set one or more method responses via options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200'
                    }, {
                        statusCode: "400",
                        responseParameters: {
                            'method.response.header.killerbees': false
                        }
                    }, {
                        statusCode: "500",
                        responseParameters: {
                            'method.response.header.errthing': true
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL
                        }
                    }
                ]
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: "200"
                }, {
                    StatusCode: "400",
                    ResponseParameters: {
                        'method.response.header.killerbees': false
                    }
                }, {
                    StatusCode: "500",
                    ResponseParameters: {
                        'method.response.header.errthing': true
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error'
                    }
                }
            ]
        }));
        test.done();
    },
    'multiple integration responses can be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.AwsIntegration({
            service: 'foo-service',
            action: 'BarAction',
            options: {
                integrationResponses: [
                    {
                        statusCode: '200',
                        responseTemplates: { 'application/json': JSON.stringify({ success: true }) },
                    },
                    {
                        selectionPattern: 'Invalid',
                        statusCode: '503',
                        responseTemplates: { 'application/json': JSON.stringify({ success: false, message: 'Invalid Request' }) },
                    }
                ],
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                IntegrationResponses: [
                    {
                        ResponseTemplates: { 'application/json': '{"success":true}' },
                        StatusCode: '200',
                    },
                    {
                        ResponseTemplates: { 'application/json': '{"success":false,"message":"Invalid Request"}' },
                        SelectionPattern: 'Invalid',
                        StatusCode: '503',
                    }
                ],
                Type: 'AWS',
                Uri: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':apigateway:', { Ref: 'AWS::Region' }, ':foo-service:action/BarAction']] }
            }
        }));
        test.done();
    },
    'method is always set as uppercase'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        api.root.addMethod('get');
        api.root.addMethod('PoSt');
        api.root.addMethod('PUT');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: "POST" }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: "GET" }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: "PUT" }));
        test.done();
    },
    'requestModel can be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const model = api.addModel('test-model', {
            contentType: "application/json",
            modelName: 'test-model',
            schema: {
                title: "test",
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } }
            }
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestModels: {
                    "application/json": model
                }
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            RequestModels: {
                "application/json": { Ref: stack.getLogicalId(model.node.findChild('Resource')) }
            }
        }));
        test.done();
    },
    'methodResponse has a mix of response modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const htmlModel = api.addModel('my-model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: "test",
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } }
            }
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200'
                    }, {
                        statusCode: "400",
                        responseParameters: {
                            'method.response.header.killerbees': false
                        }
                    }, {
                        statusCode: "500",
                        responseParameters: {
                            'method.response.header.errthing': true
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                            'text/html': htmlModel
                        }
                    }
                ]
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: "200"
                }, {
                    StatusCode: "400",
                    ResponseParameters: {
                        'method.response.header.killerbees': false
                    }
                }, {
                    StatusCode: "500",
                    ResponseParameters: {
                        'method.response.header.errthing': true
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                        'text/html': { Ref: stack.getLogicalId(htmlModel.node.findChild('Resource')) }
                    }
                }
            ]
        }));
        test.done();
    },
    'method has a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('validator', {
            validateRequestBody: true,
            validateRequestParameters: false
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidator: validator
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: { Ref: stack.getLogicalId(validator.node.findChild('Resource')) }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            ValidateRequestBody: true,
            ValidateRequestParameters: false
        }));
        test.done();
    },
    'use default requestParameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                requestParameters: { "method.request.path.proxy": true }
            }
        });
        // WHEN
        new apigw.Method(stack, 'defaultRequestParameters', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultRequestParameters'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultRequestParameters',
            RequestParameters: {
                "method.request.path.proxy": true
            }
        }));
        test.done();
    },
    'authorizer is bound correctly'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: DUMMY_AUTHORIZER
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'ANY',
            AuthorizationType: 'CUSTOM',
            AuthorizerId: DUMMY_AUTHORIZER.authorizerId,
        }));
        test.done();
    },
    'authorizer via default method options'(test) {
        const stack = new cdk.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new apigw.TokenAuthorizer(stack, 'myauthorizer1', {
            authorizerName: 'myauthorizer1',
            handler: func
        });
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: auth
            }
        });
        restApi.root.addMethod('ANY');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Name: 'myauthorizer1',
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId)
        }));
        test.done();
    },
    'fails when authorization type does not match the authorizer'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.IAM,
                authorizer: DUMMY_AUTHORIZER
            });
        }, /Authorization type is set to AWS_IAM which is different from what is required by the authorizer/);
        test.done();
    },
    'fails when authorization type does not match the authorizer in default method options'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: DUMMY_AUTHORIZER
            }
        });
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.NONE,
            });
        }, /Authorization type is set to NONE which is different from what is required by the authorizer/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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