"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', { deployment });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                testapiD6451F70: {
                    Type: "AWS::ApiGateway::RestApi",
                    Properties: {
                        Name: "test-api"
                    }
                },
                testapiGETD8DE4ED1: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "GET",
                        ResourceId: {
                            "Fn::GetAtt": [
                                "testapiD6451F70",
                                "RootResourceId"
                            ]
                        },
                        RestApiId: {
                            Ref: "testapiD6451F70"
                        },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                },
                mydeployment71ED3B4B: {
                    Type: "AWS::ApiGateway::Deployment",
                    Properties: {
                        RestApiId: {
                            Ref: "testapiD6451F70"
                        }
                    },
                    DependsOn: [
                        "testapiGETD8DE4ED1"
                    ]
                },
                mystage7483BE9A: {
                    Type: "AWS::ApiGateway::Stage",
                    Properties: {
                        RestApiId: {
                            Ref: "testapiD6451F70"
                        },
                        DeploymentId: {
                            Ref: "mydeployment71ED3B4B"
                        },
                        StageName: "prod"
                    }
                }
            }
        });
        test.done();
    },
    'common method settings can be set at the stage level'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            loggingLevel: apigateway.MethodLoggingLevel.INFO,
            throttlingRateLimit: 12
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            MethodSettings: [
                {
                    HttpMethod: "*",
                    LoggingLevel: "INFO",
                    ResourcePath: "/*",
                    ThrottlingRateLimit: 12,
                }
            ]
        }));
        test.done();
    },
    'custom method settings can be set by their path'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            loggingLevel: apigateway.MethodLoggingLevel.INFO,
            throttlingRateLimit: 12,
            methodOptions: {
                '/goo/bar/GET': {
                    loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                }
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            MethodSettings: [
                {
                    HttpMethod: "*",
                    LoggingLevel: "INFO",
                    ResourcePath: "/*",
                    ThrottlingRateLimit: 12
                },
                {
                    HttpMethod: "GET",
                    LoggingLevel: "ERROR",
                    ResourcePath: "/~1goo~1bar"
                }
            ]
        }));
        test.done();
    },
    'default "cacheClusterSize" is 0.5 (if cache cluster is enabled)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterEnabled: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: "0.5"
        }));
        test.done();
    },
    'setting "cacheClusterSize" implies "cacheClusterEnabled"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterSize: '0.5'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: "0.5"
        }));
        test.done();
    },
    'fails when "cacheClusterEnabled" is "false" and "cacheClusterSize" is set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cacheClusterSize: '0.5',
            cacheClusterEnabled: false
        }), /Cannot set "cacheClusterSize" to 0.5 and "cacheClusterEnabled" to "false"/);
        test.done();
    },
    'if "cachingEnabled" in method settings, implicitly enable cache cluster'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // WHEN
        new apigateway.Stage(stack, 'my-stage', {
            deployment,
            cachingEnabled: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Stage', {
            CacheClusterEnabled: true,
            CacheClusterSize: "0.5",
            MethodSettings: [
                {
                    CachingEnabled: true,
                    HttpMethod: "*",
                    ResourcePath: "/*"
                }
            ],
            StageName: "prod"
        }));
        test.done();
    },
    'if caching cluster is explicitly disabled, do not auto-enable cache cluster when "cachingEnabled" is set in method options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        const deployment = new apigateway.Deployment(stack, 'my-deployment', { api });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => new apigateway.Stage(stack, 'my-stage', {
            cacheClusterEnabled: false,
            deployment,
            cachingEnabled: true
        }), /Cannot enable caching for method \/\*\/\* since cache cluster is disabled on stage/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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