/*
 * Copyright (C) 2020 <Florian Alu - Prolibre - https://prolibre.com
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

(function ( $ ) {

	var options = {
		series: {
			fillBetween: null	// or number
		}
	};

	function init( plot ) {

		function findBottomSeries( s, allseries ) {

			var i;

			for ( i = 0; i < allseries.length; ++i ) {
				if ( allseries[ i ].id === s.fillBetween ) {
					return allseries[ i ];
				}
			}

			if ( typeof s.fillBetween === "number" ) {
				if ( s.fillBetween < 0 || s.fillBetween >= allseries.length ) {
					return null;
				}
				return allseries[ s.fillBetween ];
			}

			return null;
		}

		function computeFillBottoms( plot, s, datapoints ) {

			if ( s.fillBetween == null ) {
				return;
			}

			var other = findBottomSeries( s, plot.getData() );

			if ( !other ) {
				return;
			}

			var ps = datapoints.pointsize,
				points = datapoints.points,
				otherps = other.datapoints.pointsize,
				otherpoints = other.datapoints.points,
				newpoints = [],
				px, py, intery, qx, qy, bottom,
				withlines = s.lines.show,
				withbottom = ps > 2 && datapoints.format[2].y,
				withsteps = withlines && s.lines.steps,
				fromgap = true,
				i = 0,
				j = 0,
				l, m;

			while ( true ) {

				if ( i >= points.length ) {
					break;
				}

				l = newpoints.length;

				if ( points[ i ] == null ) {

					// copy gaps

					for ( m = 0; m < ps; ++m ) {
						newpoints.push( points[ i + m ] );
					}

					i += ps;

				} else if ( j >= otherpoints.length ) {

					// for lines, we can't use the rest of the points

					if ( !withlines ) {
						for ( m = 0; m < ps; ++m ) {
							newpoints.push( points[ i + m ] );
						}
					}

					i += ps;

				} else if ( otherpoints[ j ] == null ) {

					// oops, got a gap

					for ( m = 0; m < ps; ++m ) {
						newpoints.push( null );
					}

					fromgap = true;
					j += otherps;

				} else {

					// cases where we actually got two points

					px = points[ i ];
					py = points[ i + 1 ];
					qx = otherpoints[ j ];
					qy = otherpoints[ j + 1 ];
					bottom = 0;

					if ( px === qx ) {

						for ( m = 0; m < ps; ++m ) {
							newpoints.push( points[ i + m ] );
						}

						//newpoints[ l + 1 ] += qy;
						bottom = qy;

						i += ps;
						j += otherps;

					} else if ( px > qx ) {

						// we got past point below, might need to
						// insert interpolated extra point

						if ( withlines && i > 0 && points[ i - ps ] != null ) {
							intery = py + ( points[ i - ps + 1 ] - py ) * ( qx - px ) / ( points[ i - ps ] - px );
							newpoints.push( qx );
							newpoints.push( intery );
							for ( m = 2; m < ps; ++m ) {
								newpoints.push( points[ i + m ] );
							}
							bottom = qy;
						}

						j += otherps;

					} else { // px < qx

						// if we come from a gap, we just skip this point

						if ( fromgap && withlines ) {
							i += ps;
							continue;
						}

						for ( m = 0; m < ps; ++m ) {
							newpoints.push( points[ i + m ] );
						}

						// we might be able to interpolate a point below,
						// this can give us a better y

						if ( withlines && j > 0 && otherpoints[ j - otherps ] != null ) {
							bottom = qy + ( otherpoints[ j - otherps + 1 ] - qy ) * ( px - qx ) / ( otherpoints[ j - otherps ] - qx );
						}

						//newpoints[l + 1] += bottom;

						i += ps;
					}

					fromgap = false;

					if ( l !== newpoints.length && withbottom ) {
						newpoints[ l + 2 ] = bottom;
					}
				}

				// maintain the line steps invariant

				if ( withsteps && l !== newpoints.length && l > 0 &&
					newpoints[ l ] !== null &&
					newpoints[ l ] !== newpoints[ l - ps ] &&
					newpoints[ l + 1 ] !== newpoints[ l - ps + 1 ] ) {
					for (m = 0; m < ps; ++m) {
						newpoints[ l + ps + m ] = newpoints[ l + m ];
					}
					newpoints[ l + 1 ] = newpoints[ l - ps + 1 ];
				}
			}

			datapoints.points = newpoints;
		}

		plot.hooks.processDatapoints.push( computeFillBottoms );
	}

	$.plot.plugins.push({
		init: init,
		options: options,
		name: "fillbetween",
		version: "1.0"
	});

})(jQuery);
