# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/24_pharmacophore.ipynb (unless otherwise specified).

__all__ = ['surface', 'pharm_surface', 'Pharmacophore', 'MolPharmacophore']

# Cell

from .imports import *
from .core import *
from .chem import *
from numba import jit

from rdkit import Chem
from rdkit.Chem import ChemicalFeatures
from rdkit import RDConfig

# Cell

@jit(nopython=True)
def surface(n_points, coords, vdw_array):

    out_pts = 0
    out = []
    for k, n in enumerate(n_points):
        u = []
        eps = 1e-10
        nequat = int(np.sqrt(np.pi*n))
        nvert = int(nequat/2)
        nu = 0
        for i in range(nvert+1):
            fi = np.pi*i/nvert
            z = np.cos(fi)
            xy = np.sin(fi)
            nhor = int(nequat*xy+eps)
            if nhor < 1:
                nhor = 1
            for j in range(nhor):
                fj = 2*np.pi*j/nhor
                x = np.cos(fj)*xy
                y = np.sin(fj)*xy

                nu += 1
                u.append([x, y, z])

        v = coords[k] + vdw_array[k]*np.array(u)
        out_pts += v.shape[0]
        out.append(v)

    out_array = np.zeros((out_pts, 3))
    idx = 0
    for item in out:
        out_array[idx : idx+item.shape[0]] = item
        idx += item.shape[0]

    return out_array

# Cell

def pharm_surface(coords, vdw_array, density, classes=None):

    n_points = (density * 4.0 * np.pi* np.power(vdw_array, 2))
    surface_points = surface(n_points, coords, vdw_array)

    distances = (((np.expand_dims(surface_points,1) - np.expand_dims(coords, 0))**2).sum(-1)**0.5)
    mask = (distances < vdw_array-1e-4).sum(-1)==0

    surface_points = surface_points[mask]

    if classes is not None:
        distances = (((np.expand_dims(surface_points,1) - np.expand_dims(coords, 0))**2).sum(-1)**0.5)
        surface_classes = classes[distances.argmin(-1)]
    else:
        surface_classes = None

    return surface_points, surface_classes

# Cell

class Pharmacophore():
    def __init__(self, feature_file=None, pharm_families=None):

        if feature_file is None:
            feature_file = os.path.join(RDConfig.RDDataDir,'BaseFeatures.fdef')

        self.feature_file = feature_file
        self.factory = ChemicalFeatures.BuildFeatureFactory(feature_file)
        self.allowed_pharms = list(self.factory.GetFeatureFamilies())

        if pharm_families is None:
            pharm_families = list(self.factory.GetFeatureFamilies())
        else:
            for pharm in pharm_families:
                assert pharm in self.allowed_pharms

        self.pharm_families = pharm_families

        self.pharm_to_id = {self.pharm_families[i]:i for i in range(len(self.pharm_families))}

    def get_feats_for_mol(self, mol, conf_id=-1):

        feats = self.factory.GetFeaturesForMol(mol, confId=conf_id)

        pharm_class = []
        pharm_coords = []
        atom_ids = []

        for feat in feats:
            pc = feat.GetFamily()
            if pc in self.pharm_families:
                pharm_class.append(self.pharm_to_id[pc])
                pharm_coords.append(np.array(feat.GetPos()))
                atom_ids.append(feat.GetAtomIds())

        pharm_class = np.array(pharm_class)
        pharm_coords = np.array(pharm_coords)

        return pharm_class, pharm_coords, atom_ids

class MolPharmacophore():
    def __init__(self, mol, coords, pharm_class, pharm_coords, atom_ids):
        self.mol = mol
        self.coords = coords
        self.pharm_class = pharm_class
        self.pharm_coords = pharm_coords
        self.atom_ids = atom_ids
        self.surface = None

    @classmethod
    def create_from_pharmacophore(cls, mol, pharmacophore, conf_id=-1):

        try:
            coords = mol.GetConformer(id=conf_id).GetPositions()
        except:
            mol = add_hs(mol)
            _ = conformer_generation(mol, 1, nthreads=1)
            mol = remove_hs(mol)
            coords = mol.GetConformer(id=conf_id).GetPositions()

        pharm_class, pharm_coords, atom_ids = pharmacophore.get_feats_for_mol(mol, conf_id=conf_id)

        return cls(mol, coords, pharm_class, pharm_coords, atom_ids)

    def build_pharm_surface(self, density, scale):
        vdw_array = np.array([scale for i in self.pharm_coords])

        points, classes = pharm_surface(self.pharm_coords, vdw_array, density, self.pharm_class)
        return points, classes

    def build_atom_surface(self, density):

        pt = Chem.GetPeriodicTable()
        vdw_array = []
        atom_types = []

        for i, atom in enumerate(mol.GetAtoms()):
            vdw_array.append(pt.GetRvdw(atom.GetAtomicNum()))
            atom_types.append(atom.GetSymbol())

        vdw_array = np.array(vdw_array)
        atom_types = np.array(atom_types)

        points, classes = pharm_surface(self.coords, vdw_array, density, atom_types)
        return points, classes