import base64
import os
import unittest

from tonclient.errors import TonException
from tonclient.test.helpers import async_core_client, sync_core_client, SAMPLES_DIR
from tonclient.types import (
    CallSet,
    DeploySet,
    FunctionHeader,
    ParamsOfEncodeExternalInMessage,
    ParamsOfEncodeMessage,
    ParamsOfParse,
    ParamsOfGetBocHash,
    ParamsOfGetBlockchainConfig,
    ParamsOfGetCodeFromTvc,
    ParamsOfBocCacheSet,
    BocCacheType,
    ParamsOfBocCacheGet,
    ParamsOfBocCacheUnpin,
    ParamsOfEncodeBoc,
    BuilderOp,
    ParamsOfGetCodeSalt,
    ParamsOfSetCodeSalt,
    ResultOfDecodeTvc,
    ParamsOfDecodeTvc,
    ParamsOfParseShardstate,
    ParamsOfEncodeTvc,
    ParamsOfGetCompilerVersion,
    ParamsOfGetBocDepth,
    Abi,
    KeyPair,
    Signer,
)


class TestTonBocAsyncCore(unittest.TestCase):
    def test_parse_message(self):
        message = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        params = ParamsOfParse(boc=message)
        result = async_core_client.boc.parse_message(params=params)
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            result.parsed['id'],
        )
        self.assertEqual(
            '-1:0000000000000000000000000000000000000000000000000000000000000000',
            result.parsed['src'],
        )
        self.assertEqual(
            '-1:3333333333333333333333333333333333333333333333333333333333333333',
            result.parsed['dst'],
        )

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            async_core_client.boc.parse_message(params=params)

    def test_parse_transaction(self):
        transaction = 'te6ccgECBwEAAZQAA7V75gA6WK5sEDTiHFGnH9ILOy2irjKLWTkWQMyMogsg40AAACDribjoE3gOAbYNpCaX4uLeXPQHt2Kw/Jp2OKkR2s+BASyeQM6wAAAg64IXyBX2DobAABRrMENIBQQBAhUEQojmJaAYazBCEQMCAFvAAAAAAAAAAAAAAAABLUUtpEnlC4z33SeGHxRhIq/htUa7i3D8ghbwxhQTn44EAJxC3UicQAAAAAAAAAAAdwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgnJAnYEvIQY6SnQKc3lXk6x1Z/lyplGFRbwAuNtVBi9EeceU3Ojl0F3EkRdylowY5x2qlgHNv4lNZUjhq0WqrLMNAQGgBgC3aADLL4ChL2HyLHwOLub5Mep87W3xdnMW8BpxKyVoGe3RPQAvmADpYrmwQNOIcUacf0gs7LaKuMotZORZAzIyiCyDjQ5iWgAGFFhgAAAEHXC9CwS+wdDGKTmMFkA='
        params = ParamsOfParse(boc=transaction)
        result = async_core_client.boc.parse_transaction(params=params)
        self.assertEqual(
            'd6315dbb2a741a2765da250bea4a186adf942469369c703c57c2050e2d6e9fe3',
            result.parsed['id'],
        )
        self.assertEqual('0x20eb89b8e81', result.parsed['lt'])
        self.assertEqual(1600186476, result.parsed['now'])

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            async_core_client.boc.parse_transaction(params=params)

    def test_parse_account(self):
        account = 'te6ccgECHQEAA/wAAnfAArtKDoOR5+qId/SCUGSSS9Qc4RD86X6TnTMjmZ4e+7EyOobmQvsHNngAAAg6t/34DgJWKJuuOehjU0ADAQFBlcBqp0PR+QAN1kt1SY8QavS350RCNNfeZ+ommI9hgd/gAgBToB6t2E3E7a7aW2YkvXv2hTmSWVRTvSYmCVdH4HjgZ4Z94AAAAAvsHNwwAib/APSkICLAAZL0oOGK7VNYMPShBgQBCvSkIPShBQAAAgEgCgcBAv8IAf5/Ie1E0CDXScIBn9P/0wD0Bfhqf/hh+Gb4Yo4b9AVt+GpwAYBA9A7yvdcL//hicPhjcPhmf/hh4tMAAY4SgQIA1xgg+QFY+EIg+GX5EPKo3iP4RSBukjBw3vhCuvLgZSHTP9MfNCD4I7zyuSL5ACD4SoEBAPQOIJEx3vLQZvgACQA2IPhKI8jLP1mBAQD0Q/hqXwTTHwHwAfhHbvJ8AgEgEQsCAVgPDAEJuOiY/FANAdb4QW6OEu1E0NP/0wD0Bfhqf/hh+Gb4Yt7RcG1vAvhKgQEA9IaVAdcLP3+TcHBw4pEgjjJfM8gizwv/Ic8LPzExAW8iIaQDWYAg9ENvAjQi+EqBAQD0fJUB1ws/f5NwcHDiAjUzMehfAyHA/w4AmI4uI9DTAfpAMDHIz4cgzo0EAAAAAAAAAAAAAAAAD3RMfijPFiFvIgLLH/QAyXH7AN4wwP+OEvhCyMv/+EbPCwD4SgH0AMntVN5/+GcBCbkWq+fwEAC2+EFujjbtRNAg10nCAZ/T/9MA9AX4an/4Yfhm+GKOG/QFbfhqcAGAQPQO8r3XC//4YnD4Y3D4Zn/4YeLe+Ebyc3H4ZtH4APhCyMv/+EbPCwD4SgH0AMntVH/4ZwIBIBUSAQm7Fe+TWBMBtvhBbo4S7UTQ0//TAPQF+Gp/+GH4Zvhi3vpA1w1/ldTR0NN/39cMAJXU0dDSAN/RVHEgyM+FgMoAc89AzgH6AoBrz0DJc/sA+EqBAQD0hpUB1ws/f5NwcHDikSAUAISOKCH4I7ubIvhKgQEA9Fsw+GreIvhKgQEA9HyVAdcLP3+TcHBw4gI1MzHoXwb4QsjL//hGzwsA+EoB9ADJ7VR/+GcCASAYFgEJuORhh1AXAL74QW6OEu1E0NP/0wD0Bfhqf/hh+Gb4Yt7U0fhFIG6SMHDe+EK68uBl+AD4QsjL//hGzwsA+EoB9ADJ7VT4DyD7BCDQ7R7tU/ACMPhCyMv/+EbPCwD4SgH0AMntVH/4ZwIC2hsZAQFIGgAs+ELIy//4Rs8LAPhKAfQAye1U+A/yAAEBSBwAWHAi0NYCMdIAMNwhxwDcIdcNH/K8UxHdwQQighD////9vLHyfAHwAfhHbvJ8'
        params = ParamsOfParse(boc=account)
        result = async_core_client.boc.parse_account(params=params)
        self.assertEqual(
            '0:2bb4a0e8391e7ea8877f4825064924bd41ce110fce97e939d3323999e1efbb13',
            result.parsed['id'],
        )
        self.assertEqual('0x20eadff7e03', result.parsed['last_trans_lt'])
        self.assertEqual('0x958a26eb8e7a18d', result.parsed['balance'])

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            async_core_client.boc.parse_account(params=params)

    def test_parse_block(self):
        block = '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'
        params = ParamsOfParse(boc=block)
        result = async_core_client.boc.parse_block(params=params)
        self.assertEqual(
            '048f59d5d652459939ea5c5e7b291155205696b71e0c556f641df69e70e1e725',
            result.parsed['id'],
        )
        self.assertEqual(4296363, result.parsed['seq_no'])
        self.assertEqual(1600234696, result.parsed['gen_utime'])

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            async_core_client.boc.parse_block(params=params)

    def test_parse_shardstate(self):
        boc = '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'
        state_id = 'zerostate:-1'
        workchain_id = -1
        params = ParamsOfParseShardstate(boc=boc, id='zerostate:-1', workchain_id=-1)
        result = async_core_client.boc.parse_shardstate(params=params)
        self.assertEqual(state_id, result.parsed['id'])
        self.assertEqual(workchain_id, result.parsed['workchain_id'])
        self.assertEqual(0, result.parsed['seq_no'])

    def test_get_boc_hash(self):
        boc = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        params = ParamsOfGetBocHash(boc=boc)
        result = async_core_client.boc.get_boc_hash(params=params)
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            result.hash,
        )

        # Get boc hash from cached boc
        result = async_core_client.boc.cache_set(
            params=ParamsOfBocCacheSet(boc=boc, cache_type=BocCacheType.Unpinned())
        )
        boc_ref = result.boc_ref

        result = async_core_client.boc.get_boc_hash(
            params=ParamsOfGetBocHash(boc=boc_ref)
        )
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            result.hash,
        )

    def test_get_boc_depth(self):
        boc = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        params = ParamsOfGetBocDepth(boc=boc)
        result = async_core_client.boc.get_boc_depth(params)
        self.assertEqual(0, result.depth)

    def test_get_blockchain_config(self):
        block = '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'
        params = ParamsOfGetBlockchainConfig(block_boc=block)
        result = async_core_client.boc.get_blockchain_config(params=params)
        self.assertEqual(
            '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',
            result.config_boc,
        )

    def test_get_code_from_tvc(self):
        tvc = 'te6ccgECHAEABDkAAgE0BgEBAcACAgPPIAUDAQHeBAAD0CAAQdgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABAIm/wD0pCAiwAGS9KDhiu1TWDD0oQkHAQr0pCD0oQgAAAIBIAwKAej/fyHTAAGOJoECANcYIPkBAXDtRND0BYBA9A7yitcL/wHtRyJvde1XAwH5EPKo3u1E0CDXScIBjhb0BNM/0wDtRwFvcQFvdgFvcwFvcu1Xjhj0Be1HAW9ycG9zcG92yIAgz0DJ0G9x7Vfi0z8B7UdvEyG5IAsAYJ8wIPgjgQPoqIIIG3dAoLneme1HIW9TIO1XMJSANPLw4jDTHwH4I7zyudMfAfFAAQIBIBgNAgEgEQ4BCbqLVfP4DwH67UdvYW6OO+1E0CDXScIBjhb0BNM/0wDtRwFvcQFvdgFvcwFvcu1Xjhj0Be1HAW9ycG9zcG92yIAgz0DJ0G9x7Vfi3u1HbxaS8jOX7Udxb1btV+IA+ADR+CO1H+1HIG8RMAHIyx/J0G9R7VftR28SyPQA7UdvE88LP+1HbxYQABzPCwDtR28RzxbJ7VRwagIBahUSAQm0ABrWwBMB/O1Hb2FujjvtRNAg10nCAY4W9ATTP9MA7UcBb3EBb3YBb3MBb3LtV44Y9AXtRwFvcnBvc3BvdsiAIM9AydBvce1X4t7tR29lIG6SMHDecO1HbxKAQPQO8orXC/+68uBk+AD6QNEgyMn7BIED6HCBAIDIcc8LASLPCgBxz0D4KBQAjs8WJM8WI/oCcc9AcPoCcPoCgEDPQPgjzwsfcs9AIMki+wBfBTDtR28SyPQA7UdvE88LP+1HbxbPCwDtR28RzxbJ7VRwatswAQm0ZfaLwBYB+O1Hb2FujjvtRNAg10nCAY4W9ATTP9MA7UcBb3EBb3YBb3MBb3LtV44Y9AXtRwFvcnBvc3BvdsiAIM9AydBvce1X4t7R7UdvEdcLH8iCEFDL7ReCEIAAAACxzwsfIc8LH8hzzwsB+CjPFnLPQPglzws/gCHPQCDPNSLPMbwXAHiWcc9AIc8XlXHPQSHN4iDJcfsAWyHA/44e7UdvEsj0AO1HbxPPCz/tR28WzwsA7UdvEc8Wye1U3nFq2zACASAbGQEJu3MS5FgaAPjtR29hbo477UTQINdJwgGOFvQE0z/TAO1HAW9xAW92AW9zAW9y7VeOGPQF7UcBb3Jwb3Nwb3bIgCDPQMnQb3HtV+Le+ADR+CO1H+1HIG8RMAHIyx/J0G9R7VftR28SyPQA7UdvE88LP+1HbxbPCwDtR28RzxbJ7VRwatswAMrdcCHXSSDBII4rIMAAjhwj0HPXIdcLACDAAZbbMF8H2zCW2zBfB9sw4wTZltswXwbbMOME2eAi0x80IHS7II4VMCCCEP////+6IJkwIIIQ/////rrf35bbMF8H2zDgIyHxQAFfBw=='
        code = 'te6ccgECFgEAA/8AAib/APSkICLAAZL0oOGK7VNYMPShAwEBCvSkIPShAgAAAgEgBgQB6P9/IdMAAY4mgQIA1xgg+QEBcO1E0PQFgED0DvKK1wv/Ae1HIm917VcDAfkQ8qje7UTQINdJwgGOFvQE0z/TAO1HAW9xAW92AW9zAW9y7VeOGPQF7UcBb3Jwb3Nwb3bIgCDPQMnQb3HtV+LTPwHtR28TIbkgBQBgnzAg+COBA+iogggbd0Cgud6Z7Uchb1Mg7VcwlIA08vDiMNMfAfgjvPK50x8B8UABAgEgEgcCASALCAEJuotV8/gJAfrtR29hbo477UTQINdJwgGOFvQE0z/TAO1HAW9xAW92AW9zAW9y7VeOGPQF7UcBb3Jwb3Nwb3bIgCDPQMnQb3HtV+Le7UdvFpLyM5ftR3FvVu1X4gD4ANH4I7Uf7UcgbxEwAcjLH8nQb1HtV+1HbxLI9ADtR28Tzws/7UdvFgoAHM8LAO1HbxHPFsntVHBqAgFqDwwBCbQAGtbADQH87UdvYW6OO+1E0CDXScIBjhb0BNM/0wDtRwFvcQFvdgFvcwFvcu1Xjhj0Be1HAW9ycG9zcG92yIAgz0DJ0G9x7Vfi3u1Hb2UgbpIwcN5w7UdvEoBA9A7yitcL/7ry4GT4APpA0SDIyfsEgQPocIEAgMhxzwsBIs8KAHHPQPgoDgCOzxYkzxYj+gJxz0Bw+gJw+gKAQM9A+CPPCx9yz0AgySL7AF8FMO1HbxLI9ADtR28Tzws/7UdvFs8LAO1HbxHPFsntVHBq2zABCbRl9ovAEAH47UdvYW6OO+1E0CDXScIBjhb0BNM/0wDtRwFvcQFvdgFvcwFvcu1Xjhj0Be1HAW9ycG9zcG92yIAgz0DJ0G9x7Vfi3tHtR28R1wsfyIIQUMvtF4IQgAAAALHPCx8hzwsfyHPPCwH4KM8Wcs9A+CXPCz+AIc9AIM81Is8xvBEAeJZxz0AhzxeVcc9BIc3iIMlx+wBbIcD/jh7tR28SyPQA7UdvE88LP+1HbxbPCwDtR28RzxbJ7VTecWrbMAIBIBUTAQm7cxLkWBQA+O1Hb2FujjvtRNAg10nCAY4W9ATTP9MA7UcBb3EBb3YBb3MBb3LtV44Y9AXtRwFvcnBvc3BvdsiAIM9AydBvce1X4t74ANH4I7Uf7UcgbxEwAcjLH8nQb1HtV+1HbxLI9ADtR28Tzws/7UdvFs8LAO1HbxHPFsntVHBq2zAAyt1wIddJIMEgjisgwACOHCPQc9ch1wsAIMABltswXwfbMJbbMF8H2zDjBNmW2zBfBtsw4wTZ4CLTHzQgdLsgjhUwIIIQ/////7ogmTAgghD////+ut/fltswXwfbMOAjIfFAAV8H'
        getcode_params = ParamsOfGetCodeFromTvc(tvc=tvc)
        result = async_core_client.boc.get_code_from_tvc(params=getcode_params)
        self.assertEqual(code, result.code)

    def test_pinned_cache(self):
        with open(os.path.join(SAMPLES_DIR, 'Hello.tvc'), 'rb') as fp:
            hello_tvc = base64.b64encode(fp.read()).decode()
        with open(os.path.join(SAMPLES_DIR, 'Events.tvc'), 'rb') as fp:
            events_tvc = base64.b64encode(fp.read()).decode()

        pin1 = 'pin1'
        pin2 = 'pin2'

        result = async_core_client.boc.cache_set(
            params=ParamsOfBocCacheSet(
                boc=hello_tvc, cache_type=BocCacheType.Pinned(pin=pin1)
            )
        )
        ref1 = result.boc_ref
        self.assertEqual('*', ref1[0])
        self.assertEqual(65, len(ref1))

        hello_boc = async_core_client.boc.cache_get(
            params=ParamsOfBocCacheGet(boc_ref=ref1)
        )
        self.assertEqual(hello_tvc, hello_boc.boc)

        result = async_core_client.boc.cache_set(
            params=ParamsOfBocCacheSet(
                boc=events_tvc, cache_type=BocCacheType.Pinned(pin=pin1)
            )
        )
        ref2 = result.boc_ref
        self.assertNotEqual(ref2, ref1)

        result = async_core_client.boc.cache_set(
            params=ParamsOfBocCacheSet(
                boc=hello_tvc, cache_type=BocCacheType.Pinned(pin=pin2)
            )
        )
        ref3 = result.boc_ref
        self.assertEqual(ref3, ref1)

        async_core_client.boc.cache_unpin(params=ParamsOfBocCacheUnpin(pin=pin1))

        boc = async_core_client.boc.cache_get(params=ParamsOfBocCacheGet(boc_ref=ref1))
        self.assertEqual(hello_tvc, boc.boc)

        boc = async_core_client.boc.cache_get(params=ParamsOfBocCacheGet(boc_ref=ref2))
        self.assertIsNone(boc.boc)

        async_core_client.boc.cache_unpin(
            params=ParamsOfBocCacheUnpin(pin=pin2, boc_ref=ref1)
        )

    def test_unpinned_cache(self):
        with open(os.path.join(SAMPLES_DIR, 'Hello.tvc'), 'rb') as fp:
            hello_tvc = base64.b64encode(fp.read()).decode()

        result = async_core_client.boc.cache_set(
            params=ParamsOfBocCacheSet(
                boc=hello_tvc, cache_type=BocCacheType.Unpinned()
            )
        )
        ref = result.boc_ref

        result = async_core_client.boc.cache_get(
            params=ParamsOfBocCacheGet(boc_ref=ref)
        )
        self.assertEqual(hello_tvc, result.boc)

    def test_encode_boc(self):
        params = ParamsOfEncodeBoc(
            builder=[
                BuilderOp.Integer(size=1, value=1),
                BuilderOp.Integer(size=1, value=0),
                BuilderOp.Integer(size=8, value=255),
                BuilderOp.Integer(size=8, value=127),
                BuilderOp.Integer(size=8, value=-127),
                BuilderOp.Integer(size=128, value=123456789123456789),
                BuilderOp.BitString(value='8A_'),
                BuilderOp.BitString(value='x{8A0_}'),
                BuilderOp.BitString(value='123'),
                BuilderOp.BitString(value='x2d9_'),
                BuilderOp.BitString(value='80_'),
                BuilderOp.Cell(
                    builder=[
                        BuilderOp.BitString(value='n101100111000'),
                        BuilderOp.BitString(value='N100111000'),
                        BuilderOp.Integer(size=3, value=-1),
                        BuilderOp.Integer(size=3, value=2),
                        BuilderOp.Integer(size=16, value=312),
                        BuilderOp.Integer(size=16, value='0x123'),
                        BuilderOp.Integer(size=16, value='0x123'),
                        BuilderOp.Integer(size=16, value='-0x123'),
                    ]
                ),
            ]
        )
        result = async_core_client.boc.encode_boc(params=params)
        self.assertEqual(
            'te6ccgEBAgEAKQABL7/f4EAAAAAAAAAAAG2m0us0F8ViiEjLZAEAF7OJx0AnACRgJH/bsA==',
            result.boc,
        )

        # Test with cell BOC
        inner_cell = 'te6ccgEBAQEADgAAF7OJx0AnACRgJH/bsA=='
        params = ParamsOfEncodeBoc(
            builder=[
                BuilderOp.Integer(size=1, value=1),
                BuilderOp.Integer(size=1, value=0),
                BuilderOp.Integer(size=8, value=255),
                BuilderOp.Integer(size=8, value=127),
                BuilderOp.Integer(size=8, value=-127),
                BuilderOp.Integer(size=128, value=123456789123456789),
                BuilderOp.BitString(value='8A_'),
                BuilderOp.BitString(value='x{8A0_}'),
                BuilderOp.BitString(value='123'),
                BuilderOp.BitString(value='x2d9_'),
                BuilderOp.BitString(value='80_'),
                BuilderOp.CellBoc(boc=inner_cell),
            ]
        )
        result = async_core_client.boc.encode_boc(params=params)
        self.assertEqual(
            'te6ccgEBAgEAKQABL7/f4EAAAAAAAAAAAG2m0us0F8ViiEjLZAEAF7OJx0AnACRgJH/bsA==',
            result.boc,
        )

    def test_code_salt(self):
        # `mycode_sel_dict_nosalt.boc`
        code_no_salt = 'te6ccgECGAEAAmMAAgaK2zUXAQQkiu1TIOMDIMD/4wIgwP7jAvILFAMCDgKE7UTQ10nDAfhmIds80wABn4ECANcYIPkBWPhC+RDyqN7TPwH4QyG58rQg+COBA+iogggbd0CgufK0+GPTHwHbPPI8BgQDSu1E0NdJwwH4ZiLQ1wsDqTgA3CHHAOMCIdcNH/K8IeMDAds88jwTEwQDPCCCEDKVn7a64wIgghBgeXU+uuMCIIIQaLVfP7rjAg8HBQIiMPhCbuMA+Ebyc9H4ANs88gAGEAE+7UTQ10nCAYqOFHDtRND0BYBA9A7yvdcL//hicPhj4hICdjD4RvLgTNTR2zwhjicj0NMB+kAwMcjPhyDOjQQAAAAAAAAAAAAAAAAOB5dT6M8WzMlw+wCRMOLjAPIACBACMvhBiMjPjits1szOyTCBAIbIy/8B0AHJ2zwXCQIWIYs4rbNYxwWKiuILCgEIAds8yQwBJgHU1DAS0Ns8yM+OK2zWEszPEckMAWbViy9KQNcm9ATTCTEg10qR1I6A4osvShjXJjAByM+L0pD0AIAgzwsJz4vShswSzMjPEc4NAQSIAQ4AAAOEMPhG8uBM+EJu4wDT/9HbPCGOKCPQ0wH6QDAxyM+HIM6NBAAAAAAAAAAAAAAAAAspWftozxbL/8lw+wCRMOLbPPIAEhEQABz4Q/hCyMv/yz/Pg8ntVAAgIMECkXGYUwCltf/wHKjiMQAe7UTQ0//TP9MAMdH4Y/hiAAr4RvLgTAIK9KQg9KEWFQAUc29sIDAuNTEuMAAqoAAAABwgwQKRcZhTAKW1//AcqOIxAAwg+GHtHtk='

        # Get code salt
        params = ParamsOfGetCodeSalt(code=code_no_salt)
        salt = async_core_client.boc.get_code_salt(params).salt
        self.assertIsNone(salt)

        # Set code salt
        code_salt = 'te6ccgEBAQEAJAAAQ4AGPqCXQ2drhdqhLLt3rJ80LxA65YMTwgWLLUmt9EbElFA='
        params = ParamsOfSetCodeSalt(code=code_no_salt, salt=code_salt)
        code_with_salt = async_core_client.boc.set_code_salt(params).code
        self.assertNotEqual(code_no_salt, code_with_salt)

        # Get code salt
        params = ParamsOfGetCodeSalt(code=code_with_salt)
        salt = async_core_client.boc.get_code_salt(params).salt
        self.assertEqual(code_salt, salt)

    def test_encode_decode_tvc(self):
        # `state_init_lib.boc`
        tvc_check = 'te6ccgECDAEAARMAAwF/CAcBAgFiBQIBQr9BJCkgXqZtbyAE7fpXD29Ws+heWbqhvvvHO32l1VvcYQMBBBI0BAAEVngBQr9aLu9QVndfW5Vy/zrWPdKnHR+ygcoXel4cdHMOzLLlEwYAD6usq62rrKuoAEgR71YDAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABFP8A9KQT9LzyyAsJAgEgCwoA36X//3aiaGmP6f/o5CxSZ4WPkOeF/+T2qmRnxET/s2X/wQgC+vCAfQFANeegZLh9gEB354V/wQgD39JAfQFANeegZLhkZ82JA6Mrm6RBCAOt5or9AUA156BF6kMrY2N5YQO7e5NjIQxni2S4fYB9gEAAAtI='
        decoded_check = ResultOfDecodeTvc(
            code='te6ccgEBBAEAhwABFP8A9KQT9LzyyAsBAgEgAwIA36X//3aiaGmP6f/o5CxSZ4WPkOeF/+T2qmRnxET/s2X/wQgC+vCAfQFANeegZLh9gEB354V/wQgD39JAfQFANeegZLhkZ82JA6Mrm6RBCAOt5or9AUA156BF6kMrY2N5YQO7e5NjIQxni2S4fYB9gEAAAtI=',
            code_hash='45910e27fe37d8dcf1fac777ebb3bda38ae1ea8389f81bfb1bc0079f3f67ef5b',
            code_depth=2,
            data='te6ccgEBAQEAJgAASBHvVgMAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==',
            data_hash='97bfef744b0d45f78b901e2997fb55f6dbc1d396a8d2f8f4c3a5468c010db67a',
            data_depth=0,
            library='te6ccgEBBgEAYAACAWIEAQFCv0EkKSBepm1vIATt+lcPb1az6F5ZuqG++8c7faXVW9xhAgEEEjQDAARWeAFCv1ou71BWd19blXL/OtY90qcdH7KByhd6Xhx0cw7MsuUTBQAPq6yrrausq6g=',
            tick=True,
            tock=True,
        )

        # Decode TVC
        params = ParamsOfDecodeTvc(tvc=tvc_check)
        decoded = async_core_client.boc.decode_tvc(params)
        self.assertEqual(decoded_check.code, decoded.code)
        self.assertEqual(decoded_check.code_hash, decoded.code_hash)
        self.assertEqual(decoded_check.code_depth, decoded.code_depth)
        self.assertEqual(decoded_check.data, decoded.data)
        self.assertEqual(decoded_check.data_hash, decoded.data_hash)
        self.assertEqual(decoded_check.data_depth, decoded.data_depth)
        self.assertEqual(decoded_check.library, decoded.library)
        self.assertEqual(decoded_check.split_depth, decoded.split_depth)
        self.assertEqual(decoded_check.tick, decoded.tick)
        self.assertEqual(decoded_check.tock, decoded.tock)
        self.assertIsNone(decoded.compiler_version)

        # Encode tvc
        params = ParamsOfEncodeTvc(
            code=decoded.code,
            data=decoded.data,
            library=decoded.library,
            tick=decoded.tick,
            tock=decoded.tock,
            split_depth=decoded.split_depth,
        )
        tvc = async_core_client.boc.encode_tvc(params).tvc
        self.assertEqual(tvc_check, tvc)

    def test_get_compiler_version(self):
        # t24_initdata.tvc
        tvc = 'te6ccgECEwEAAbYAAgE0AwEBAcACAEPQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgBCSK7VMg4wMgwP/jAiDA/uMC8gsQBQQSAoTtRNDXScMB+GYh2zzTAAGfgQIA1xgg+QFY+EL5EPKo3tM/AfhDIbnytCD4I4ED6KiCCBt3QKC58rT4Y9MfAds88jwIBgNK7UTQ10nDAfhmItDXCwOpOADcIccA4wIh1w0f8rwh4wMB2zzyPA8PBgIoIIIQBoFGw7rjAiCCEGi1Xz+64wILBwIiMPhCbuMA+Ebyc9H4ANs88gAIDAIW7UTQ10nCAYqOgOIOCQFccO1E0PQFcSGAQPQOk9cLB5Fw4vhqciGAQPQPjoDf+GuAQPQO8r3XC//4YnD4YwoBAogSA3Aw+Eby4Ez4Qm7jANHbPCKOICTQ0wH6QDAxyM+HIM6AYs9AXgHPkhoFGw7LB8zJcPsAkVvi4wDyAA4NDAAq+Ev4SvhD+ELIy//LP8+DywfMye1UAAj4SvhLACztRNDT/9M/0wAx0wfU0fhr+Gr4Y/hiAAr4RvLgTAIK9KQg9KESEQAUc29sIDAuNTEuMAAA'

        # Decode tvc
        params = ParamsOfDecodeTvc(tvc=tvc)
        decoded = async_core_client.boc.decode_tvc(params)

        # Get compiler version
        params = ParamsOfGetCompilerVersion(code=decoded.code)
        version = async_core_client.boc.get_compiler_version(params).version
        self.assertEqual('sol 0.51.0', version)

    def test_encode_external_in_message(self):
        # Encode message with `abi` module.
        events_abi = Abi.from_path(path=os.path.join(SAMPLES_DIR, 'Events.abi.json'))
        keypair = KeyPair(
            public='4c7c408ff1ddebb8d6405ee979c716a14fdd6cc08124107a61d3c25597099499',
            secret='cc8929d635719612a9478b9cd17675a39cfad52d8959e8a177389b8c0b9122a7',
        )
        with open(os.path.join(SAMPLES_DIR, 'Events.tvc'), 'rb') as fp:
            events_tvc = base64.b64encode(fp.read()).decode()
        events_time = 1599458364291
        events_expire = 1599458404

        depoly_set = DeploySet(tvc=events_tvc)
        header = FunctionHeader(
            expire=events_expire, time=events_time, pubkey=keypair.public
        )
        call_set = CallSet(function_name='constructor', header=header)
        params = ParamsOfEncodeMessage(
            abi=events_abi,
            signer=Signer.Keys(keypair),
            deploy_set=depoly_set,
            call_set=call_set,
        )
        abi_encoded = async_core_client.abi.encode_message(params)
        self.assertEqual(
            'te6ccgECGAEAA6wAA0eIAAt9aqvShfTon7Lei1PVOhUEkEEZQkhDKPgNyzeTL6YSEbAHAgEA4bE5Gr3mWwDtlcEOWHr6slWoyQlpIWeYyw/00eKFGFkbAJMMFLWnu0mq4HSrPmktmzeeAboa4kxkFymCsRVt44dTHxAj/Hd67jWQF7peccWoU/dbMCBJBB6YdPCVZcJlJkAAAF0ZyXLg19VzGRotV8/gAQHAAwIDzyAGBAEB3gUAA9AgAEHaY+IEf47vXcayAvdLzji1Cn7rZgQJIIPTDp4SrLhMpMwCJv8A9KQgIsABkvSg4YrtU1gw9KEKCAEK9KQg9KEJAAACASANCwHI/38h7UTQINdJwgGOENP/0z/TANF/+GH4Zvhj+GKOGPQFcAGAQPQO8r3XC//4YnD4Y3D4Zn/4YeLTAAGOHYECANcYIPkBAdMAAZTT/wMBkwL4QuIg+GX5EPKoldMAAfJ64tM/AQwAao4e+EMhuSCfMCD4I4ED6KiCCBt3QKC53pL4Y+CANPI02NMfAfgjvPK50x8B8AH4R26S8jzeAgEgEw4CASAQDwC9uotV8/+EFujjXtRNAg10nCAY4Q0//TP9MA0X/4Yfhm+GP4Yo4Y9AVwAYBA9A7yvdcL//hicPhjcPhmf/hh4t74RvJzcfhm0fgA+ELIy//4Q88LP/hGzwsAye1Uf/hngCASASEQDluIAGtb8ILdHCfaiaGn/6Z/pgGi//DD8M3wx/DFvfSDK6mjofSBv6PwikDdJGDhvfCFdeXAyfABkZP2CEGRnwoRnRoIEB9AAAAAAAAAAAAAAAAAAIGeLZMCAQH2AGHwhZGX//CHnhZ/8I2eFgGT2qj/8M8ADFuZPCot8ILdHCfaiaGn/6Z/pgGi//DD8M3wx/DFva4b/yupo6Gn/7+j8AGRF7gAAAAAAAAAAAAAAAAhni2fA58jjyxi9EOeF/+S4/YAYfCFkZf/8IeeFn/wjZ4WAZPaqP/wzwAgFIFxQBCbi3xYJQFQH8+EFujhPtRNDT/9M/0wDRf/hh+Gb4Y/hi3tcN/5XU0dDT/9/R+ADIi9wAAAAAAAAAAAAAAAAQzxbPgc+Rx5YxeiHPC//JcfsAyIvcAAAAAAAAAAAAAAAAEM8Wz4HPklb4sEohzwv/yXH7ADD4QsjL//hDzws/+EbPCwDJ7VR/FgAE+GcActxwItDWAjHSADDcIccAkvI74CHXDR+S8jzhUxGS8jvhwQQighD////9vLGS8jzgAfAB+EdukvI83g==',
            abi_encoded.message,
        )

        # Parse message.
        params = ParamsOfParse(boc=abi_encoded.message)
        abi_parsed = async_core_client.boc.parse_message(params).parsed

        # Encode TVC.
        params = ParamsOfEncodeTvc(code=abi_parsed['code'], data=abi_parsed['data'])
        init = async_core_client.boc.encode_tvc(params).tvc

        # Encode message with `boc` module.
        params = ParamsOfEncodeExternalInMessage(
            dst=abi_encoded.address, init=init, body=abi_parsed['body']
        )
        boc_encoded = async_core_client.boc.encode_external_in_message(params)
        self.assertEqual(abi_encoded.message, boc_encoded.message)


class TestTonBocSyncCore(unittest.TestCase):
    """Sync core is not recommended to use, so make just a couple of tests"""

    def test_parse_message(self):
        message = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        params = ParamsOfParse(boc=message)
        result = sync_core_client.boc.parse_message(params=params)
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            result.parsed['id'],
        )
        self.assertEqual(
            '-1:0000000000000000000000000000000000000000000000000000000000000000',
            result.parsed['src'],
        )
        self.assertEqual(
            '-1:3333333333333333333333333333333333333333333333333333333333333333',
            result.parsed['dst'],
        )

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            sync_core_client.boc.parse_message(params=params)

    def test_parse_block(self):
        block = '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'
        params = ParamsOfParse(boc=block)
        result = sync_core_client.boc.parse_block(params=params)
        self.assertEqual(
            '048f59d5d652459939ea5c5e7b291155205696b71e0c556f641df69e70e1e725',
            result.parsed['id'],
        )
        self.assertEqual(4296363, result.parsed['seq_no'])
        self.assertEqual(1600234696, result.parsed['gen_utime'])

        with self.assertRaises(TonException):
            params = ParamsOfParse(boc='Wrong==')
            sync_core_client.boc.parse_block(params=params)
