"""
Assistant Message module for MCP.

This module defines the AssistantMessage class, which represents messages
generated by an assistant in the Model Control Protocol (MCP) system. These
messages can contain different types of content, including text and images.
"""

from typing import Literal

from agentle.mcp.sampling.messages.image_message_content import (
    ImageMessageContent,
)
from agentle.mcp.sampling.messages.text_message_content import (
    TextMessageContent,
)
from rsb.models.base_model import BaseModel
from rsb.models.field import Field


class AssistantMessage(BaseModel):
    """
    Message generated by an assistant in the MCP system.

    AssistantMessage represents a message that has been generated by an AI assistant.
    It contains content which can be of different types (text or image) and has a fixed
    role of "assistant" to identify the sender in a conversation.

    Attributes:
        content (TextMessageContent | ImageMessageContent): The content of the message,
            which can be either text or image. The specific type is determined by the
            "type" discriminator field in the content.
        role (Literal["assistant"]): The role of the message sender, always set to
            "assistant" for this class.
    """

    content: TextMessageContent | ImageMessageContent = Field(
        description="Content of the message, either text or image"
    )
    role: Literal["assistant"] = Field(
        default="assistant",
        description="Role of the message sender, always 'assistant'",
    )
