# Pond
Pond is a high performance object-pooling library for Python, it has a smaller memory usage and a higher borrow hit rate.  For more details, see our user's guide or my blog([https://qin.news](https://qin.news/pond/)).

[English](https://github.com/T-baby/pondpond/blob/master/README.md) | [中文版](https://github.com/T-baby/pondpond/blob/master/README_ZH.md)

Pond is probably the first publicly available object pooling library in Python with complete test cases, over 90% coverage, complete code comments, and good documentation.

Inspired by Apache Commons Pool, Netty Recycler, HikariCP, Caffeine. Pond counts the frequency of usage of each object pool using approximate counting with extremely little memory usage, and recycles it automatically.

The default policy and weights reduce the memory usage by 48.85% and the borrow hit rate at 100% when the traffic is more randomly averaged.
![](https://raw.githubusercontent.com/T-baby/pondpond/master/images/1.png)

The default policy and weights reduce the memory usage by 45.7% and the borrow hit rate at 100% when the traffic is more in line with the 2/8 law.

![](https://raw.githubusercontent.com/T-baby/pondpond/master/images/2.png)

# Overview

Using Pond requires the implementation of an object factory, PooledObjectFactory, which provides object creation, initialization, destruction, validation, and other operations called by Pond. So in order for the object pool to support holding completely different objects, Pond uses a dictionary to save the name of each factory class and the instantiated objects of the factory class it implements.

Ponds are thread-safe for borrowing and recycling, and Python's queue module provides a first-in, first-out (FIFO) data structure for multi-threaded programming. It can be used to safely pass messages or other data between producer and consumer threads. Locks are handled by the caller, and all multiple threads can safely and easily work with the same Queue instance. The borrowing and recycling of a Pond is all about manipulating the queue, so it is basically considered thread-safe.

The auto-recycle is executed at regular intervals, 300s by default. Automatically cleans up the objects in the infrequently used object pool.


# User guide

To install Pond via pip, use the following pip command:

```shell
pip install pondpond
```

```python
from pond import Pond, PooledObjectFactory, PooledObject
```

First you need to declare a factory class for the type of object you want to put in. For example, in the example below we want the pooled object to be Dog, so we first declare a PooledDogFactory class and implement PooledObjectFactory.

```python
class Dog:
    name: str
    validate_result:bool = True


class PooledDogFactory(PooledObjectFactory):
    def creatInstantce(self) -> PooledObject:
        dog = Dog()
        dog.name = "puppy"
        return PooledObject(dog)

    def destroy(self, pooled_object: PooledObject):
        del pooled_object

    def reset(self, pooled_object: PooledObject) -> PooledObject:
        pooled_object.keeped_object.name = "puppy"
        return pooled_object

    def validate(self, pooled_object: PooledObject) -> bool:
        return pooled_object.keeped_object.validate_result
```

You need creat a new instance of Pond：

```python
pond = Pond(borrowed_timeout=2,
            time_between_eviction_runs=-1,
            thread_daemon=True,
            eviction_weight=0.8)
```

Pond can be passed a number of parameters in:

`borrowed_timeout`: The maximum duration of the borrowed object. Defaults to 60.

`time_between_eviction_runs`: The interval for automatic recycling. Defaults to 300.

`thread_daemon`: A boolean value indicating whether the pond's thread is a daemon thread. Defaults to True.

`eviction_weight`: Automatic recycling weight. Defaults to 0.8.

`loop`: Automatic recycling weight. Defaults to 0.8.

Creat a new instance of your custom class of fatory：

```python
factory = PooledDogFactory(pooled_maxsize=10, least_one=False)
```

`pooled_maxsize`: The maximum number of objects that can be placed in the object pool of the objects generated by this factory class.

`least_one`: If True, the object pool of objects generated by this factory class will retain at least one object when it enters auto-recycle.

Register the factory object with Pond; by default, the class name of the factory class is used as the PooledObjectTree's key.

```python
pond.register(factory)
```

Of course you can also customize its name：

```python
pond.register(factory, name="PuppyFactory")
```

If the register succeeds, the Pond will begin creating objects based on the pooled_maxsize set in the factory until the pool is full.

Borrow and recycle object:

```python
pooled_object: PooledObject = pond.borrow(factory)
dog: Dog = pooled_object.use()
pond.recycle(pooled_object, factory)
```

Borrow and recycle object by name:

```python
pooled_object: PooledObject = pond.borrow(name="PuppyFactory")
dog: Dog = pooled_object.use()
pond.recycle(pooled_object, name="PuppyFactory")
```

Clear a object pool:

```python
pond.clear(factory)
```

Clear a object pool by name:

```python
pond.clear(name="PuppyFactory")
```

## More Details
For more details, see our user's guide or my blog（[https://qin.news](https://qin.news/pond/)）.

## References
```
@software{Pond,
  author = {Andy Qin},
  title = {{Pond: A high performance object-pooling library for Python}},
  year = {2022},
  url = {https://github.com/T-baby/pondpond},
}
```

