import warnings
import math
import time

from artap.problem import Problem
from artap.results import GraphicalResults
from artap.algorithm_genetic import NSGAII
from artap.datastore import FileDataStore
from artap.algorithm_sweep import SweepAlgorithm
from artap.operators import LHSGeneration
from artap.surrogate_scikit import SurrogateModelScikit

from agrossuite import agros

from sklearn.gaussian_process import GaussianProcessRegressor
from sklearn.gaussian_process.kernels import RBF, Matern, DotProduct, WhiteKernel, ConstantKernel, RationalQuadratic, ExpSineSquared

from sklearn.model_selection import GridSearchCV
from sklearn.model_selection import RandomizedSearchCV

from sklearn.metrics import r2_score

from numpy import array
import numpy as np


class TestProblem(Problem):
    def __init__(self, name):
        parameters = {'R2': {'initial_value': 3.0, 'bounds': [2.6, 3.4]},
                      'h2': {'initial_value': 1.0, 'bounds': [0.408, 2.2]},
                      'd2': {'initial_value': 0.3, 'bounds': [0.1, 0.4]}}
        costs = ['F1', 'F2']

        working_dir = "team_22/"
        super().__init__(name, parameters, costs, working_dir=working_dir)

    def evaluate(self, x: list):
        # problem
        problem = agros.problem(clear=True)
        problem.coordinate_type = "axisymmetric"
        problem.mesh_type = "triangle"
        problem.parameters["J1"] = 2.25e+7
        problem.parameters["J2"] = -2.25e+7
        problem.parameters["R1"] = 2.0
        problem.parameters["R2"] = x[0]
        problem.parameters["d1"] = 0.27
        problem.parameters["d2"] = x[2]
        problem.parameters["h1"] = 1.6
        problem.parameters["h2"] = x[1]

        # fields
        # magnetic
        magnetic = problem.field("magnetic")
        magnetic.analysis_type = "steadystate"
        magnetic.matrix_solver = "umfpack"
        magnetic.number_of_refinements = 2
        magnetic.polynomial_order = 2
        magnetic.adaptivity_type = "disabled"
        magnetic.solver = "linear"

        # boundaries
        magnetic.add_boundary("A = 0", "magnetic_potential", {"magnetic_potential_real": 0})
        magnetic.add_boundary("Neumann", "magnetic_surface_current", {"magnetic_surface_current_real": 0})

        # materials
        magnetic.add_material("Coil 1", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": "J1",
                                         "magnetic_permeability": 1, "magnetic_remanence": 0,
                                         "magnetic_remanence_angle": 0, "magnetic_total_current_prescribed": 0,
                                         "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0,
                                         "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})
        magnetic.add_material("Coil 2", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": "J2",
                                         "magnetic_permeability": 1, "magnetic_remanence": 0,
                                         "magnetic_remanence_angle": 0, "magnetic_total_current_prescribed": 0,
                                         "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0,
                                         "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})
        magnetic.add_material("Air", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": 0,
                                      "magnetic_permeability": 1, "magnetic_remanence": 0,
                                      "magnetic_remanence_angle": 0, "magnetic_total_current_prescribed": 0,
                                      "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0,
                                      "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})

        # geometry
        geometry = problem.geometry()
        geometry.add_edge("R1-d1/2", 0, "R1+d1/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge("R1+d1/2", 0, "R1+d1/2", "h1/2")
        geometry.add_edge("R1+d1/2", "h1/2", "R1-d1/2", "h1/2")
        geometry.add_edge("R1-d1/2", "h1/2", "R1-d1/2", 0)
        geometry.add_edge("R2-d2/2", "h2/2", "R2+d2/2", "h2/2")
        geometry.add_edge("R2+d2/2", "h2/2", "R2+d2/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge("R2-d2/2", "h2/2", "R2-d2/2", 0)
        geometry.add_edge("R2-d2/2", 0, "R2+d2/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge(0, 0, "R1-d1/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge("R1+d1/2", 0, "R2-d2/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge(0, 25, 0, 0, boundaries={"magnetic": "A = 0"})
        geometry.add_edge(25, 0, "R2+d2/2", 0, boundaries={"magnetic": "Neumann"})
        geometry.add_edge(0, 25, 25, 25, boundaries={"magnetic": "A = 0"})
        geometry.add_edge(25, 25, 25, 0, boundaries={"magnetic": "A = 0"})

        geometry.add_label(12, 12, materials={"magnetic": "Air"})
        geometry.add_label("R1", "h1/4", materials={"magnetic": "Coil 1"})
        geometry.add_label("R2", "h2/4", materials={"magnetic": "Coil 2"})

        # recipes
        magnetic.add_recipe_volume_integral("Wm", "magnetic_energy", [], -1, -1)
        magnetic.add_recipe_local_value("B00", "magnetic_flux_density_real", "magnitude", 0, 10, -1, -1)
        magnetic.add_recipe_local_value("B01", "magnetic_flux_density_real", "magnitude", 1, 10, -1, -1)
        magnetic.add_recipe_local_value("B02", "magnetic_flux_density_real", "magnitude", 2, 10, -1, -1)
        magnetic.add_recipe_local_value("B03", "magnetic_flux_density_real", "magnitude", 3, 10, -1, -1)
        magnetic.add_recipe_local_value("B04", "magnetic_flux_density_real", "magnitude", 4, 10, -1, -1)
        magnetic.add_recipe_local_value("B05", "magnetic_flux_density_real", "magnitude", 5, 10, -1, -1)
        magnetic.add_recipe_local_value("B06", "magnetic_flux_density_real", "magnitude", 6, 10, -1, -1)
        magnetic.add_recipe_local_value("B07", "magnetic_flux_density_real", "magnitude", 7, 10, -1, -1)
        magnetic.add_recipe_local_value("B08", "magnetic_flux_density_real", "magnitude", 8, 10, -1, -1)
        magnetic.add_recipe_local_value("B09", "magnetic_flux_density_real", "magnitude", 9, 10, -1, -1)
        magnetic.add_recipe_local_value("B10", "magnetic_flux_density_real", "magnitude", 10, 10, -1, -1)
        magnetic.add_recipe_local_value("B11", "magnetic_flux_density_real", "magnitude", 10, 10, -1, -1)
        magnetic.add_recipe_local_value("B12", "magnetic_flux_density_real", "magnitude", 10, 9, -1, -1)
        magnetic.add_recipe_local_value("B13", "magnetic_flux_density_real", "magnitude", 10, 8, -1, -1)
        magnetic.add_recipe_local_value("B14", "magnetic_flux_density_real", "magnitude", 10, 7, -1, -1)
        magnetic.add_recipe_local_value("B15", "magnetic_flux_density_real", "magnitude", 10, 6, -1, -1)
        magnetic.add_recipe_local_value("B16", "magnetic_flux_density_real", "magnitude", 10, 5, -1, -1)
        magnetic.add_recipe_local_value("B17", "magnetic_flux_density_real", "magnitude", 10, 4, -1, -1)
        magnetic.add_recipe_local_value("B18", "magnetic_flux_density_real", "magnitude", 10, 3, -1, -1)
        magnetic.add_recipe_local_value("B19", "magnetic_flux_density_real", "magnitude", 10, 2, -1, -1)
        magnetic.add_recipe_local_value("B20", "magnetic_flux_density_real", "magnitude", 10, 1, -1, -1)
        magnetic.add_recipe_local_value("B21", "magnetic_flux_density_real", "magnitude", 10, 0, -1, -1)

        computation = problem.computation()
        computation.solve()

        solution = computation.solution("magnetic")

        B00 = solution.local_values( 0, 10)["Br"]
        B01 = solution.local_values( 1, 10)["Br"]
        B02 = solution.local_values( 2, 10)["Br"]
        B03 = solution.local_values( 3, 10)["Br"]
        B04 = solution.local_values( 4, 10)["Br"]
        B05 = solution.local_values( 5, 10)["Br"]
        B06 = solution.local_values( 6, 10)["Br"]
        B07 = solution.local_values( 7, 10)["Br"]
        B08 = solution.local_values( 8, 10)["Br"]
        B09 = solution.local_values( 9, 10)["Br"]
        B10 = solution.local_values(10, 10)["Br"]
        B11 = solution.local_values(10, 10)["Br"]
        B12 = solution.local_values(10,  9)["Br"]
        B13 = solution.local_values(10,  8)["Br"]
        B14 = solution.local_values(10,  7)["Br"]
        B15 = solution.local_values(10,  6)["Br"]
        B16 = solution.local_values(10,  5)["Br"]
        B17 = solution.local_values(10,  4)["Br"]
        B18 = solution.local_values(10,  3)["Br"]
        B19 = solution.local_values(10,  2)["Br"]
        B20 = solution.local_values(10,  1)["Br"]
        B21 = solution.local_values(10,  0)["Br"]

        Wm = solution.volume_integrals()["Wm"]

        of1 = (B00**2 + B01**2 + B02**2 + B03**2 + B04**2 + B05**2 + B06**2 + B07**2 + B08**2 + B09**2 + B10**2 + B11**2 + B12**2 + B13**2 + B14**2 + B15**2 + B16**2 + B17**2 + B18**2 + B19**2 + B20**2 + B21**2)/22/9e-6
        of2 = math.fabs(2*Wm - 180e6) / 180e6

        return [of1, of2]


def scikit():
    problem = TestProblem("AgrosProblem")
    # create data store
    problem.data_store = FileDataStore(problem, database_name=problem.working_dir + "/data_surrogate.db")

    # enable surrogate
    problem.surrogate = SurrogateModelScikit(problem)
    problem.surrogate.regressor = GaussianProcessRegressor(kernel=1.0 * RBF(length_scale=1.0, length_scale_bounds=(1e-1, 10.0)), n_restarts_optimizer=9)
    problem.surrogate.train_step = 800
    problem.surrogate.score_threshold = 0.0

    # sweep analysis (for training)
    gen = LHSGeneration(problem)
    gen.init(problem.surrogate.train_step)
    algorithm_sweep = SweepAlgorithm(problem, generator=gen)
    algorithm_sweep.run()

    print(problem.surrogate.x_data)
    print(problem.surrogate.y_data)

    # optimization
    # NSGA_II
    algorithm = NSGAII(problem)
    algorithm.options['max_population_number'] = 100
    algorithm.options['max_population_size'] = 15
    algorithm.run()

    problem.logger.info("surrogate.predict_counter: {}".format(problem.surrogate.predict_counter))
    problem.logger.info("surrogate.eval_counter: {}".format(problem.surrogate.eval_counter))

    results = GraphicalResults(problem)
    results.plot_scatter('F1', 'F2', filename="/tmp/scatter.pdf")


def plot():
    problem = Problem()
    problem.data_store = FileDataStore(problem, database_name="team_22/data_surrogate.db", mode="read")

    results = GraphicalResults(problem)
    results.plot_scatter('F1', 'F2', filename="/tmp/scatter.pdf")


def regressor_data():
    # 800 samples
    x_data = [[2.870934443020384, 1.5318638159238287, 0.35717809048733795], [3.3003845143661033, 1.7977244701176764, 0.2591008562785658], [2.7223119470351507, 1.783045134954803, 0.25192850740457096], [2.604195838110124, 0.5362256180159962, 0.34699003897713676], [2.851159066355722, 0.7562584088690385, 0.3755149806992789], [2.7192464699356096, 0.6447968809604556, 0.32420115085469836], [2.6616361226559784, 0.45894255666613415, 0.32552467459355205], [3.297533113160648, 1.8371451568213055, 0.24106307999377258], [3.2630729042930087, 1.7542598545656085, 0.11497867894258644], [3.162919963932449, 1.7478550283177678, 0.3215059932290157], [2.8612553400909286, 0.9601964150039735, 0.1754756693782998], [2.9953199747233965, 2.1019840600467723, 0.3919058139375494], [3.0490922603583006, 1.560689264973026, 0.169211705806008], [2.7694786358724377, 1.4898418210870754, 0.3146292916108874], [3.2646015133464505, 1.5054642463653267, 0.23541041808758487], [3.1588238866693374, 1.938522471298158, 0.3855459212335466], [3.3348495211890588, 1.7317741585909219, 0.11200232900229164], [2.9770942205139748, 1.0465720981080524, 0.38158655296177046], [2.9414352818173386, 2.131762711927383, 0.1647025940078253], [2.6120321009844183, 1.4205021642635793, 0.321029495277635], [2.785795104811955, 0.9395984705699745, 0.2775279673206394], [3.3907758451470222, 1.8644762884990393, 0.1855641698997556], [2.9872341239974842, 1.8675946423396335, 0.3789496125751529], [3.30202951912573, 2.1364345055602563, 0.20867445195692652], [3.007487928818076, 0.5095308209532569, 0.28869742159685186], [3.2146976524921893, 0.5991900763081482, 0.12388527432209917], [2.650883746395509, 0.7415090555631221, 0.31920308450046764], [3.2356731034646518, 1.0864748526454098, 0.38396596423190166], [2.613807236691483, 1.558030214154457, 0.14347416776022226], [2.68053316937874, 0.7849643634894745, 0.2700755838468276], [3.1643689490731317, 0.8925750766126623, 0.3777769861473075], [3.344144458200745, 1.97785543680803, 0.1473142842292426], [3.0755975426866677, 1.8173090957932898, 0.1382408475730966], [3.0714896734018966, 1.230864534240342, 0.22387893140315726], [3.3886063475320736, 2.016110284533356, 0.34234773205152], [3.1162560015409664, 1.0806194863953054, 0.2743974767330376], [3.394172392661889, 0.6193303553669315, 0.15378018945024274], [2.745846150331589, 1.8588999366762722, 0.3563831043426283], [3.3840567427652912, 0.45566296955446056, 0.3975636866162845], [2.8831064712255037, 1.4792877186042055, 0.3276305367073378], [3.23359982983453, 1.0144126012892631, 0.23013535108783478], [2.6261371023838658, 0.8304597430878558, 0.20159480391889717], [3.0461550332908245, 0.847078898443923, 0.29580160898920693], [2.900802694032921, 1.5125367037591781, 0.23162045216136867], [3.332086641984342, 0.8143605108520103, 0.3285494870289315], [2.7212564897802274, 1.6313012371600961, 0.15066543338690747], [3.0147522007606895, 1.0193279183950155, 0.2634594195778017], [2.698749850023281, 1.8781408652691605, 0.12088517506028601], [2.7689558540551165, 1.0974744064646578, 0.30618427413203986], [3.3641546798612323, 1.4229675013932512, 0.24038728790245745], [2.8624780677468165, 0.5771992836546654, 0.3344932046563597], [3.217643457031949, 1.3289522199574322, 0.21569833854901538], [3.2202130841889267, 0.9916087038166239, 0.30184042652691656], [3.3992272743480694, 1.346937893023998, 0.25301788783724255], [3.105877959097996, 0.7245578391299797, 0.2737804688377692], [2.996322859628199, 0.9886981942106827, 0.16952250590384696], [2.9696076947768817, 1.4068670067169444, 0.16069808068246114], [2.641211723159581, 0.6311626240094153, 0.3340289920560552], [2.9328392791779714, 0.4106024533534802, 0.17643595568757653], [3.3521794748179112, 0.7179546210334822, 0.3744679396389905], [3.317710852562381, 1.6535127548948296, 0.3653000700472485], [3.2751133171712974, 1.803509014646978, 0.217447325450818], [2.668068082078791, 1.3087631097662573, 0.12049255588333307], [3.1781640577070713, 1.7344645853258935, 0.24500118866444506], [2.9794991700261955, 0.5214846060121545, 0.34064251858279027], [3.0891661193990347, 2.195986310190726, 0.2759472202625353], [2.7898577774746127, 1.0401513361945243, 0.15928770905322012], [3.1851372605919153, 1.37043578433782, 0.3158364809872066], [2.926633488535486, 1.0717217548738565, 0.3135149052700521], [3.259018571764575, 2.094236340849324, 0.1777997533107671], [3.2295418673264615, 1.7655339952734168, 0.12244927257440132], [3.2769740522574593, 0.8245460412981862, 0.32084393014967405], [2.6822775733870983, 1.7683656831242869, 0.39021956848031203], [2.756272314280406, 1.4987510866474576, 0.36202088514735586], [3.211984562936927, 1.4137765397912017, 0.1883820789745594], [3.0302061801078666, 2.1537984454581545, 0.3218646798761018], [2.904644546083713, 0.4625919317416495, 0.16760581844287636], [2.98422124915518, 2.000025792560058, 0.39530950815445276], [3.343950239944153, 0.5318228720988466, 0.18353793807220142], [2.8255979018610478, 0.5800981688216293, 0.11300566475905197], [2.9936500318333605, 0.4369141205724364, 0.31256025062975024], [3.135775906249944, 1.1507516160045783, 0.25438515526706196], [3.3053816072175164, 1.1906987681184724, 0.32828338074076335], [3.1004568459578508, 1.3344338205478554, 0.11569222703620059], [3.1213727242583826, 1.7909375932018907, 0.13477778987795463], [3.1945229381216533, 1.1004222976513234, 0.20186567921499793], [2.8811054304537205, 0.6140271553825158, 0.38318829129880105], [2.952446063046887, 2.111240480931735, 0.10384970411348415], [3.062179086017203, 1.941013010950815, 0.13554428620496684], [3.2650222200572934, 1.9665711934629737, 0.1492979354637618], [3.2987838481952894, 1.4757811097598255, 0.15022062300750935], [3.101241473398236, 1.3504480462359585, 0.30636032468394514], [2.7381433296301445, 0.727047625314701, 0.1891205136189245], [3.1269479462198206, 1.331307219069187, 0.11073277409414012], [2.854813812744865, 1.2262492451115037, 0.13035060064369502], [2.6778698523061464, 1.6851383148371266, 0.2963775234080569], [3.2278071875712273, 1.2190890587761307, 0.31637076252248786], [2.603810607646495, 1.351725319678184, 0.10891758250885998], [2.922955687212716, 0.9499073020135935, 0.25403007631621527], [2.9343469429746065, 1.3408454236160279, 0.23289508389626284], [2.9394380586363402, 0.9113892294025283, 0.37860962450163627], [3.14622814804959, 2.0559383053800997, 0.2720091293133364], [3.1984108825370665, 1.4108961121091976, 0.2074892239368958], [2.814448721086953, 1.6815630657566547, 0.15038749215591665], [3.104217350838677, 2.1879613894803707, 0.13928786047222871], [2.651277062105746, 1.951718933523912, 0.3703610520944379], [3.1882971174312766, 0.642337374640506, 0.3182987949120712], [2.734873786225882, 1.2921431638119534, 0.30166673359472806], [3.385639937370949, 0.904172528840323, 0.2678732830333086], [3.33055792617021, 1.9746755363923971, 0.17028269284799155], [2.782618387840939, 0.8398946131702962, 0.28198188756339937], [2.617126045389957, 0.4888242620511556, 0.21880828236205396], [2.7240998834275407, 1.459290880725461, 0.35748081673300014], [2.7296953397533983, 1.7276303276629854, 0.33700772767162535], [2.6153981353996003, 1.9427054688585927, 0.37806045558769885], [3.1972998075438173, 0.9642781953681894, 0.3066584148600541], [2.802088768937349, 1.3673395001366417, 0.15775422674395032], [3.357515701863102, 1.5083041908904666, 0.15839024866510942], [2.660703596070901, 1.745033031930945, 0.1923748880222979], [2.6650051781230646, 2.0487619953838605, 0.12016610894920905], [3.3311759963513654, 0.44440369360329385, 0.29491559243101917], [3.2951568063493992, 1.2808257258352898, 0.15696824714364788], [2.9096662336555528, 1.0100044085682733, 0.3912407954027969], [3.3923757972447595, 0.48375591604491724, 0.13160912873216815], [2.704529249188919, 0.47514109204231497, 0.2758534724902574], [2.7132034929942135, 1.1530221452365257, 0.19089327755100688], [2.89437259102376, 0.8100767878654267, 0.24456593384792283], [3.333399432688817, 0.4945619974817301, 0.22018771290962413], [3.0486267395306785, 1.7411220627132948, 0.38940794301926185], [2.720277033906494, 0.8894090147757594, 0.25946425256093975], [3.2820866591063633, 1.6946195198426692, 0.12534466965743316], [2.9815693239384577, 1.1847248891023263, 0.13640176198934656], [2.8808369425790583, 1.0841110167774843, 0.28454085677212737], [3.1544125526052578, 1.9494333902704926, 0.11026930194431234], [2.8050382261884153, 0.7950400908660455, 0.2813019303666374], [3.0761502024341145, 1.2472077738614906, 0.18213220058573237], [2.9880620306198056, 1.1980322579603013, 0.24692038411605494], [3.072293012082793, 1.1889531821540809, 0.23945029239294574], [2.9556441604566475, 1.7061411942940259, 0.16440283467734418], [2.7324453054994065, 0.9528388893915096, 0.27264064165671903], [2.8243768269015033, 0.42680411757736386, 0.2279404085108611], [2.958089628258764, 0.5195109305164304, 0.19562514697852545], [2.896321032188456, 0.9184733079598024, 0.235345550421483], [3.3670465237158798, 1.7334352340338774, 0.35166073559994804], [3.1304265224952132, 0.617324818980175, 0.25257667394837563], [3.287535199871796, 1.022352013628746, 0.11736968243385341], [3.094430882378552, 1.775975560071237, 0.35345761272735754], [3.0831379157235417, 0.5043812594542199, 0.11987406694445824], [2.6075214059228884, 1.2061710683248115, 0.12429719510376883], [3.2163626165159025, 1.1449368680119572, 0.3274906060287811], [2.8581537315551917, 0.6399472793640245, 0.24024601260952605], [2.889045615543252, 1.1255661169741609, 0.20375870433150314], [2.9168346343623437, 0.8846866273726073, 0.3610751560782022], [2.7805891927000452, 2.058020111456153, 0.39369248159159764], [2.917117720474384, 1.756953529740815, 0.2420988871263308], [3.3219456740772757, 0.9295379447626642, 0.2666615703311753], [2.7436775649372183, 1.9614595569024922, 0.34840657175301853], [2.965052649279491, 0.5670779258038883, 0.3822838810042021], [2.7231723401824453, 1.7469738212726273, 0.2231805229051414], [3.045233114263205, 0.6804259102992483, 0.383106741651119], [3.3509220391270325, 1.9807405969599645, 0.36675117266520096], [2.6576174754868975, 2.1846923204627493, 0.34760954936941624], [3.310508553887948, 0.8851532222258488, 0.36337634816269926], [2.876538239057234, 2.1785039104356225, 0.3511205710121005], [3.2554009018091445, 1.3844550793091943, 0.3291000335919453], [3.037169148716832, 2.059879361114223, 0.3222974593127925], [2.7954272763014707, 1.4394586060468522, 0.18181397315490475], [3.0478168623824597, 1.282072548074301, 0.3812176148320632], [2.8879985719111594, 1.4038178712784055, 0.208778263534461], [3.3704485580425496, 0.9509480195454938, 0.3429124495718713], [3.1336962474676926, 0.5535883543145518, 0.38055809204589364], [3.0213532841244066, 0.6687527402099801, 0.3681817599904852], [2.8576747124789748, 1.1129170342591548, 0.1587280115558673], [3.02945180783809, 1.4668664185383877, 0.29851035349491806], [2.6389947929666073, 1.629078357203475, 0.33276841633888987], [3.13721233566304, 1.5110536459713326, 0.3731644685389848], [2.9427071443324486, 1.0306678710871238, 0.1748499495047452], [3.161725541480706, 2.1630700325947845, 0.14903514486998481], [2.7420984585229617, 0.5605932110344637, 0.23876016434204475], [3.215508137843243, 1.1688505162882719, 0.17715306184814505], [2.762350893349018, 0.601411974166867, 0.390348576417468], [2.6716810978285417, 0.7195451441242166, 0.24063267540273983], [3.26623193822769, 1.3552794679629152, 0.21483979424050842], [3.3149167873256133, 1.1869503161116315, 0.268644736489294], [3.1490692342859603, 0.906621036712083, 0.1193314416653149], [2.8827474859882374, 1.9220071727623946, 0.2110715254699923], [2.9245203108360998, 1.860050621009758, 0.2561520657066], [2.914281790871775, 2.175364326942345, 0.32021885428819946], [2.6215009983541444, 1.0326597104328104, 0.19332317317727854], [2.7971948274124614, 1.4770872495282603, 0.3324416703770867], [3.1288243998729564, 1.0631064335756826, 0.12884257027311266], [3.0051944866543345, 1.7205658384026137, 0.1334985170129082], [2.7175352280682348, 1.7649105821959141, 0.1167423718371644], [3.3957169438805312, 1.473561642060275, 0.132562628420636], [2.6359869237872378, 0.5845234754872171, 0.221452069849781], [3.208162023255536, 1.9893030094502493, 0.26280500045253063], [3.1123436248944785, 0.9226204987516042, 0.3460086781979499], [2.7841592591490123, 1.6042956839235745, 0.10299765134947986], [2.8931659923035755, 1.6694391441929477, 0.25846732791548854], [2.827311138891374, 1.6636373563438673, 0.3699535351687343], [3.056120532782247, 1.6971601218551529, 0.1935601093339031], [3.241892458475662, 0.9205906559934312, 0.3392281138244031], [3.1416797978690822, 2.047070765660799, 0.2459860524360928], [3.2322528315178545, 2.0307820634595624, 0.28432285633657234], [3.225439838259356, 0.7227835922615248, 0.21160769660279968], [3.2314839903849553, 0.5807809896111894, 0.3523529551283303], [3.012552629611588, 1.416534065044607, 0.20558002628260272], [3.05423962124399, 0.5868871746584818, 0.36808298491652114], [3.111155665819468, 0.4153487626488748, 0.21290666901360078], [2.7091963582958347, 0.8600179540894177, 0.16736561896633473], [2.6943409885484044, 1.0754972177835909, 0.30856956282326065], [3.156437550349628, 0.8762894238423535, 0.31675069653804416], [3.3695498450304155, 0.8744623117838, 0.1743422149644806], [2.7159440701257234, 1.453751883686317, 0.1544170876658961], [3.3755326791147766, 0.8500882670020826, 0.16704378648690255], [3.299803494822102, 1.0904603642077975, 0.21663342431841026], [2.6990942795606254, 1.312384987815533, 0.39637294341331153], [2.630464813974497, 0.9786451946389363, 0.28929938599877014], [3.1147497312897006, 1.1075454593703382, 0.2482979207457925], [3.1639958684817997, 1.7014936466571382, 0.3383804380382547], [3.3586832840799774, 1.2425863789811307, 0.19161203550086736], [2.9183718510383416, 1.485248628129839, 0.16099098804286077], [3.065024489255071, 1.9315700311069772, 0.3686244894935413], [3.1766648213251454, 1.1208304458030713, 0.1395655133788134], [3.0846642323015487, 0.556958286854054, 0.36149921016086506], [3.1844692637575704, 0.9090742203300315, 0.357965136227243], [2.744397030787826, 0.7133657984496112, 0.1350514099029039], [3.3084687492347005, 1.2346488330077459, 0.3228972814262955], [3.0870394644718475, 1.5679946001115979, 0.13149676529785173], [3.1571716733149207, 1.547151049413344, 0.1879920873427116], [2.610161004617901, 0.621158911948098, 0.3333760834620464], [2.8559969714340263, 1.241095860029885, 0.3312641932000156], [3.383323229280819, 1.5076816959996435, 0.17175178330938862], [2.912609589390875, 1.2225820802408391, 0.15716933543817946], [3.351779069324754, 2.0962596185472986, 0.2246013962032637], [2.9519632065907087, 1.8630817979472647, 0.2913131124425083], [2.8776885109313084, 1.4125385126601089, 0.1482550734835772], [3.2480639050011715, 2.0661318713373555, 0.1555360656090165], [3.113085290857353, 0.5018519074204387, 0.323686285170363], [3.1951832057515417, 1.3771485188874182, 0.3350762261303948], [2.8787477013362985, 0.5068136706859273, 0.2803980929939713], [2.8188448839170492, 1.3896853166715808, 0.2396693476911812], [2.8525946546142413, 1.2494893415379575, 0.32705146353108194], [2.8454746172150154, 1.3216332743902028, 0.12336382269227847], [3.366801269256949, 1.0704217827457663, 0.22604860337777646], [3.127767473836281, 1.3223336083651014, 0.10712247129752592], [2.9805458350258185, 1.3652263415451877, 0.10340282903940835], [3.0198525409060406, 1.2705184509220293, 0.15643654361309714], [2.8537662164941007, 1.2697880472024479, 0.3361542689608653], [2.816435109189275, 1.4459230302108599, 0.34465970229619025], [3.328811411472905, 2.146700765926614, 0.26140937017861376], [2.7169558002327783, 1.0279910815005444, 0.30960083113975856], [2.864506487558606, 1.5995110116004951, 0.15534852496445653], [2.8359036435030665, 0.8050448547594831, 0.22082840467038817], [3.252016467432807, 1.6066109494134517, 0.23677148393984282], [3.0578059696210316, 0.9825131878280171, 0.3637940369188766], [2.8390170087205036, 2.029510681143053, 0.3883321329777061], [2.6690081190696886, 2.115258969927309, 0.24850450248295566], [3.2210183695526955, 0.5702984075462101, 0.20510559058289102], [3.004436878488466, 2.002801683482028, 0.3591917104717923], [2.8031131868575914, 0.7041696417525025, 0.3042071500537264], [2.954913235567053, 1.9734591649669908, 0.37101050188194284], [2.9708450075778643, 1.8947563556667426, 0.18854475176233892], [3.1229173540872717, 1.1180019593633626, 0.2104943598399371], [2.7933193316495912, 1.6026363415722948, 0.2161835971877141], [2.915257971365071, 1.9277069805998863, 0.11790668619248773], [2.989327362002812, 1.2532960672960558, 0.12782773609400605], [3.0772523688859317, 0.4372537480021559, 0.3646220877507319], [2.8741047397859383, 1.9839756009275036, 0.33548535769993165], [3.236435601896602, 1.9149756729179306, 0.25675797362439357], [2.8387317646466927, 1.8839487093956537, 0.39596737946560123], [3.381675440716878, 0.7791190955046595, 0.2020538412345726], [3.170704243002378, 0.819592702896157, 0.2606073688645335], [2.643868194462931, 1.4075628650153444, 0.12915980466906163], [3.2071771270079, 0.8377430999070652, 0.15923417268391488], [2.622272506154791, 1.5221661860895341, 0.12307386746889964], [3.0825510984005238, 1.9370085235594616, 0.19687877853434566], [3.201699147684293, 0.7745770743054343, 0.28853682841642486], [3.073523811391756, 1.1322760776799574, 0.27845433937083125], [2.9729700403635917, 1.6122431619814395, 0.35148803270618745], [3.3877391658558493, 0.488106694897741, 0.271034497082024], [3.3388189367378867, 0.5637050954939505, 0.33080822659173514], [3.0818398633818704, 1.2144935598504782, 0.10789531749253024], [3.3622977023542244, 0.8083459167950873, 0.28811874333445664], [2.7674758003207818, 1.8079347495511102, 0.269149028043987], [2.623746481520454, 1.6750480757702138, 0.3942882530395998], [2.620355061177045, 1.1051039737578259, 0.3358415055900029], [2.943830538988266, 2.1897425739364196, 0.33943823821405694], [3.1395998211001577, 1.772645687858829, 0.25126230494578883], [3.0444920316709374, 1.7364850861325352, 0.22200679661751044], [3.226383025472277, 0.9757278908711637, 0.23448530434040712], [2.606737503709706, 1.2766508059539807, 0.2860439296645584], [2.9035800259284223, 2.004118576196734, 0.3266261807501114], [3.2222242701126236, 0.7108741456857237, 0.17143514436708526], [2.9472328281582323, 2.124500861881105, 0.1843268993447854], [3.2807563025013255, 0.5493169187028475, 0.24662373915237829], [2.9376683689177754, 0.9252160517537371, 0.17959683126435239], [2.950122228044937, 2.0339028574894735, 0.34534243976344053], [3.323405831582758, 1.4649682607664893, 0.24524042310865332], [3.3180684214680065, 0.9356229395771889, 0.1015279024067608], [2.8758031780248423, 1.9567355939239446, 0.20995446453019184], [3.2608597610546886, 0.902810177894648, 0.33676210154888786], [2.8887613201390043, 1.1363893369657534, 0.1682603996861634], [2.6965186083183323, 0.9125457354658155, 0.1378362572181267], [2.649554125018293, 1.5719827218732723, 0.22730433721983548], [3.1908249861190714, 1.9453198483595768, 0.20420649931116958], [3.2944503564099903, 1.8307092764211688, 0.3401294040940036], [2.737586223215637, 2.1609166371500934, 0.134344601398771], [3.193026338619915, 0.7159280255508269, 0.3101732626465381], [2.8109891986767406, 1.8486068429669935, 0.22164783102791874], [2.7014110540508476, 1.1347472172029012, 0.15407896957594144], [3.307478530919014, 1.2625791907120147, 0.24392445109796201], [3.3467542746920746, 1.7587670721589588, 0.32475423883239585], [3.175487306574171, 1.29664331605518, 0.30751660917003615], [3.2855879285165135, 0.4411612190230603, 0.12280562600827839], [2.804151843189479, 1.648095869408869, 0.16981020115653367], [2.976626091730948, 0.7826447719733212, 0.23871969515484592], [3.0322034317411486, 1.8914874691141017, 0.185985613772742], [2.7551601593888577, 1.99435535531788, 0.28761822946989435], [3.0095513106487553, 0.8624199525752103, 0.18054084203205475], [3.2720263719301528, 1.1624330985551032, 0.20591872461668423], [2.8591934183961922, 0.7968899028935608, 0.26263813208185505], [3.024750561651561, 2.0428146115824632, 0.2828224139808546], [2.7903485851147916, 0.7997366183570911, 0.3745854937205384], [2.6273994813271173, 0.5165356899405734, 0.37180692205035026], [2.9576354020963516, 1.0021825690726196, 0.1008783540076832], [2.9750646853949956, 1.099494930317655, 0.16878143394982476], [3.256570102503943, 2.113976520716176, 0.2664798527193677], [3.2735533775374206, 0.5256738892615493, 0.36287717137396447], [3.3036854400382003, 0.6092021410735354, 0.38639202173079534], [3.151878154471695, 1.0370850952099318, 0.16303237640008902], [3.3377837699104003, 2.1707626303459273, 0.17092778871113953], [2.8171455331645414, 1.8442002570343488, 0.264827827238338], [2.6813127200404323, 1.5687804571243813, 0.11807894612243851], [3.0976316755051982, 1.6843120686479593, 0.11369670535011647], [3.33979907972371, 0.5960039199329783, 0.29721678007456864], [3.001873049174172, 0.790406257920186, 0.11896491503995674], [3.0904502007430814, 2.1212813970728557, 0.2955773658522899], [2.9601802581540047, 1.2566013840972334, 0.25996732155283064], [2.9235501895550278, 2.023856959748515, 0.3055177572936716], [2.690086738394286, 0.8966061871161595, 0.25478936221066295], [3.0169300462256716, 1.8094759422857218, 0.36590201331796823], [2.866942296060388, 1.1782915952435178, 0.10140146266089488], [2.8284621453031225, 0.8019864857817696, 0.1476926054099135], [2.77072452084431, 1.5919735934503778, 0.1681173592221918], [2.6797106633776018, 0.8655071940415263, 0.17424350157939256], [3.0131221911187565, 1.9696251934680404, 0.37672291802691493], [3.189447667153943, 0.8635374873867601, 0.17658414832384123], [3.25765548016522, 1.4330458790885627, 0.2644165756615761], [2.718593632943735, 1.5288718119601212, 0.2810276094985173], [2.779140044595779, 1.3011198698416377, 0.38362568919922024], [3.0938446994685402, 0.5558183039580525, 0.3409487718126286], [2.646587043219798, 0.47888048066608935, 0.3526894472971871], [3.244576225539242, 0.6033348444211639, 0.24796529147444815], [2.666520016379993, 1.39914682778384, 0.3986296331953396], [2.6440160480686865, 2.0794169016887216, 0.15976099851864875], [3.251702040190438, 1.9082006122295851, 0.1994297707681803], [2.9855748808955753, 1.1460226316684607, 0.10046149448102186], [3.1150745319572763, 0.637700319837399, 0.17217900876607406], [2.8604951882906478, 1.7847845163136948, 0.19554884630535124], [2.8696058564076075, 1.6215176385203476, 0.1340006214005088], [2.9453536323263423, 2.0632759550770916, 0.1530848204517606], [3.288986018946124, 1.68747639834938, 0.24993599570383998], [3.27092315952352, 2.0769266298656843, 0.21991829585982298], [2.6559460572824816, 1.9016552510067775, 0.2331899929326205], [3.0366086497888483, 1.7808375551969422, 0.31059817131581413], [3.0638157529791785, 1.0504628059471142, 0.11160323612704066], [2.614694182742437, 1.1697574462457594, 0.29231717836449367], [2.850522077377212, 2.076063506437962, 0.3051536159850663], [3.1925699872307174, 1.4680288371931376, 0.12119691930162856], [3.3279553536231274, 0.44901462562027467, 0.3473152200729157], [2.9253730140323038, 0.4315026099199605, 0.24767434066642782], [3.18753591320112, 0.5418695096461027, 0.38271257057965247], [2.672303709201003, 0.741782038968936, 0.31214439832445817], [2.7865356516303916, 0.8113493687424153, 0.2867670182676204], [2.8424636998548123, 1.4820757100032496, 0.1602049324446713], [3.2810675177202704, 0.5113358106019325, 0.39360117307976594], [3.067039220910988, 2.1376329881559535, 0.20719245313667767], [3.3916882015634684, 0.8808400442070949, 0.3339794476733572], [3.268259566032622, 1.1718440332509614, 0.24248049568217567], [2.840811698530996, 1.5267603227937894, 0.3714054173030833], [2.9409843346753557, 0.6625009706141063, 0.35016545263450116], [3.0422706398281862, 1.0005559639866535, 0.22351022065280782], [2.6843405846133193, 1.2738732119021492, 0.32614267111591166], [3.0387048643911423, 0.428825653796256, 0.2773576715992356], [2.659829719157738, 1.3271049511006718, 0.12586877283207598], [3.228384681605644, 2.118386247411983, 0.3895147751041741], [2.928787502637331, 1.3818763747415304, 0.21736076657944498], [2.70075020650861, 1.1410963356589263, 0.33293696108331317], [2.997707651579368, 2.021518915223065, 0.16553973347200268], [3.365677408056274, 1.0604217086421086, 0.34190184416479086], [3.2543275364859823, 1.2453697065766087, 0.3804747456780049], [3.345689077753871, 1.2132695172586185, 0.25531895443341834], [2.686785050011575, 1.228709701641096, 0.27055273641291416], [3.080293049774126, 0.6635888698997094, 0.27073362285260716], [3.2391178191918693, 1.799965472249365, 0.3021395368227067], [3.0559644022274526, 0.6674106642781472, 0.2289703762750868], [3.361910126754045, 1.4956068463844094, 0.34384025580609073], [3.061471189362169, 1.794079892415991, 0.17576116051365737], [3.2038186225191048, 1.9894501438687264, 0.2695001709632912], [2.846497868623035, 0.5463395788653528, 0.3112526571720908], [2.982213626686503, 1.2987653220197894, 0.10243796288208044], [2.62908208183134, 1.4423009863674519, 0.34352650067053114], [3.0029117019215144, 1.232538536412025, 0.3545520681737594], [2.6338353504749556, 1.4612604392661948, 0.3676485750712599], [3.249811197901999, 1.7523596519601026, 0.16602689208208773], [3.05945685422267, 1.457097756276661, 0.1075713364540243], [2.754436631830633, 0.5342460916808481, 0.37238699484632476], [3.043149765162606, 1.8221469696679513, 0.3914206423911799], [3.3129774631001485, 1.2038413645848762, 0.17267561433516776], [2.6004840888361342, 1.5632293273065894, 0.10456784818743081], [2.8419658520440674, 1.5821531280030683, 0.2895262084423751], [3.3765496502972567, 0.7612993494311331, 0.29749774417616637], [2.773231120991777, 2.1940670589091464, 0.19768846512490967], [3.342519705308227, 2.08791207111378, 0.3490796156313847], [3.353039248626525, 1.4348539078294464, 0.3761040695129315], [2.7768574378200683, 1.5659382983405385, 0.17058297133124062], [3.160055494746403, 1.4021157214292503, 0.2803411934662887], [3.1691429300223883, 1.3877937348799967, 0.395732034821226], [2.65369722295287, 1.4854856202048796, 0.19791717852046414], [3.123300028419076, 0.6564937002033245, 0.18630310913838366], [3.174038842291205, 0.9415370877631095, 0.3250790852184131], [2.7305351124219532, 1.8128795421643893, 0.15163457924321241], [3.324415987051662, 0.6723923621297849, 0.1638424619788013], [2.619192765911721, 1.959409293454052, 0.26020168691701095], [2.8637519584231304, 0.5275825036560864, 0.3479767311284806], [2.6369235246441645, 0.9632773785904192, 0.10900150658022803], [2.7025500109010667, 1.3627822029205785, 0.180708772330491], [2.911768949102108, 1.8259205895662585, 0.1445803691081863], [2.637916986815079, 1.7137869090805105, 0.19192332838672016], [3.290744658310625, 0.7593197884494859, 0.19718598000230803], [3.1426259524913287, 0.60543811626552, 0.11403524158512757], [3.349658002383843, 1.2238085173223423, 0.22942099687930162], [2.6971862379539724, 2.1403305188938337, 0.10203458455155462], [2.9561403391402274, 0.41384255683326526, 0.3604470482579937], [2.811290052110027, 2.034856238717212, 0.3656594104980524], [3.155540881468716, 1.4513043698393846, 0.20487622336795785], [2.9628495260314294, 1.6727705412271026, 0.3070826169101728], [3.329244643358466, 1.1748247755338521, 0.3929718805150191], [3.040006512109275, 1.5947142869583368, 0.21238881942218804], [2.656742157487936, 2.0374064159728134, 0.1084476092328977], [2.844008743529227, 0.9856341738371155, 0.24406375592875337], [2.7871117442702267, 1.4492298840363285, 0.3129262743450031], [2.7881500530448884, 1.2008917329628146, 0.2080055746588344], [2.898178172192754, 0.9460072050284292, 0.16122470450884954], [3.398301300095471, 1.0421944359030375, 0.21261672480993637], [2.9025154524508965, 2.0138943969113288, 0.25279034320639904], [3.1098496761884977, 1.4279779285054321, 0.187725832251873], [2.706127717211135, 1.5482282039619786, 0.20788816937452453], [3.2102716950763344, 1.727324541555707, 0.2659028863510333], [3.3605417101907333, 1.1099740171001606, 0.12457247638102104], [2.6925599092332537, 1.8301884373172692, 0.2993479671120969], [2.974759834744562, 0.5136234680091318, 0.2165041386327359], [3.1810320162965304, 1.9102179121198353, 0.2864016610771758], [2.7051495000753896, 1.2661233118904442, 0.1461806914544176], [3.3168914060654813, 1.2511622665711548, 0.326002023267219], [3.0580947413114594, 1.9229948689951886, 0.229355456867346], [3.0915515597860934, 1.540112810996999, 0.18496252947904984], [2.8913694847307454, 0.6344996378605722, 0.3030391657176803], [3.301788129662517, 0.7671536940642396, 0.29390315626975827], [3.2894697895093694, 0.6523031498614071, 0.20071281263845797], [2.731454097755442, 0.7326516896270244, 0.2501123521058167], [2.8990539097865833, 1.7964298915381103, 0.11097768571256629], [2.7289368942586414, 0.5934111987530664, 0.14124770705690678], [3.0794121180974487, 1.148307824128743, 0.20977389203244434], [3.3961884361420958, 1.979265063980886, 0.26538897621134605], [2.966158195604909, 1.3177442762012883, 0.22045864338446441], [3.3481058878642607, 2.0066371678223582, 0.1304561175013589], [3.1347294297270154, 0.5225868135735084, 0.2789575434707662], [2.712944948046301, 1.9933905022274063, 0.3314556987953171], [3.2023668767868934, 1.344471521820132, 0.2721889386824265], [2.675003714753772, 0.828623567486181, 0.30094323407517126], [3.07016800823993, 1.1145933434032997, 0.10997165844098518], [2.7259786252427234, 0.736371803843519, 0.21810547656554946], [2.7101662074971955, 1.5153116361237595, 0.20037722334754426], [3.1455141200035666, 0.5874728801778032, 0.3495996875201456], [2.9785694787766785, 0.5389990603440881, 0.1985556995710262], [2.6761683069375115, 1.0924439149677783, 0.17912130687320454], [3.336702435780488, 1.9126773170750284, 0.32989970043334366], [3.0358553423677264, 1.948326418359222, 0.3849517136596483], [2.8198626114161556, 0.7654501302712917, 0.2902003626923618], [2.931490569359963, 0.6869755824438515, 0.380049734438778], [3.3896098747223116, 0.5433448525107957, 0.30382674587854464], [3.2612395455966636, 0.9305333947282306, 0.31758473456646286], [3.0857288027469134, 1.7407787783608728, 0.13827563820743632], [2.8239826333650297, 1.6602863479904209, 0.32338770700082475], [3.092884835446688, 0.5026403997936424, 0.17737766306615652], [3.0520454630813028, 0.966822952083563, 0.10012904121712042], [2.8366456604167762, 0.4523395499333618, 0.20317732076975673], [2.8733896119232867, 1.6098033276636547, 0.3670196958267671], [2.9067969977227355, 1.5453551958213334, 0.24258467278767998], [3.326127231170676, 0.7767806202658506, 0.23748755953791412], [2.7657382792827216, 0.46441572828815925, 0.21956582525778273], [2.9994552334245785, 0.8418575604220546, 0.16158740258649795], [3.242871399662751, 0.4996500173612006, 0.19132706696093194], [3.1673285545415584, 0.7492725367630467, 0.35838370810305864], [3.237463278089889, 1.3964319575615496, 0.10603067859232682], [3.278408500646685, 2.199922641026858, 0.39710654685966806], [3.095898748183248, 2.1677179888302316, 0.13883727061614423], [2.6855912943480744, 1.2658156139456287, 0.16351214236848233], [2.832045569176697, 2.1515988614651973, 0.3860210794925082], [3.0089361877309697, 0.702087867834978, 0.38708919329161184], [3.212779477743891, 1.691011336819543, 0.29003546764293203], [2.7667486266937877, 2.0830245255696407, 0.3642592575565883], [2.9737800139813633, 1.5787140393154937, 0.24154006449567875], [3.064752701626087, 1.1288513718612134, 0.31422860999194946], [3.0158393233655008, 1.3022985976998818, 0.2753211862778776], [3.3792438242231326, 1.3913676251134588, 0.1829844046006696], [2.9902189356239623, 0.4205915031094763, 0.18368255152477014], [3.2309817243046375, 0.8535005917221301, 0.10417032773218213], [2.927792457542967, 0.47134485337243454, 0.17862127673227135], [3.0885658774529277, 0.44223393590676335, 0.20124128506085276], [3.3064114361191104, 1.078728816884715, 0.2377832783244595], [3.234537179279787, 1.1555100056249872, 0.20649578351199926], [3.1863356802977663, 1.7100991654714932, 0.2636830288006502], [3.274884006572666, 1.2100025518754, 0.30482013807508673], [2.8073660552555326, 1.0574373300009787, 0.3034965802643592], [2.6084194246573946, 0.49192947470209464, 0.10507226713258004], [2.6879769586352196, 0.4462349281800286, 0.2798364181498755], [3.3201923748309925, 0.7538899279444744, 0.37071869463798346], [3.1195959320993967, 1.650809057124646, 0.2316975115868477], [2.7488501363369133, 1.1303213358210897, 0.3565541764109712], [3.1448949736458647, 1.3160231509744686, 0.32974099638340737], [2.751637369590485, 1.8876141643622264, 0.30101084821473756], [2.7465152452746584, 2.0905348785351627, 0.14057911022194014], [3.0289127268315137, 0.6928855409927386, 0.14475000926362783], [2.856778003247276, 1.53752235433841, 0.14864351408017776], [3.3780950180002014, 0.8435465762349446, 0.25374222843018157], [2.8067840460276687, 0.6950799141935068, 0.22528251867221719], [2.777039998093504, 1.3054229476172665, 0.21091566416933066], [2.847037606282753, 2.107329846719916, 0.3692039956156662], [3.017659890672959, 0.7304945766479217, 0.12716997705256033], [3.069935352820984, 0.8677495134799978, 0.192967786703576], [3.354859300536093, 1.4362006624618542, 0.3993622475718741], [3.096636443475491, 0.8996205586239467, 0.14368212232287003], [3.2775171745080374, 2.1233793586383976, 0.245639348969677], [2.91980928703438, 0.4680132740598478, 0.11457664786485651], [2.9331233712372575, 0.6913309463873091, 0.2942551211560369], [3.17219082832736, 1.0650947162497384, 0.17530866341491355], [2.7267865285260884, 1.3716053403795376, 0.30892448550983986], [3.183845441129628, 1.855634653187899, 0.14650383004299936], [2.7942297001709155, 0.6268977679661113, 0.21354003487731416], [2.6451012315866436, 1.0241742039793882, 0.20684735569874269], [3.018413128568154, 0.7621406517948264, 0.11584828774364611], [3.258577544030496, 0.657787648283162, 0.2931072354882383], [2.9593069445268503, 1.8689338978867025, 0.19405230750934932], [3.08683135181448, 2.183833580833856, 0.399173289699851], [3.022656075158088, 1.0158503190647457, 0.1868902345050449], [3.3821768470751676, 0.6757458087904856, 0.1370431984941094], [3.1525712920198314, 1.4922089977377637, 0.3130483005262624], [2.868775021578379, 0.6891818848793585, 0.2965481261958721], [3.3136046343709684, 1.692309857092981, 0.13608596604674558], [3.1807516546433563, 0.7527599662595188, 0.14173576311602037], [2.949557516418806, 1.443503840768179, 0.2489030190668643], [2.986155722080115, 1.9253579074276008, 0.11723574306711668], [2.93515187699216, 0.737732666512912, 0.3180891836726655], [3.2675840237112515, 0.8250301139166831, 0.35891759982031257], [3.107780493058989, 0.6515448371869244, 0.2518059985420421], [2.938123773956364, 1.5568875025372715, 0.3430756270793335], [3.0007955362899965, 0.8158716872023319, 0.26225573189531604], [2.747662477216765, 1.8763014005439762, 0.29103438792069886], [3.1177095768216376, 1.6360066994708302, 0.22286600801630413], [2.8679339916156286, 1.9303664115259953, 0.19875611198202048], [3.2910983594599417, 1.3597465348783695, 0.21522417624340442], [3.2061740040372775, 0.9873438186906551, 0.3727591349318904], [2.8433612188953576, 1.1182744833010514, 0.35531398759721977], [3.341371604764458, 0.4222297577546399, 0.2652327365463729], [2.9080174397618106, 0.932800563509445, 0.2322779399530848], [3.1063019616476244, 1.7869352122993463, 0.17927599414334588], [3.0510741318949455, 1.905077688038699, 0.17831184911713216], [3.1317844237340697, 1.8211895903108917, 0.23575462371246358], [3.3224342158785847, 1.38651480439408, 0.38794669090838274], [2.7140819644817715, 1.28643051538438, 0.35819982054638966], [2.761723648431145, 2.1337156546338085, 0.337872070037888], [2.796025618849711, 1.6442796559595658, 0.130865283887879], [3.050142610704224, 1.6457411708290037, 0.22975900127354462], [3.205694271023404, 1.5852083812747357, 0.39093270333296104], [3.120746150709226, 1.0066385225208923, 0.2242654083543326], [2.8903638579647346, 0.4752819001647717, 0.3997831722060179], [3.368307401564094, 0.40862517934259907, 0.12930933280215404], [2.9988524101809824, 0.9442696426236654, 0.14277970949740815], [2.82217707482122, 1.077734707743948, 0.3396715542665167], [3.129556228251336, 1.2588345049920875, 0.19417205389347883], [3.356664407866641, 0.4238377501674221, 0.27479999605936467], [2.88641450676961, 2.0266219214316186, 0.35981095965731336], [2.813935380682643, 2.0650204850875937, 0.3945128628635276], [3.3774090614976724, 2.1005431214151433, 0.20296711879121793], [2.7645179367277612, 2.105833068780505, 0.3028618665021825], [3.125827675261861, 1.45413931659311, 0.3978300941816477], [3.296357716574761, 1.4709217117009366, 0.19506146404348357], [2.884479485490875, 1.3389624796765665, 0.27937339181704934], [2.670910906132427, 2.1496563463720153, 0.12609937321633363], [3.209824626362789, 2.0505826673715064, 0.16595282838487993], [2.752154070498554, 2.0704803078536407, 0.19601454352866104], [3.2458464243068734, 0.7084523475577176, 0.238310491135755], [2.92062131251185, 1.5952123373389326, 0.2927491687689059], [2.798955464002235, 1.2600687823974268, 0.21841370828695497], [2.8710229074922067, 1.6779941070343358, 0.39685071814936057], [2.964198480175938, 1.7048499910494586, 0.15229993502902733], [3.034131709543685, 1.3076004628611237, 0.14266255756474178], [2.7368858317888556, 1.8996655396204905, 0.3734566653254988], [2.905364126013332, 1.634053484111776, 0.1665190054982942], [2.801815762194763, 1.6277342207746974, 0.34886575537979336], [2.7993845157292183, 1.0595270013277132, 0.13292376515949988], [3.0233223457544227, 1.8255765110873408, 0.28728950301576855], [2.750367371467371, 1.715821295704862, 0.15768660704444848], [3.0986609323825105, 1.4909396553385144, 0.1319720365243671], [3.2790158195248638, 1.3143666543240746, 0.2257776764925139], [3.1388329702183477, 1.1591808669753114, 0.2823048606459345], [2.7114831875053724, 0.6295065110529916, 0.28561745882046896], [2.9713531975907737, 1.1785821631504843, 0.36275749937014545], [3.311488681110479, 1.812121295564413, 0.10968007400741352], [2.7912927017923153, 0.7708197669917383, 0.3984880850176744], [2.8302836286768294, 0.9693116718584127, 0.22252088522156616], [2.6620468588111335, 1.395425001536806, 0.3043989462432689], [2.929615270247167, 1.5892117588217503, 0.3458704799143467], [3.2132043700669306, 0.8215589726912467, 0.16260603190817968], [3.24342989018499, 1.4259559179679364, 0.10327544901139742], [2.7536410355178074, 1.038965482996093, 0.2639459008976851], [2.8952064996507954, 0.646747633735109, 0.22834298430551744], [2.733657926637004, 0.47986804247503695, 0.3949211684974696], [2.772814098622526, 1.5759866774053035, 0.3227458336456742], [2.703212570349235, 0.9148832060849417, 0.26688653211738156], [3.2230788824635495, 1.1632308258578299, 0.2619924960167884], [3.0330896424722464, 2.1266602875252865, 0.31868904825535005], [2.983544441667032, 1.7709254545984225, 0.11321873460438643], [2.6742989198773337, 1.997738731929007, 0.3418692859099801], [3.027814406582183, 1.8409712271619123, 0.27417766552119965], [3.2403208767649696, 0.565174877102718, 0.31520527893904354], [2.9079498970419104, 2.1745492396595028, 0.38838027269244957], [3.3741498151944467, 2.1093028288906397, 0.36236921140788325], [2.6918850357679642, 2.142852081578175, 0.2835141576257296], [2.8156436678891974, 1.7236538763324072, 0.20446807846212994], [3.168030266835923, 1.5369296123099854, 0.34496144284831975], [3.039381932663576, 0.834720405848155, 0.2858269270693], [3.386720680470741, 1.373698405851496, 0.1470184908783866], [2.6052828753263673, 1.617727294390432, 0.19993795879309634], [2.6630675692131174, 2.171576876449271, 0.10528272234446469], [2.6470027299515877, 1.6658942685216422, 0.15483018527668427], [2.9487508859634497, 1.5207212745898826, 0.29961202254120795], [3.347687122131943, 1.668720189158658, 0.31403067428186926], [3.397596077707199, 1.5033016371013197, 0.18947081617952566], [3.2479692686547064, 0.9372446181054332, 0.1219430759411025], [2.7580908447406047, 1.1824090796020656, 0.3110075140833296], [2.6090905612582747, 1.8739454310998842, 0.3797284739165696], [2.774121424701465, 1.0336991003605047, 0.29525299513492165], [3.284097282139877, 1.8801649784716223, 0.1162665305590403], [3.0066042043187715, 0.7328628256076014, 0.3096296668422252], [2.84896051213028, 2.0205575888667724, 0.3154999681689431], [3.1409061760034866, 1.9342364753153996, 0.2577589760830109], [3.219538386690865, 1.601325460805119, 0.17331034064146766], [2.7397048061353564, 1.2858453737802402, 0.12654508556595032], [2.7080786597756528, 1.1958616659048722, 0.2610651520963986], [2.892764315240435, 1.6147405225541493, 0.14964984106850132], [2.8205656123866274, 0.6854023350662459, 0.34437752731033205], [3.363449528244825, 1.1930811469791864, 0.24706944339484366], [3.3599881185486673, 1.1233659187649534, 0.127377530267478], [2.6241209915306456, 0.9264490068815849, 0.2495572845475156], [2.640697527099411, 1.533163395190495, 0.1412736585024152], [2.8088803602826538, 1.2898217613822203, 0.24300706652087722], [2.63421959556513, 0.7723672158473065, 0.3195664844481836], [2.9915596536254068, 1.0050124123512385, 0.2362061589295115], [3.2386602437331256, 1.6564462662953294, 0.12488343480583927], [3.0607913857624305, 1.3357354722007209, 0.37714651571142144], [2.9446901262230285, 0.8955097889060752, 0.23118503866358942], [2.74114994373255, 2.084819686594557, 0.25871430780802507], [2.829922024719802, 1.1562545639409265, 0.2141181181989584], [2.632519433431705, 1.1657675452774832, 0.3549568524840303], [3.292385976422444, 0.5735528469900277, 0.237070336917358], [2.667226054845049, 1.523750196431273, 0.35405352500800114], [3.3355693440272884, 1.8892313692166742, 0.38679838703788705], [2.8971945002089257, 0.9703306281083954, 0.190455656654147], [2.9361505779802695, 0.6974133453709424, 0.3887870886997441], [2.652354561008152, 1.5420445695703704, 0.3695787493879831], [3.1657506257866457, 0.9930284748397602, 0.2558596685225903], [3.010269450995067, 2.040194310255401, 0.22756861712942314], [2.94608572446937, 0.7077231712225702, 0.3197599723460277], [3.1664667293940205, 0.6769032461663074, 0.33197904945916207], [3.1503213338133524, 1.0202954681580738, 0.22672940244192652], [3.026497315572118, 0.7459306176492301, 0.11164312027737966], [3.0993345557377183, 0.9981145060694157, 0.2192113649503397], [2.992402828689252, 0.8030470274825183, 0.27163565474676216], [3.1993090860475673, 1.0528921095176078, 0.21195137154602073], [3.074522397539527, 2.166134984734103, 0.15600258406508294], [2.727909043015887, 1.6225451673149072, 0.37390544784371893], [2.6647914506463914, 0.6992052900708559, 0.10715033649217931], [3.293103768832311, 1.3563005171809404, 0.20252470492415628], [2.9679002141461983, 1.5524441983875223, 0.16280256632091394], [2.968844121426069, 0.7881088121085622, 0.27630263330376237], [2.961245829559928, 0.6589858169363915, 0.3750049798708256], [2.7833808986022963, 0.6111460696019284, 0.2341261572644582], [3.148892274105768, 1.6589994018193182, 0.29891575126958564], [2.75738443252312, 1.2024074920621521, 0.3560041001457058], [3.1249131861260864, 2.157965634965869, 0.20947978698240904], [2.8266488182478833, 2.089625181791749, 0.2937828748429352], [3.0662143440920504, 1.7076065225797308, 0.297640838594631], [2.618646469196058, 1.7891818130244517, 0.25740547362938326], [3.003843981996648, 1.9627853553769632, 0.1269003603814112], [2.837687083923168, 0.43416659980198047, 0.2550061442920295], [3.3099556087141515, 1.1032926396169447, 0.21384111432701247], [3.118744029266874, 0.6718522114546979, 0.24351985757524033], [2.7492254678834827, 0.8696837049258462, 0.11844537000299316], [3.0532079288855756, 1.2386812534019664, 0.34121958399337904], [2.910713385737051, 0.45355676926734556, 0.3529267085239688], [3.1473575600105983, 0.8785575159166803, 0.1532953012536527], [2.6392466149504368, 0.8560736826795996, 0.10651883699389156], [2.6421131960564876, 2.1554105092664093, 0.15211407795982596], [2.707734575791827, 2.181642163339251, 0.1215223989835734], [2.8858069481433466, 0.4691953384791107, 0.18122498686081745], [3.2868629689843263, 1.836641852193432, 0.39253979856213406], [3.2836700906076772, 2.0987703465290615, 0.30058446671080286], [3.108499187605423, 0.9792893140304377, 0.35054229313195473], [3.0205133769024792, 1.9544173235969011, 0.3293495887830224], [3.3556227440566375, 1.8549132870943692, 0.2840972587489031], [2.6587602469749827, 2.0449535959300187, 0.19024414804878148], [3.3253029192465418, 1.639180121910997, 0.35007743207021347], [2.695526634549284, 1.9032993468794146, 0.2778412645700895], [3.110628720753884, 1.6163767477366198, 0.300154711211871], [3.0688884527278253, 1.2943625423274083, 0.3759014488103328], [2.87968041393095, 0.49707072496442956, 0.17277905290852907], [2.6737414686369707, 1.0870080162914637, 0.13306509650696738], [2.9134970277993437, 1.8718133376147026, 0.3768319394984606], [2.781940873019381, 1.4990814747442767, 0.16490649805853613], [3.103910850636084, 0.5897940042193062, 0.33050689288845925], [3.3725256964696873, 2.0180134858496896, 0.2918984123382718], [2.809900240538617, 1.4301955195806133, 0.2983673046699325], [2.9219494805072634, 1.3779580047442266, 0.22707527209542555], [2.8319026863785974, 1.5877435032962315, 0.23257047705861206], [3.218066449737179, 2.0538298093935192, 0.13720508340469098], [2.7630747339274757, 2.069408521679537, 0.14398267176786073], [3.1325918030793463, 1.9172933841417306, 0.26880018276969886], [2.79241256290522, 0.5738526314578993, 0.2830927122702157], [2.994529708708419, 1.8427260503728446, 0.14504362612913624], [2.6284861686000527, 1.9860485922105775, 0.3536329013635957], [2.6936544996942047, 1.6997486134309416, 0.2933634724323123], [3.196990255295904, 0.633308393160338, 0.1399454982326886], [2.6111475714175256, 0.8316863329926147, 0.2565596062478661], [3.1732933810845565, 1.2774916254927169, 0.23371514226175788], [3.3738140952323956, 1.8858983216116, 0.33640182074830655], [2.8128223929292377, 1.0121936981885098, 0.35542458850043457], [2.771273873707371, 2.0093664782222005, 0.3386622577522552], [3.200279923194236, 1.5528673908884554, 0.25037971355532757], [2.872417347960851, 1.3436639858264212, 0.3083074190404973], [3.1821009692495332, 0.8455801812110386, 0.34665082112944867], [2.7590071217306114, 1.814815006368768, 0.13570833520268927], [3.393844064522674, 1.419131351697121, 0.11525395228608344], [3.2246392270218798, 1.5805354945062944, 0.3819155346179338], [2.821799239060509, 1.6528254021414495, 0.14213574699998188], [3.250310449094525, 1.0668228407101346, 0.17369528518263094], [2.601187662368441, 1.1385184953559813, 0.19907837984917803], [2.8341868647645434, 1.8467487964343796, 0.3722090864510271], [3.1715599790882116, 1.7180437917436955, 0.36491103844283523], [2.7408967541201004, 0.596330009470005, 0.18263512578478502], [3.2042608142525624, 1.573180225705495, 0.2683499769512996], [2.6896467274509024, 0.6237726591145143, 0.15131319390358147], [3.1919532725140876, 1.3251097772924583, 0.3606541750730481], [3.1364220665876523, 2.1911025606457772, 0.2307553421820517], [3.271062598242471, 0.8730873968484606, 0.2769821230111772], [2.930760215659802, 0.8901057013221645, 0.32452105932175035], [2.963407679899913, 0.6820527750074395, 0.250967469446019], [3.3047722114508526, 1.6801391748449546, 0.10582669097885901], [2.631242309732132, 0.95744973242053, 0.2785209441344598], [3.143628130714185, 0.4615921845834373, 0.12835789870757877], [2.7353325860000663, 2.0742068940038987, 0.36638788292416713], [2.9537994212800736, 1.9688754522681076, 0.129627781371563], [3.011281611785447, 1.5187859292925738, 0.27353332610748704], [2.849644105547668, 1.7510896042742214, 0.3876069404831822], [3.3150289165932074, 0.4180524693482469, 0.1813902157621703], [2.616376159562057, 0.48436452570515254, 0.3117988068499279], [3.2532831844370915, 0.6158622805787729, 0.37930849988617654], [2.8000311362409387, 0.9546677130061556, 0.38441683387350023], [3.041294759079288, 0.6481351587405538, 0.3852121182603928], [3.159700661508865, 0.5470499284722795, 0.14599220469153945], [2.8650377058212224, 2.1442717325259366, 0.3079984058405749], [2.6025203712067007, 1.2169162573808956, 0.18012859910582285], [2.7784869797410074, 2.011341598116537, 0.16219533769033762], [3.0786595206484004, 0.7816715543843994, 0.18527216205253602], [3.2697987011622693, 0.9732218018523373, 0.1896439262453633], [2.678057735254891, 1.6401938238411777, 0.290832981845432], [2.625020903866201, 1.8975423053446197, 0.2730526137202275], [2.775096307097189, 0.85464510106633, 0.14028752857319096], [3.340343416250433, 0.5311296087728155, 0.21452713288284425], [3.025561836069029, 1.9185909814818365, 0.31721675958173734], [3.371660083808539, 1.7611764241641437, 0.22516722194590594], [3.1794315292279856, 1.0447446336358435, 0.19662127390748307], [3.0317987755244094, 1.7785239106158481, 0.2818677758004313], [2.833905714017572, 1.053298607113497, 0.26725981532594345], [3.2461363688870692, 1.3617913705376459, 0.2579014289746444], [2.6488098615900446, 1.7210163257183995, 0.11267705813107629], [2.760970881178256, 1.805440821411004, 0.33743386588144586], [2.6541229060389617, 0.7920954817162866, 0.18462002210999223], [3.102292452705853, 1.8333795041241923, 0.23480464112894933], [2.9010544234161273, 1.0948387126971793, 0.36006588440704346], [3.1532866814620273, 0.5581051155383687, 0.31661671257642665], [3.2628131060930867, 0.7474993290452989, 0.19485981870241986], [3.380069242865689, 2.1298493615953697, 0.3923250321803733], [3.1773557763670772, 1.85274727554431, 0.28511805322437445], [3.319231770442043, 1.6247803301451853, 0.1456242743837308], [2.6835200603560443, 0.99711904236143, 0.18728117821151752], [2.6880234843069926, 1.2092288700126006, 0.15281323820052178]]
    y_data = [[50.75564486291919, 1.1677102206076155], [71.57153650858518, 1.3010283336738053], [16.790334693056458, 0.2379979913220611], [1.6855296399921578, 0.2741314792388195], [2.0296430492146755, 0.03644864262497309], [0.34991689724612396, 0.19673402095368173], [3.351107084427339, 0.22610535397248913], [60.84488804338064, 1.1080923826519817], [0.9933875502334821, 0.00913627162117362], [98.32450041089082, 1.8655588770429938], [1.052498204938612, 0.165327845085125], [212.51587886683163, 3.801366267787137], [3.1766423742861654, 0.02617705284557773], [22.783445299938077, 0.4923842690423876], [26.387521540058177, 0.6018752007813993], [214.94356690611647, 3.6792485797034047], [1.038446234136835, 0.008765046873728434], [20.980354824345298, 0.6177365199618529], [10.960001821829678, 0.10951775132768353], [12.565140228750034, 0.14755123045082755], [0.49839639667350133, 0.10530393626346671], [31.723352579404377, 0.5923277584404859], [138.72583310617907, 2.688528832456051], [62.830311985692894, 1.03069485586954], [1.371460605151943, 0.08399132511900233], [6.858075063697748, 0.034181220895663236], [0.0914162787868594, 0.23273956948047148], [44.24330955524184, 1.1153500527925497], [0.36814565228325924, 0.3301081566625561], [0.4096509801219424, 0.23310974501491305], [17.008764606825487, 0.5629793946313547], [14.468801153523799, 0.27123106311314804], [2.5013443718896493, 0.015468547995334533], [4.607596638884766, 0.12785899750289387], [247.9870942333038, 3.6807341871743136], [8.16208712940437, 0.2741402904727262], [3.2419421773076285, 0.01370501560472101], [71.21104243948031, 1.5049144703468849], [0.7058496395795534, 0.12667361922749165], [34.465237432913774, 0.8068502610548307], [2.987585860988964, 0.14237896242194606], [2.4892563324816535, 0.271598345928958], [2.1011797794732585, 0.09419941033198684], [9.410166629313672, 0.1674271598484589], [9.374483743636782, 0.38259114316944975], [0.0513855063194504, 0.24917029411627287], [3.0489178540305004, 0.09607535972685831], [0.08960523371042128, 0.28468876129328485], [4.507345195597657, 0.04305410951843361], [28.683619506945057, 0.6600496052434475], [0.2823203907164128, 0.11656833488186598], [9.43229879708253, 0.26952442036336843], [11.668593914568145, 0.40205859917366077], [30.447542961397403, 0.7108476288363365], [0.2166981799737264, 0.013973465869305696], [0.540637841517286, 0.10877821381714343], [0.3929337958940951, 0.08931251925186531], [0.5879088978214597, 0.24995170761440844], [8.821537349872266, 0.06827064271067812], [10.168353661480932, 0.421432480152808], [155.94584028629458, 2.7246552956584393], [38.98633789528274, 0.7500108335509502], [2.756567786969026, 0.2654095471068773], [42.0612617117673, 0.8459665445815083], [0.32154033277013533, 0.06877489581350502], [107.68032087910747, 1.8414167108211217], [1.5493110822299683, 0.20340720386288133], [45.29516372767846, 1.041823771782406], [8.29220316331732, 0.23942592189268586], [31.46318928346929, 0.5438463386295189], [1.5932651195003638, 0.0020740048885699775], [7.49801968850468, 0.3247631870372764], [70.19054340532723, 1.5437774619290534], [37.799049802896306, 0.8741090436804086], [6.298689701122989, 0.1726763851298074], [141.812830299004, 2.4978884019403127], [8.35030001620091, 0.08156853932539539], [186.78869498511472, 3.460804829598055], [3.2749151732252804, 0.017938174307723508], [10.391913029256463, 0.07588674744574775], [2.0687283675417043, 0.09111548119715221], [8.133959170043672, 0.2613122321848374], [40.792186737774685, 0.9860607242466923], [0.6346390461804176, 0.10035227390613093], [2.294271428220463, 0.006082616126756204], [1.6748437631758806, 0.07230626558314611], [0.2898079340838944, 0.04122874393431494], [0.1831799282747042, 0.15428520625058362], [3.2995249878514543, 0.009851170149524675], [12.164350553507173, 0.22245997722407992], [2.7348296859711825, 0.07974152598375926], [32.356877353383254, 0.792576416577211], [3.6403854000352345, 0.19779854861381335], [0.8187449494653956, 0.0947599456362486], [1.480410167970328, 0.18489744131079267], [22.69454494211357, 0.39957115953956795], [33.15808568432829, 0.8369620327892936], [3.9105262561931955, 0.2822041875754056], [0.6315445416730224, 0.03673948248138246], [5.720894978256355, 0.10457979568349007], [9.892979312338449, 0.32890242558102856], [94.33265542111431, 1.6479386711704025], [9.872143897882474, 0.26206810021196586], [0.42717299930523867, 0.17774754892458203], [9.133187853164225, 0.10529986682616671], [75.75019060231381, 1.5872379372940577], [0.7002834349552906, 0.08178891561269247], [9.062699981735603, 0.13920795906809105], [6.561778858923112, 0.29012475748920824], [25.128426416578847, 0.4593204670768435], [0.07603481709639864, 0.13687708714466923], [7.148573461533771, 0.19547957705630692], [30.279113008358827, 0.688821557198306], [45.2276175716504, 0.9638962754491256], [73.39313568158875, 1.5440120430205395], [10.314402756770319, 0.370249684759846], [0.06893437705852784, 0.20067204938884178], [5.555305407719345, 0.15927032265134156], [2.6497216918185322, 0.18862355851495333], [0.014691148237368994, 0.3050901745911102], [0.8471070033153768, 0.008563915047180156], [1.1755222855282281, 0.048470955200451776], [16.733696177858217, 0.5021399191989051], [7.37721285939477, 0.0020334364152878523], [4.374746864053483, 0.19494260383192286], [0.16336149504136968, 0.23836245506108933], [0.13417042819578942, 0.1123490085904229], [2.289511695996564, 0.013474269988875753], [137.23016220427516, 2.676129088814319], [0.05442859060530786, 0.19825218513100346], [1.8026758689160929, 0.025902882556049858], [0.8038784962463061, 0.13188211550810636], [4.268452413538055, 0.08593035250395188], [0.957079437181291, 0.050639207331216005], [0.045701779254845074, 0.13447197744294803], [1.0130407875138632, 0.004804796281383435], [5.115857795643416, 0.12872573749465413], [5.622210185921112, 0.16750490268066162], [3.17425291616062, 0.022298811229779322], [0.25117156206439273, 0.15582212484438188], [6.610807194390977, 0.12306420285079694], [5.239428095322061, 0.09802696098732766], [0.06860846415371824, 0.09781629961007701], [172.53550193238573, 2.86046667645844], [0.503918581262868, 0.03636553945227398], [1.6605308410036719, 0.04686145783900652], [119.82625920799289, 2.289235467574056], [9.624174385003377, 0.03218281500186672], [3.7037340710799787, 0.28129285338582866], [29.196220033269114, 0.7774515945867194], [1.7993896229212543, 0.14814320203615128], [0.19608549970051914, 0.10568513911682367], [6.089684328748387, 0.202515355369269], [138.20069623660322, 2.7778379092479155], [23.119843964155383, 0.45385273943802995], [6.075153968042852, 0.2669963571150296], [78.03567476943141, 1.605128664489291], [0.19302633040941702, 0.016456330011447935], [8.747040837001357, 0.030066406897130773], [2.842985709108136, 0.1489068385870859], [267.84183810761584, 4.056842830558708], [89.35686845744536, 1.7726118990488078], [20.06077099847978, 0.63433495226767], [141.14732013693967, 2.645991570124097], [61.39276247912368, 1.3230809518910696], [127.06879296067414, 2.296380529466568], [0.5757316370101734, 0.15785372788647975], [51.10507052692986, 1.2414426142398023], [3.1311588207305263, 0.00778501727164901], [24.264764190745407, 0.7054710196153654], [0.6650909855807463, 0.07457128488154494], [1.5824940588211986, 0.08674853473172585], [0.7468963869776478, 0.17478995806304762], [33.74661878528232, 0.7893456555135869], [27.836554608640608, 0.5385879406413966], [94.67663586763226, 1.9530227975124157], [0.34995559998787606, 0.12609793391782212], [14.050980669549501, 0.21538885249005887], [3.88767047707045, 0.18585818213593777], [0.967021629140309, 0.03668198579177956], [0.04885018702938654, 0.13335273968149425], [1.8848588990558344, 0.242328886356542], [11.682167680501276, 0.3231378964677521], [19.556472008448047, 0.5434004993058682], [3.3677836150982805, 0.07381834707026515], [17.45983811322598, 0.2764897160716429], [35.8853984710815, 0.706366359162469], [116.43888097384078, 2.14983273211096], [1.0292451755727776, 0.29633173306660604], [29.26106213823878, 0.6708798020004564], [1.339303328732898, 0.08421154449248927], [0.677401448392551, 0.090073590407323], [0.39759542163588674, 0.2731110929462863], [1.5670354831889455, 0.05948165193487654], [5.021902540604371, 0.2235218792091176], [86.18602205223667, 1.5074552206820302], [11.58652776380894, 0.4070020305758193], [1.583782523224483, 0.228407581504464], [22.951484365894736, 0.4704139421657259], [67.32177527629908, 1.4916625319105234], [11.262013044677678, 0.21739868273926014], [14.970588655905177, 0.5047760185520729], [68.12564206167328, 1.2092307815217047], [119.95121093012152, 2.013350762513798], [0.3465161054140767, 0.014288345135617588], [0.9570137472449657, 0.11149282332583185], [5.130041271863622, 0.09650516417963273], [0.4718337248913846, 0.037471173021546504], [5.689993702471477, 0.054648870327645704], [3.379934750479937, 0.22025695303905854], [2.8899889887316297, 0.058426416771823496], [6.551955962355397, 0.2732518057417151], [0.16050571653088447, 0.019954239938130808], [0.09907570036419312, 0.2559961748376952], [0.32578760066578116, 0.007482785571124488], [4.2217383785392135, 0.18029551849053468], [28.033888320457226, 0.6726542235425995], [0.3040440468845048, 0.22023693138725792], [5.364604262149395, 0.18383361109992896], [105.67148106168008, 2.020285773020662], [5.462167272274108, 0.19756413871194803], [0.5355790270586779, 0.10858338787313351], [161.80194587367328, 2.9527423023077284], [0.35639461120167026, 0.059391914823145334], [0.22163177082908145, 0.024271161712777285], [15.408089006349853, 0.5179008909484737], [7.275015037394699, 0.1497738954689981], [43.89252332320651, 1.0301461195860764], [0.31385416395283827, 0.06955065202932639], [8.439363283747458, 0.1952164400902816], [0.8386333731769438, 0.2731520323603495], [16.98841496448504, 0.4331352716328639], [9.250052669241365, 0.24715923868201028], [0.1784799596508146, 0.14644109496378716], [81.16977755570578, 1.3036733783994947], [58.998152741157725, 1.1813649156926158], [0.05099505303618448, 0.16465577093019154], [17.31315498669267, 0.3012830791293654], [0.3621661100409671, 0.027307404670918318], [57.00417870543102, 1.2684172592167549], [2.381615581432749, 0.13408895678920646], [5.293881895026734, 0.037520329420863426], [16.40139273421102, 0.41332764639009484], [1.368298084031328, 0.1933709895854369], [6.1486632451405425, 0.2467435584140339], [1.085350721284107, 0.09612842200854868], [1.8303194768997866, 0.14298592569461116], [0.07710580719783124, 0.08875940764165852], [19.718660252679964, 0.49940728557675285], [32.38349814481296, 0.7642381024136483], [129.99201377016135, 2.0140402012954635], [2.380448037144534, 0.058346565481393206], [0.5477901043456352, 0.14147084346688704], [0.8040979032010899, 0.16153066689874696], [33.29393163889451, 0.7092120506966789], [16.436111020046525, 0.5209277844371624], [142.81096951122612, 2.8252468917340727], [27.11019309599211, 0.39195644227943205], [2.222268519194181, 0.03883073249690268], [149.62097581128333, 2.7727759960617946], [0.09035323328088167, 0.13909614681322557], [143.5537646915505, 2.7490052727418814], [12.934936084836291, 0.2001855549106901], [1.8732812124358154, 0.062030341662800144], [6.026585194253572, 0.011142879970796903], [0.22726776498808982, 0.15830509194005696], [0.8262334942094884, 0.133387157808806], [0.55140396663982, 0.04362405803913606], [94.31046411500496, 1.86446689384307], [75.35650590148046, 1.3449029619658335], [122.1751492833273, 2.5166329949167987], [0.14868649660096736, 0.04207520622453988], [1.0028247070503684, 0.07476555548648768], [1.6209443507715842, 0.29628186032557263], [1.1969424792426384, 0.04941347407629225], [1.4978100653711892, 0.31375644758661503], [22.95519970122086, 0.42254932915110094], [1.8372137633724246, 0.12790043795647754], [9.71420194953056, 0.30052974182135295], [70.35539539137103, 1.5238546870455658], [0.6172011934967758, 0.02386674155442582], [0.7117974571343046, 0.11561020557378365], [1.9340484595480212, 0.10605649924054858], [4.851417270543724, 0.2542001023718382], [26.267752983514512, 0.483888061679381], [53.28698676305834, 1.1784841598510265], [4.292496409494158, 0.057529911415559715], [147.48262914576125, 2.6743093618430973], [45.56566782288534, 0.9064828809969749], [21.73272527617019, 0.4428125667595554], [2.496814343878886, 0.1291861490928635], [3.6128284379148554, 0.1255564970035217], [95.14591379554398, 1.8507892328295932], [1.8523169272169309, 0.04431262468839669], [18.06159276483367, 0.26416788438451044], [0.8132397364276449, 0.006005117217714919], [0.8098191406682643, 0.12988924225587], [127.20178263994686, 2.403173846809216], [31.521435508299152, 0.7075433941040248], [3.683748953209557, 0.039787365364784666], [18.050986352644458, 0.280950891294083], [13.984610443671997, 0.4839989417128404], [0.24692472087181064, 0.15268120939966573], [4.8618517678110615, 0.21011336646184772], [3.8755645877087512, 0.13330655237320663], [34.21730225162104, 0.6347990374101732], [163.62687143895886, 2.748334270331491], [1.2242683597252293, 0.21896412877283725], [1.6615961587734214, 0.12613595331449393], [14.772918085227934, 0.20642780493328572], [1.6113708270334395, 0.2526890374883768], [16.665942941190803, 0.4594489624143998], [140.11669482640949, 2.374238038006787], [33.387564898565074, 0.8237353202338394], [9.515589961268983, 0.0020297744907491738], [1.33537341000043, 0.14165941608648763], [0.15665414872185485, 0.07939993340563675], [14.396317575724881, 0.24687679047296693], [45.8284366747492, 0.8760096243159747], [0.942265893785985, 0.10207224697828028], [4.135753104128745, 0.16199431672388565], [0.07244045168657583, 0.11790417079322835], [82.95076739551428, 1.5286107340053614], [2.2269835569156182, 0.011145016194806331], [1.2898932365700964, 0.2591675694866168], [5.286730273742346, 0.11449818466703594], [0.17139092945026793, 0.1114472794837599], [116.70929795468693, 1.8953408251100534], [0.5820213157631294, 0.09908261936611996], [4.196970467954244, 0.2545360796464521], [0.15487034175281003, 0.047647232714422706], [36.638841077591486, 0.6039267025170595], [30.267238160512896, 0.5779968319420172], [1.2475117181137938, 0.28301360972055567], [0.09594671647123448, 0.09144574253774086], [0.38233000557142527, 0.0886580054371756], [5.752849780126065, 0.09233739800087164], [118.99224944774858, 2.069198335522371], [8.042948955303782, 0.19988575346230286], [83.49306409173177, 1.6030524453177961], [0.10920438840694673, 0.2243501559953943], [121.39943569099869, 2.3731439376138614], [4.163988196011952, 0.15893896762731655], [4.530375370758558, 0.15650695878492493], [0.6159274514202362, 0.18366569421632323], [3.123577540950441, 0.2377640336212443], [165.3484758370496, 3.063095937289615], [0.4431438791578405, 0.039276268010736005], [33.20217658667934, 0.7611600260352364], [13.641067037373398, 0.21168128189002872], [29.64422962949762, 0.7402126532148606], [0.10051242259059837, 0.005115524294994937], [2.2697488040061145, 0.2434577428700116], [0.39591024116383317, 0.002646388936161995], [33.20907124023433, 0.7739059459223124], [2.239725808833618, 0.24096611178481644], [33.427795912949996, 0.6250090666538195], [3.751290993808431, 0.1235408726903172], [3.430718341919814, 0.06932593906079017], [8.08182303035831, 0.06676576397853841], [0.04381781377489214, 0.17642413517376151], [6.43889896611713, 0.017250878983718156], [51.10523707755208, 0.9999480300518102], [63.9801596679541, 1.0806961401412445], [12.91227347002554, 0.07851734674142516], [74.18053109670555, 1.4849535402842016], [3.0820363028784743, 0.10186458716977768], [2.325201500672295, 0.15308912988940743], [77.5730997496769, 1.4960020402805414], [0.0984591640987044, 0.054073125109484296], [0.15985968591688385, 0.04632341272019396], [4.932371645446777, 0.10656059164590273], [0.82390429926546, 0.0975097069554405], [0.10767614300427646, 0.2201096000453308], [0.05581523656392017, 0.13688266468368604], [0.20034338708293126, 0.16163640197857104], [1.1050944078620162, 0.13261714467504324], [38.481989592624956, 0.6685140092799674], [16.854078007751017, 0.5546097575680319], [10.462903490576291, 0.33296091263093486], [53.35319246006847, 1.2357780426488778], [0.44053844610173803, 0.008236741828830043], [0.6601681916285623, 0.0017193566962654392], [10.387143147158953, 0.1590350284350708], [3.105887678144199, 0.0797893202170122], [2.268873146590966, 0.27687598738421515], [309.17768867245775, 4.790823392730086], [4.4291371840672715, 0.0543211536364562], [6.965166691930582, 0.08554941842333476], [10.35753005597607, 0.1279751255895603], [35.450992442268934, 0.9078091767261817], [69.1462576408321, 1.5347390627797886], [18.709888467771506, 0.5159869899587419], [2.8146542650558746, 0.09304226642361317], [0.11876595753171063, 0.027729225729766323], [102.13819170544504, 1.8440959426381087], [0.8796198442159188, 0.07158496950115247], [106.56996956414284, 1.9859330496757772], [9.41073445202175, 0.15772700901375544], [92.20640445375547, 1.6087751561269614], [1.061298269134746, 0.14190871010672368], [2.353898006319206, 0.1380743196429112], [19.015719587466826, 0.34466494383896984], [31.483261678531736, 0.8222677676092962], [26.530938476562476, 0.554830509599394], [11.150477651167106, 0.2344590030579956], [0.6860511666844715, 0.11942293327717417], [0.4720306950767638, 0.17497735641734435], [156.37382362184644, 2.9700080751217794], [1.7579473674213142, 0.08343846138925569], [2.8754432652582453, 0.30511124187495775], [25.29501823931562, 0.5444404973135478], [4.30140843425901, 0.24516973452264718], [16.976905945644212, 0.17827756911053078], [266.6513352752686, 3.9666742103715533], [120.52469807856612, 2.2577466793503302], [0.6359278015931985, 0.1770608059287528], [30.66682648564717, 0.7318020917762432], [92.71049939893028, 1.9610544201567657], [0.6929392557142137, 0.24070470991115106], [2.341668282064012, 0.06687318485777262], [11.374277452412123, 0.40228223863089185], [0.6029101996531616, 0.2240747930199393], [2.2287968289469173, 0.024180748362920848], [20.859861111147108, 0.2536912103838126], [0.5103701267731781, 0.12258472511269781], [7.17713647497975, 0.19854796873847494], [0.03875584999981654, 0.24321750007236534], [1.5468588985421088, 0.10224887741577658], [1.9662802751162554, 0.22397584536119783], [0.2944058253037201, 0.00010082482380006048], [8.019408784151688, 0.03862965373416444], [12.368608995362735, 0.3711616970819182], [0.1985447058150987, 0.29076802997438245], [1.5173889555098414, 0.09990383376216905], [115.18665616462577, 2.301278991183997], [9.38086739639656, 0.23596661606328787], [50.41992413922391, 1.081397232857884], [73.0564418139558, 1.5960222964107944], [12.37481981893208, 0.2612229609317801], [0.2671290661602184, 0.3131315226210497], [0.2653862759219852, 0.1017446740952207], [20.982725477956116, 0.46170036919623797], [0.2543250717412221, 0.1558424273730038], [1.639337104261356, 0.15220349976696437], [4.051658782720478, 0.19376415052786336], [42.9831400942559, 0.8055559361405916], [4.520448864387918, 0.1011477707285747], [2.457501665500335, 0.14293156565028992], [57.92209687308236, 1.132431076985349], [0.4575581092042849, 0.020364519434430035], [33.45481933676853, 0.6338923238312404], [4.192877462756723, 0.09875121851016432], [93.9657885887528, 1.6916155850822288], [1.1370556419823201, 0.25951886917686207], [48.14268809734519, 1.1037043537938642], [37.08397151737146, 0.7053525634912663], [6.060726594591554, 0.12231966436253355], [0.2294319253386415, 0.10495454248660786], [3.1246821505989306, 0.19601064995105796], [1.0023620271013283, 0.015565113416856362], [1.0867862058139595, 0.20952575098222892], [0.09161087494213764, 0.18205014987795137], [8.757690985880792, 0.1287981625623198], [1.8057800864992923, 0.04390669109072917], [120.1720466509508, 1.900451803080929], [4.188768251125558, 0.06995676986687896], [8.86600010617031, 0.16327913037615585], [0.8796748813167409, 0.03983979380019026], [65.74388347660312, 1.3271718975811022], [25.855737851383456, 0.6428787329488195], [0.03636016792781258, 0.20685342216862904], [2.612486152326916, 0.10415409734527667], [2.212794813579503, 0.21350997158821655], [1.5354874860478587, 0.17114013980913925], [0.5311648629194444, 0.06555792602080454], [4.564135996963285, 0.09700377140894781], [0.8928283919407214, 0.26634341288601593], [181.04857432089835, 2.8980419314204977], [177.01163557767214, 3.2495391200282606], [0.04756485183755696, 0.12298519562501642], [1.5705907932034018, 0.05448528762912137], [0.23506290204198585, 0.08339666927877598], [12.328099847676722, 0.4348151698014158], [1.864880829161216, 0.024129773806972968], [42.901774721403704, 0.9363976162416769], [5.713955837505113, 0.06117419759888103], [5.111057740628073, 0.09009698932836138], [7.109904735486126, 0.11425976332390839], [65.3179450892161, 1.457681368276658], [13.207452544195677, 0.2638255272774249], [0.5693217188567605, 0.08199267514887419], [6.544234615922878, 0.14466835292336477], [2.0662698267645303, 0.11169636273692714], [4.067767132257912, 0.03544553310261534], [5.649523847274795, 0.26784873633571615], [0.4394836872141182, 0.06373013210824977], [383.86414946256787, 5.618405283773873], [8.348300383626336, 0.0896855355503377], [0.5051559326615034, 0.2702463949415261], [163.0164063072106, 3.1339713243584257], [3.235829745801312, 0.15250181910407246], [71.4948088541347, 1.3896936417541368], [111.64305294631427, 2.2354856058912946], [17.254529993996282, 0.37121731440120453], [16.21087920509401, 0.4765552806913475], [14.856556943872675, 0.3891979952569515], [8.384774777759914, 0.2473035299577011], [7.838790107948561, 0.06575070544023066], [4.876255853434756, 0.049334140864558354], [7.379237084702867, 0.08643533379480772], [5.76946240442694, 0.05991063724806375], [6.973958771552165, 0.2631909927374484], [3.4920430091380923, 0.1370593694464609], [52.0779049662026, 1.0405008508694977], [31.21765007499485, 0.7908450289770196], [3.9857303078298223, 0.04955060019246191], [13.410521192680433, 0.07053573868314972], [5.0020189896647755, 0.19460117614781608], [11.188398374508509, 0.44205809610659796], [25.30125046104283, 0.5446017419951217], [11.18182766953327, 0.25245827120238806], [42.01769904207322, 1.006418290644641], [43.95438525425701, 0.8727098558413721], [1.46570781670161, 0.20294401499251408], [4.906305606211563, 0.08802376992351528], [0.10874167079897971, 0.16728949856066572], [2.970939007628485, 0.18751398766907437], [1.939778809666717, 0.17353142839737534], [0.9166801450469482, 0.13454860886676145], [139.96437932086434, 2.7060185378998947], [5.777562259575446, 0.08625352886933998], [0.3364186650513388, 0.06802647666733745], [143.48684359507234, 2.624951128763983], [1.9848689792789258, 0.08665694291178087], [96.83733038162835, 1.5713785746223397], [11.539958759296887, 0.03757170340042743], [0.0653864460239048, 0.07041800061943315], [0.16372884373249455, 0.014542913728125063], [13.117331731894401, 0.24049181286715435], [6.4547818455535735, 0.10594311480447104], [3.51921591464419, 0.16963794050602465], [0.5414323040319667, 0.2785258283638098], [6.296976304043139, 0.08277652327342398], [0.5877584763730961, 0.08584371023260272], [13.377658324625559, 0.21233495969531885], [286.08254855574535, 4.7258294395373746], [0.08071621000580383, 0.07540734372743517], [3.5857960783717484, 0.018698416826149655], [54.565066701429984, 1.1875929326024346], [0.10242749889766467, 0.10782582730364054], [3.786927911580552, 0.08586220952540852], [3.321197940352392, 0.05990574549044967], [12.382561168489563, 0.27641181252688835], [0.4505554323100682, 0.12043506369555411], [0.4889416904097052, 0.014625188303233352], [12.662961609116532, 0.4664534793556402], [0.3328725874288466, 0.03759735922295054], [54.27214983857624, 1.2208234605967114], [0.23138794714153924, 0.02431216055755168], [37.98146315918298, 0.737533101623965], [20.881756426793302, 0.4547772916597235], [13.113598285819101, 0.1693854121715981], [12.792156442088023, 0.34930605886878785], [26.620847821839142, 0.7695222778437724], [14.133120741791672, 0.3813784969474503], [2.001304857795865, 0.006692793668075899], [0.08363573915842701, 0.0912112960413898], [11.624852618340272, 0.21835709705300696], [13.004577905217243, 0.2173980691553111], [39.60998781158491, 0.777638310342093], [113.51068424338956, 2.200362408113803], [18.14543511955255, 0.40608875723188137], [95.31040927729025, 1.8720831462978953], [0.07329075822781603, 0.22168182697828775], [20.39697836300735, 0.43686690817629265], [139.64229958421328, 2.640484846274106], [1.2704020656954647, 0.05120828535063598], [0.27522271908734375, 0.10014688642082926], [9.318814862758332, 0.007213374539143178], [2.1478645058270276, 0.11779265953100904], [8.961969806956205, 0.22474364582980158], [2.617839643073617, 0.07091106000093089], [1.5999798873873206, 0.00034459984514613944], [125.53620835977038, 2.4417735609388664], [149.23798936454267, 2.9584712548903584], [62.56184954398679, 1.0127657668908585], [66.10346732662867, 1.2722113740220329], [99.94397107925936, 2.0940618893135605], [11.85350451311259, 0.30555508620736666], [12.202520922788539, 0.28008938315846127], [0.23936394666952568, 0.29014242492776854], [19.93988648074884, 0.34945782421207644], [11.818255158687595, 0.06985806403314422], [0.1173050281371213, 0.02084906243285388], [33.02290921896076, 0.731828402854083], [1.0793688055894515, 0.10931557050726745], [93.56136304282711, 2.0295199804569988], [1.816116322816606, 0.06161780280890299], [0.08679677166498763, 0.09952466851267003], [86.09438046407215, 1.8172226723210405], [1.9384854370303877, 0.07258482968096303], [49.162654100926936, 1.0979860326417152], [2.9689836395758653, 0.19248570047779084], [61.250031216448896, 1.219407905006056], [0.5878890718036862, 0.20645748887917084], [0.16207675110502628, 0.07131144145755983], [13.186374094111505, 0.36783848683809306], [13.978780555978403, 0.41628024086502774], [1.336888733318784, 0.21440900135193583], [26.898935835152514, 0.7297803037538164], [1.1537668527812341, 0.0019706778065951332], [2.7193035227577997, 0.03777303502681404], [0.055293153294860634, 0.14661527950525202], [10.691300210425872, 0.12690863379036701], [58.45133460845806, 1.2998762218352946], [1.160344624644817, 0.04646016604322195], [0.44776697049264835, 0.06401901549671359], [0.717256660457023, 0.12193317874552094], [1.9360641988800344, 0.1346641793983347], [0.840201688818753, 0.19226090988929603], [31.29774187757066, 0.6841817289256831], [0.0375331116891086, 0.198995535462547], [12.782824717331934, 0.3904907833292383], [133.75156529870668, 2.3747977601294687], [0.05152734159022951, 0.13953261205119408], [66.90090753599513, 1.3578860128857224], [54.66483836902647, 1.0832645191830728], [0.1725428948026942, 0.05065830054684348], [195.44480478875133, 3.5975503454106468], [315.4349122138283, 4.53463499822305], [47.931338034465924, 0.8670195344095187], [7.229894106695446, 0.035623575584007965], [83.39267244918177, 1.7070315904117799], [1.3121035136191939, 0.05678907875575125], [1.985749878948184, 0.08244988937166267], [1.3497490510137238, 0.2606742055560666], [0.12745199735737994, 0.3106379937314296], [0.03786175258130071, 0.29877299510390287], [32.28135063661292, 0.7407861221309628], [109.91639180827286, 1.956687121827336], [14.926949225445117, 0.3678746110846185], [2.283408494348471, 0.05338715287039065], [7.286217471621397, 0.11754533214365707], [65.46893299500965, 1.3890762789654119], [2.355439213813762, 0.028734671882496444], [2.3042631960935775, 0.024010904421780672], [0.5930596623205554, 0.017963257625698382], [78.81323090755711, 1.5476471737907105], [65.88626441665198, 1.2160412495797022], [8.066307298018977, 0.18679828936345347], [1.9938048272655913, 0.23674668889590667], [1.9312444854034516, 0.11124368897886955], [0.4862805283300847, 0.13345177560852187], [0.13485114700490952, 0.09206771202199708], [15.870615889454593, 0.4578733405782976], [0.3186601295526492, 0.016101175390923685], [0.2110160839587871, 0.27670889377455365], [0.42799255337245223, 0.3114041706770789], [3.4299313382411327, 0.013074471842972437], [0.04242530987144379, 0.23998983559811893], [0.8977171287952898, 0.004212492804981437], [1.1036228079190535, 0.002657596710326605], [57.85807046037331, 1.358984873961862], [0.0661625289323738, 0.079709867005051], [35.918106444170625, 0.6215769847045971], [0.3633304976255078, 0.1216689360373911], [8.42416828288917, 0.08797215943758355], [0.7515051770807412, 0.0035692338456645607], [29.444313862269663, 0.6288419260196683], [265.6726852597531, 4.142200978152486], [0.36763519194300753, 0.1395618229689431], [2.2259377146970083, 0.08640406389245789], [34.62444474734319, 0.7542510477216595], [4.006758624286858, 0.1697338249378201], [39.92051502368732, 0.7284973663034549], [0.31774027360991797, 0.021661860623422267], [1.5942283518458142, 0.12258411058061619], [1.8729834918973702, 0.08283188503103389], [6.842166023995335, 0.07556005900527321], [0.7910721313553095, 0.022172051113938624], [0.29088811782327917, 0.019438792138444384], [1.786796982289355, 0.08467771877807048], [13.610229952580923, 0.19204399125898447], [51.592221777746815, 1.1624788081542048], [10.206298041444239, 0.1251931335224751], [9.535998965634668, 0.27418474366554085], [1.4536384203008108, 0.05694726603325771], [0.21452992400963103, 0.033364515955862074], [1.1212322434237734, 0.04397700488652521], [2.9291820356800273, 0.17661116297738302], [65.63328721601123, 1.3271909016567982], [15.106978707843144, 0.35788459077612345], [46.62536398033839, 0.8021826926583767], [67.00867335191508, 1.282951407933341], [60.36580871087747, 1.2383601152408246], [13.490233651424415, 0.09496485684711668], [1.5891032717802733, 0.07765816751564063], [5.17426052885948, 0.1331296558248212], [4.348875389056856, 0.18333657469681866], [0.3180307872995434, 0.03545107359170036], [6.556971481275269, 0.16289004726526984], [31.316070992952504, 0.8130590875032703], [1.1954296497656607, 0.11398708316783872], [1.3899748491292117, 0.06869053026903983], [8.557997756471242, 0.16895260823433067], [1.991062052876798, 0.2538617196654], [0.2630003233164382, 0.2681762025889559], [7.562180745157672, 0.09552776597992861], [238.52703988786575, 3.8911278828465146], [166.98923072807625, 2.6304204841700254], [15.846227382082892, 0.509482542880348], [112.8719678859659, 2.123141876069432], [113.42646241900557, 1.9008891570510882], [6.803496689690116, 0.10275343632734302], [137.0203450146882, 2.4263646653679483], [29.673352943216965, 0.5192733896654371], [56.788126684620664, 1.1926110743864218], [52.576458870224045, 1.2615824046193835], [7.502452165503893, 0.09861846064731081], [3.5786119191359878, 0.24460603370307485], [120.03156542486462, 2.4107356190920637], [0.18889997696629388, 0.19379338198435986], [0.16171331566516647, 0.018780330269548298], [159.0936599506686, 2.462290890209324], [17.43047373012594, 0.37844647955904237], [5.525099517825602, 0.08724860920392291], [9.71354399551183, 0.1360870215330791], [8.726408822687691, 0.13586633730481318], [1.844703173989841, 0.18140057609593355], [71.86426596860302, 1.3327493350719508], [1.7063549352036662, 0.17208397376347218], [2.6944762602020966, 0.03788926647682604], [65.88115866669133, 1.3433137245724636], [23.569464232930972, 0.4248329820635905], [5.128160931349699, 0.04720229765528854], [0.6446704048693357, 0.2862856132081005], [10.082096233021895, 0.29191334985258927], [194.53644290495268, 3.0572359468265535], [7.932592798389104, 0.20163981198401137], [81.73359418503512, 1.6486317255690028], [32.224941180894334, 0.711226222060694], [18.375504089498346, 0.44516494742407253], [9.459370075711046, 0.36955574741133784], [0.11722797531416113, 0.23242690987627607], [0.19869910506882285, 0.0013854004284416636], [133.92176218976672, 2.5245227880276992], [0.1137317316220835, 0.19119322864373806], [0.31843283623898205, 0.017252646962895492], [0.3571992717115799, 0.3150235419990068], [95.58432203650636, 1.9968962532193117], [136.32360500575027, 2.533344032918118], [6.040172970760941, 0.1644706433692038], [43.34497123156367, 0.9215015231413646], [7.869408490976892, 0.15762320968318466], [63.30601279164901, 1.4165015708273456], [68.43495892551648, 1.1600652554705162], [4.394509732002527, 0.21940595764887713], [3.45117016171959, 0.10840623290277669], [0.5430750345469353, 0.09465778004491429], [0.27707756253174165, 0.025902682708195183], [0.08083010679194302, 0.24005197937601572], [9.469832576756557, 0.022909996052707566], [105.2774247448849, 2.1312656639809915], [1.470938040360105, 0.09979212374570916], [26.67731447258877, 0.6128909737014747], [94.88938355858267, 2.0288663541487733], [6.036889474405876, 0.016355307784627876], [3.572849152025413, 0.24916588383188265], [3.272555814913205, 0.2143700771353627], [8.401688113005106, 0.22453193237084035], [1.9944627595411522, 0.11433292631998161], [6.443754086934638, 0.044972723627309835], [90.64864002488395, 1.7126868024679114], [0.5192554616491589, 0.3209507731532191], [3.9469366672116273, 0.10952442481633723], [1.1547354396963332, 0.0779890252740009], [0.36038352941063834, 0.03451657186456637], [18.871762226434875, 0.3096681982058396], [22.600709220626335, 0.3161748208595446], [4.786683062918212, 0.17606755090700968], [1.8567344111511457, 0.01041629727349364], [96.85324415737648, 1.8462897812747519], [48.89013475130641, 0.9115763404903925], [0.687391794074458, 0.029471613286694884], [53.87795873477659, 1.0918271946089648], [1.6973628617582264, 0.03252383232167164], [24.4216744060855, 0.6040334040166017], [1.072553451413934, 0.3066050282169589], [56.48507448990138, 1.193705709817589], [3.6198411068919554, 0.24188881420655994], [37.486887355700695, 0.7372185352410068], [16.169152300889444, 0.461342618087275], [0.1037690584256407, 0.009425757438056502], [0.4686047941727564, 0.013624326154703399], [397.0548997151921, 5.574044450998942], [84.06087237475506, 1.5533955949040321], [4.590827789864803, 0.1148179908117526], [1.1926256606916263, 0.2558417896233182], [1.242446769121103, 0.26580919626485144]]

    n = 10
    x_train = x_data[:len(x_data)-n]
    y_train = y_data[:len(y_data)-n]
    x_test = x_data[-n:]
    y_test = y_data[-n:]

    print("Number of samples: {}".format(len(x_data)))

    return x_train, y_train, x_test, y_test


def regressor_stats(regressor, x_test, y_test):
    # test
    print(regressor.__class__.__name__)
    for i in range(len(x_test)):
        x = x_test[i]
        y = y_test[i]
        val = regressor.predict([x])

        value_problem = y[0]
        value_surrogate = val[0]
        percent = 100.0 * math.fabs(value_problem - value_surrogate) / math.fabs(value_problem)

        print("params = [{0:8.3f}, {1:8.3f}, {2:8.3f}], \teval = {3:8.3f}, \tpred = {4:8.3f}, \tdiff = {5:8.3f} \t({6:8.3f} %)".format(x[0], x[1], x[2], value_problem, value_surrogate,
                                                                                            math.fabs(value_problem - value_surrogate), percent))

def regressor_optim():
    with warnings.catch_warnings():
        warnings.simplefilter("ignore")

        x_train, y_train, x_test, y_test = regressor_data()

        # dict_keys(['bootstrap', 'criterion', 'max_depth', 'max_features', 'max_leaf_nodes', 'min_impurity_decrease', 'min_impurity_split', 'min_samples_leaf', 'min_samples_split', 'min_weight_fraction_leaf', 'n_estimators', 'n_jobs', 'oob_score', 'random_state', 'verbose', 'warm_start'])
        # param_grid = {"max_depth": [3, None], "max_features": [1], "min_samplesx_split": [2, 3, 10], "bootstrap": [True, False]}
        # regressor = RandomForestRegressor(n_estimators=20)

        # param_grid = {"alpha": [10, 1e0, 0.1, 1e-2, 1e-3], "gamma": np.logspace(-5, -2, 2, 5, 10)}
        # regressor = KernelRidge(kernel='rbf', gamma=0.1)

        # print(regressor.get_params().keys())



        # run grid search
        start = time.time()
        # log-uniform: understand as search over p = exp(x) by varying x
        # grid_search = BayesSearchCV(
        #     SVR(),
        #     {
        #         'C': (1e-6, 1e+6, 'log-uniform'),
        #         'gamma': (1e-6, 1e+1, 'log-uniform'),
        #         'degree': (1, 8),  # integer valued parameter
        #         'kernel': ['linear', 'poly', 'rbf'],  # categorical parameter
        #     },
        #     n_iter=32
        # )

        ker_rbf1 = RBF(length_scale=1.0, length_scale_bounds=(1e-1, 10.0))
        ker_rbf2 = ConstantKernel(1.0, constant_value_bounds="fixed") * RBF(1.0, length_scale_bounds=(1e-1, 10.0))
        ker_rq = ConstantKernel(1.0, constant_value_bounds="fixed") * RationalQuadratic(alpha=0.1, length_scale=1)
        ker_expsine = ConstantKernel(1.0, constant_value_bounds="fixed") * ExpSineSquared(1.0, 5.0, periodicity_bounds=(1e-2, 1e1))
        #
        # kernel_list = [ker_rbf1, ker_rbf2, ker_rq, ker_expsine]
        kernel_list = [ker_rbf1]
        #
        param_grid = {"kernel": kernel_list,
                      # "alpha": [1e-10, 1e-12, 1e-5, 1e-2, 1e-1, 1e0, 1e1],
                      "alpha": [1e-10],
                       "optimizer": ["fmin_l_bfgs_b"],
                       # "n_restarts_optimizer": [1, 3, 6, 9, 12, 20],
                       "n_restarts_optimizer": [9],
                       "normalize_y": [False],
                       "copy_X_train": [True],
                       "random_state": [0]}
        gp = GaussianProcessRegressor()

        # gp = RandomForestRegressor()
        # param_grid = {
        #     "n_estimators": [10, 20, 30],
        #     "max_features": ["auto", "sqrt", "log2"],
        #     "min_samples_split": [2, 4, 8],
        #     "bootstrap": [True, False],
        # }
        # GaussianProcessRegressor(kernel=1.0 * RBF(length_scale=1.0, length_scale_bounds=(1e-1, 10.0)), n_restarts_optimizer=9)

        print("\nRunning grid search to tune up GPR parameters")
        # grid_search = GridSearchCV(gp, param_grid=param_grid, n_jobs=-1, scoring=None) #
        #grid_search = BayesSearchCV(gp, search_spaces=param_grid)
        grid_search = RandomizedSearchCV(gp, param_distributions=param_grid, n_jobs=-1, n_iter=1, scoring='neg_mean_squared_error')

        # grid_search = BayesSearchCV(gp, param_grid)

        # use a full grid over all parameters
        # grid_search = GridSearchCV(regressor, param_grid=param_grid, cv=5)

        grid_search.fit(x_train, np.ravel(y_train, order='C'))
        print("SearchCV took %.2f seconds for %d candidate parameter settings." % (time.time() - start, len(grid_search.cv_results_['params'])))
        # report(grid_search.cv_results_)
        print(grid_search.best_params_)

        regressor_stats(grid_search, x_test, y_test)

        grid_search = GaussianProcessRegressor(kernel=1.0 * RBF(length_scale=1.0, length_scale_bounds=(1e-1, 10.0)), n_restarts_optimizer=9)
        grid_search.fit(x_train, np.ravel(y_train, order='C'))
        regressor_stats(grid_search, x_test, y_test)

# scikit()
# plot()

regressor_optim()