#!/usr/bin/python
#
# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Utilities for our job runner.
"""
import getpass
import itertools as it
import os
import platform
import sys
from enum import Enum
from typing import (Any, Callable, Dict, Iterable, List, NamedTuple, Optional,
                    Set, Tuple, Union)

from blessings import Terminal
from pkg_resources import resource_filename

t = Terminal()

Package = NamedTuple("Package", [("executable", List[str]),
                                 ("package_path", str), ("script_path", str),
                                 ("main_module", Optional[str])])


def err(s: str) -> None:
  """Prints the supplied string to stderr in red text."""
  sys.stderr.write(t.red(s))


def current_user() -> str:
  return getpass.getuser()


def is_mac() -> bool:
  """Returns True if the current code is executing on a Mac, False otherwise.

  """
  return platform.system() == "Darwin"


def is_linux() -> bool:
  """Returns True if the current code is executing on a Linux system, False
  otherwise.

  """
  return platform.system() == "Darwin"


def resource(name: str) -> Optional[str]:
  """If the supplied resource exists in caliban.resources, returns the absolute
  path to the resource. Else, returns None.

  """
  path = resource_filename("caliban.resources", name)
  if os.path.exists(path):
    return path

  return None


def enum_vals(enum: Enum) -> List[str]:
  """Returns the list of all values for a specific enum."""
  return [v.value for v in enum]


def any_of(value_s: str, union_type: Union) -> Any:
  """Attempts to parse the supplied string into one of the components of the
  supplied Union. Returns the value if possible, else raises a value error.

  union_type must be a union of enums!

  """

  def attempt(s: str, enum_type: Enum) -> Optional[Any]:
    try:
      return enum_type(s)
    except ValueError:
      return None

  enums = union_type.__args__
  ret = None

  for enum_type in enums:
    ret = attempt(value_s, enum_type)
    if ret is not None:
      break

  if ret is None:
    raise ValueError("{} isn't a value of any of {}".format(value_s, enums))

  return ret


def dict_product(m: Dict[Any, Any]) -> Iterable[Dict[Any, Any]]:
  """Returns a dictionary generated by taking the cartesian product of each
  list-typed value iterable with all others.

  The iterable of dictionaries returned represents every combination of values.

  If any value is NOT a list it will be treated as a singleton list.

  """

  def wrap_v(v):
    return v if isinstance(v, list) else [v]

  ks = m.keys()
  vs = (wrap_v(v) for v in m.values())
  return (dict(zip(ks, x)) for x in it.product(*vs))


def flipm(table: Dict[Any, Dict[Any, Any]]) -> Dict[Any, Dict[Any, Any]]:
  """Handles shuffles for a particular kind of table."""
  ret = {}
  for k, m in table.items():
    for k2, v in m.items():
      ret.setdefault(k2, {})[k] = v

  return ret


def invertm(table: Dict[Any, Iterable[Any]]) -> Dict[Any, Set[Any]]:
  """Handles shuffles for a particular kind of table."""
  ret = {}
  for k, vs in table.items():
    for v in vs:
      ret.setdefault(v, set()).add(k)

  return ret


def reorderm(table: Dict[Any, Dict[Any, Iterable[Any]]],
             order: Tuple[int, int, int]) -> Dict[Any, Dict[Any, Set[Any]]]:
  """Handles shuffles for a particular kind of table."""
  ret = {}
  for k, m in table.items():
    for k2, vs in m.items():
      for v in vs:
        fields = [k, k2, v]
        innerm = ret.setdefault(fields[order[0]], {})
        acc = innerm.setdefault(fields[order[1]], set())
        acc.add(fields[order[2]])

  return ret


def merge(l: Dict[Any, Any], r: Dict[Any, Any]) -> Dict[Any, Any]:
  """Returns a new dictionary by merging the two supplied dictionaries."""
  ret = l.copy()
  ret.update(r)
  return ret


def dict_by(keys: Set[str], f: Callable[[str], Any]) -> Dict[str, Any]:
  """Returns a dictionary with keys equal to the supplied keyset. Each value is
  the result of applying f to a key in keys.

  """
  return {k: f(k) for k in keys}


def split_by(items: List[str],
             separator: Optional[str] = None) -> Tuple[List[str], List[str]]:
  """If the separator is present in the list, returns a 2-tuple of

  - the items before the separator,
  - all items after the separator.

  If the separator isn't present, returns a tuple of

  - (the original list, [])

  """
  if separator is None:
    separator = '--'

  try:
    idx = items.index(separator)
    return items[0:idx], items[idx + 1:]
  except ValueError:
    return (items, [])


def n_chunks(items: List[Any], n_groups: int) -> List[List[Any]]:
  """Returns a list of `n_groups` slices of the original list, guaranteed to
  contain all of the original items.
  """
  return [items[i::n_groups] for i in range(n_groups)]


def chunks_below_limit(items: List[Any], limit: int) -> List[List[Any]]:
  """Breaks the input list into a series of chunks guaranteed to be less than"""
  quot, _ = divmod(len(items), limit)
  return n_chunks(items, quot + 1)


def partition(seq: List[str], n: int) -> List[List[str]]:
  """Generate groups of n items from seq by scanning across the sequence and
  taking chunks of n, offset by 1.
  """
  for i in range(0, max(1, len(seq) - n + 1), 1):
    yield seq[i:i + n]
