#!/usr/bin/python
#
# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Utilities for interacting with the filesystem and packages.

"""
import io
import os
import shutil
import socket
import subprocess
import sys
import time
import uuid
from itertools import chain
from typing import Dict, List, NamedTuple, Optional, Tuple

from blessings import Terminal

t = Terminal()

Package = NamedTuple("Package", [("executable", List[str]),
                                 ("package_path", str), ("script_path", str),
                                 ("main_module", Optional[str])])


def file_exists_in_cwd(path: str) -> bool:
  """Returns True if the current path references a valid file in the current
  directory, False otherwise.

  """
  return os.path.isfile(os.path.join(os.getcwd(), path))


def extract_root_directory(path: str) -> str:
  """Returns the root directory of the supplied path."""
  items = path.split(os.path.sep)
  return "." if len(items) == 1 else items[0]


def module_package(main_module: str) -> Package:
  """Generates a Package instance for a python module executable that should be
  executed with python -m.

  """
  script_path = module_to_path(main_module)
  root = extract_root_directory(script_path)
  return Package(["python", "-m"],
                 package_path=root,
                 script_path=script_path,
                 main_module=main_module)


def script_package(path: str, executable: str = "/bin/bash") -> Package:
  """Generates a Package instance for a non-python-module executable."""
  root = extract_root_directory(path)
  return Package([executable],
                 package_path=root,
                 script_path=path,
                 main_module=None)


def path_to_module(path_str: str) -> str:
  return path_str.replace(".py", "").replace(os.path.sep, ".")


def module_to_path(module_name: str) -> str:
  """Converts the supplied python module (module names separated by dots) into
  the python file represented by the module name.

  """
  return module_name.replace(".", os.path.sep) + ".py"


def generate_package(path: str,
                     executable: Optional[List[str]] = None,
                     main_module: Optional[str] = None) -> Package:
  """Takes in a string and generates a package instance that we can use for
  imports.
  """
  if executable is None:
    _, ext = os.path.splitext(path)
    executable = ["python"] if ext == ".py" else ["/bin/bash"]

  if main_module is None and not file_exists_in_cwd(path):
    module_path = module_to_path(path)

    if file_exists_in_cwd(module_path):
      return generate_package(module_path,
                              executable=["python", "-m"],
                              main_module=path_to_module(module_path))

  root = extract_root_directory(path)
  return Package(executable, root, path, main_module)


class TempCopy(object):
  """TempCopy is a class that you can use as a context manager to transfer files
  into the local directory with either

  - a name specified by you, or
  - an autogenerated name.

  Supply a dictionary mapping your files => your requested filename (or None).
  When used as a context manager, TempCopy will

  - copy data from every source to every destination
  - return a dictionary of the source you provided => the destination's path in
    the local directory.

  The temporary files will live inside the current directory where python's
  being executed for the duration of TempCopy's scope.

  We did NOT use a tmp directory here because the changing UUID name
  invalidates the docker image each time a new temp path / directory is
  generated.

  mapping: dict mapping from optional src to optional destination filename. The
           source should be a relative or absolute path on the local machine.
           The destination should be a current-directory-local path, NOT an
           absolute path. If a key is None, nothing will happen.

  Returns:
  dict of each non-None key in mapping => the destination's relative path in
  the local directory.

  """

  def __init__(self,
               mapping: Optional[Dict[Optional[str], Optional[str]]] = None):
    if mapping is None:
      mapping = {}

    self._mapping = self._sanitize_entries(mapping)
    self._written = None

  @property
  def active(self) -> bool:
    """Returns true if the TmpCopy instance is managing written data, false
    otherwise.

    """
    return self._written is not None

  def _sanitize_pair(self, src: Optional[str],
                     dst: Optional[str]) -> List[Tuple[str, str]]:
    """ If the source isn't none, returns a singleton with a pair of (from
    path, to_path). Else, returns []."""

    if src is None:
      return []

    return [(src, dst or str(uuid.uuid1()))]

  def _sanitize_entries(
      self, mapping: Dict[Optional[str], Optional[str]]) -> Dict[str, str]:
    """Generates a dictionary of source file -> destination file, based on the
    information in the original mapping.

    If a key in the original mapping is None, it won't appear in the output. If
    a value is None, a filename will be generated and paired with the
    corresponding key.

    """

    processed = [self._sanitize_pair(src, dst) for src, dst in mapping.items()]
    return dict(chain.from_iterable(processed))

  def _expand(self, current_dir: str, mapping: Dict[str,
                                                    str]) -> Dict[str, str]:
    """Expands the mapping that the user sees out into a mapping from absolute
    source to absolute destination.

    """

    def final_src(src):
      return os.path.abspath(os.path.expanduser(src))

    def final_dst(dst):
      return os.path.join(current_dir, dst)

    return {final_src(k): final_dst(v) for k, v in mapping.items()}

  def __enter__(self) -> Dict[str, str]:
    """Entrypoint for the context manager. Performs the side effect of writing out
    the contents of every file specified as a key of self._mapping over into
    temporary files in the current working directory (at the time of the
    context manager's activation).

    """
    current_dir = os.getcwd()
    mapping = {**self._mapping}

    to_write = self._expand(current_dir, mapping)

    for src, dst in to_write.items():
      shutil.copy2(src, dst)

    self._written = to_write

    return mapping

  def __exit__(self, exc_type, exc_val, exc_tb):
    """If the context manager is holding on to temporary files, delete them all and
    reset the internal state.

    """
    if self._written is not None:
      for _, dst in self._written.items():
        if os.path.exists(dst):
          os.remove(dst)
      self._written = None


def capture_stdout(cmd: List[str],
                   input_str: Optional[str] = None,
                   file=None) -> str:
  """Executes the supplied command with the supplied string of std input, then
  streams the output to stdout, and returns it as a string along with the
  process's return code.

  Args:
  cmd: list of strings to send in as the command
  input_str: if supplied, this string will be passed as stdin to the supplied
             command. if None, stdin will get closed immediately.
  file: optional file-like object (stream): the output from the executed
        process's stdout will get sent to this stream. Defaults to sys.stdout.

  Returns:
  Pair of
  - string of all stdout received during the command's execution
  - return code of the process

  """
  if file is None:
    file = sys.stdout

  buf = io.StringIO()
  ret_code = None

  with subprocess.Popen(cmd,
                        stdin=subprocess.PIPE,
                        stdout=subprocess.PIPE,
                        stderr=subprocess.STDOUT,
                        universal_newlines=False,
                        bufsize=1) as p:
    if input_str:
      p.stdin.write(input_str.encode('utf-8'))
    p.stdin.close()

    out = io.TextIOWrapper(p.stdout, newline='')

    for line in out:
      buf.write(line)
      file.write(line)
      file.flush()

    # flush to force the contents to display.
    file.flush()

    while p.poll() is None:
      # Process hasn't exited yet, let's wait some
      time.sleep(0.5)

    ret_code = p.returncode
    p.stdout.close()

  return buf.getvalue(), ret_code


def next_free_port(port: int, try_n: int = 1000, max_port=65535):
  if try_n == 0 or port <= max_port:
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
      sock.bind(('', port))
      sock.close()
      return port
    except OSError:
      return next_free_port(port + 1, try_n - 1, max_port=max_port)
  else:
    raise IOError('no free ports')
