# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['stormevents', 'stormevents.coops', 'stormevents.nhc', 'stormevents.usgs']

package_data = \
{'': ['*']}

install_requires = \
['beautifulsoup4',
 'geopandas',
 'netcdf4',
 'numpy',
 'pandas',
 'pyproj>=2.6',
 'python-dateutil',
 'requests',
 'shapely',
 'typepigeon>=1.0.5',
 'xarray']

extras_require = \
{'development': ['isort', 'oitnb'],
 'documentation': ['m2r2', 'sphinx', 'sphinx-rtd-theme'],
 'testing': ['pytest', 'pytest-cov', 'pytest-socket', 'pytest-xdist']}

setup_kwargs = {
    'name': 'stormevents',
    'version': '1.4.1',
    'description': 'Python interfaces for observational data surrounding named storm events',
    'long_description': "# StormEvents\n\n[![tests](https://github.com/oceanmodeling/StormEvents/workflows/tests/badge.svg)](https://github.com/oceanmodeling/StormEvents/actions?query=workflow%3Atests)\n[![build](https://github.com/oceanmodeling/StormEvents/workflows/build/badge.svg)](https://github.com/oceanmodeling/StormEvents/actions?query=workflow%3Abuild)\n[![codecov](https://codecov.io/gh/oceanmodeling/StormEvents/branch/main/graph/badge.svg?token=BQWB1QKJ3Q)](https://codecov.io/gh/oceanmodeling/StormEvents)\n[![version](https://img.shields.io/pypi/v/StormEvents)](https://pypi.org/project/StormEvents)\n[![license](https://img.shields.io/github/license/oceanmodeling/StormEvents)](https://opensource.org/licenses/gpl-license)\n[![style](https://sourceforge.net/p/oitnb/code/ci/default/tree/_doc/_static/oitnb.svg?format=raw)](https://sourceforge.net/p/oitnb/code)\n\n`stormevents` provides Python interfaces for observational data surrounding named storm events.\n\n```bash\npip install stormevents\n```\n\nFull documentation can be found at https://stormevents.readthedocs.io\n\n## Usage\n\nThere are two ways to retrieve observational data via `stormevents`;\n\n1. retrieve data for any arbitrary time interval / region, or\n2. retrieve data surrounding a specific storm.\n\n### retrieve data for any arbitrary time interval / region\n\n`stormevents` currently implements retrieval for\n\n- storm tracks from the National Hurricane Center (NHC),\n- high-water mark (HWM) surveys provided by the United States Geological Survey (USGS), and\n- data products from the Center for Operational Oceanographic Products and Services (CO-OPS).\n\n#### storm tracks from the National Hurricane Center (NHC)\n\nThe [National Hurricane Center (NHC)](https://www.nhc.noaa.gov) tracks and tropical cyclones dating back to 1851.\n\nThe `nhc_storms()` function provides a list of NHC storms from their online archive:\n\n```python\nfrom stormevents.nhc import nhc_storms\n\nnhc_storms()\n```\n\n```\n                name class  year basin  number    source          start_date            end_date\nnhc_code\nAL021851     UNNAMED    HU  1851    AL       2   ARCHIVE 1851-07-05 12:00:00 1851-07-05 12:00:00\nAL031851     UNNAMED    TS  1851    AL       3   ARCHIVE 1851-07-10 12:00:00 1851-07-10 12:00:00\nAL041851     UNNAMED    HU  1851    AL       4   ARCHIVE 1851-08-16 00:00:00 1851-08-27 18:00:00\nAL051851     UNNAMED    TS  1851    AL       5   ARCHIVE 1851-09-13 00:00:00 1851-09-16 18:00:00\nAL061851     UNNAMED    TS  1851    AL       6   ARCHIVE 1851-10-16 00:00:00 1851-10-19 18:00:00\n...              ...   ...   ...   ...     ...       ...                 ...                 ...\nCP902021      INVEST    LO  2021    CP      90  METWATCH 2021-07-24 12:00:00                 NaT\nCP912021      INVEST    DB  2021    CP      91  METWATCH 2021-08-07 18:00:00                 NaT\nEP922021      INVEST    DB  2021    EP      92  METWATCH 2021-06-05 06:00:00                 NaT\nEP712022  GENESIS001    DB  2022    EP      71   GENESIS 2022-01-20 12:00:00                 NaT\nEP902022      INVEST    LO  2022    EP      90  METWATCH 2022-01-20 12:00:00                 NaT\n\n[2714 rows x 8 columns]\n```\n\n##### retrieve storm track by NHC code\n\n```python\nfrom stormevents.nhc import VortexTrack\n\ntrack = VortexTrack('AL112017')\ntrack.data\n```\n\n```\n    basin storm_number            datetime advisory_number  ... isowave_radius_for_SWQ extra_values                    geometry  track_start_time\n0      AL           11 2017-08-30 00:00:00                  ...                    NaN         <NA>  POINT (-26.90000 16.10000)        2017-08-30\n1      AL           11 2017-08-30 06:00:00                  ...                    NaN         <NA>  POINT (-28.30000 16.20000)        2017-08-30\n2      AL           11 2017-08-30 12:00:00                  ...                    NaN         <NA>  POINT (-29.70000 16.30000)        2017-08-30\n3      AL           11 2017-08-30 18:00:00                  ...                    NaN         <NA>  POINT (-30.80000 16.30000)        2017-08-30\n4      AL           11 2017-08-30 18:00:00                  ...                    NaN         <NA>  POINT (-30.80000 16.30000)        2017-08-30\n..    ...          ...                 ...             ...  ...                    ...          ...                         ...               ...\n168    AL           11 2017-09-12 12:00:00                  ...                    NaN         <NA>  POINT (-86.90000 33.80000)        2017-08-30\n169    AL           11 2017-09-12 18:00:00                  ...                    NaN         <NA>  POINT (-88.10000 34.80000)        2017-08-30\n170    AL           11 2017-09-13 00:00:00                  ...                    NaN         <NA>  POINT (-88.90000 35.60000)        2017-08-30\n171    AL           11 2017-09-13 06:00:00                  ...                    NaN         <NA>  POINT (-89.50000 36.20000)        2017-08-30\n172    AL           11 2017-09-13 12:00:00                  ...                    NaN         <NA>  POINT (-90.10000 36.80000)        2017-08-30\n\n[173 rows x 38 columns]\n```\n\n##### retrieve storm track by name and year\n\nIf you do not know the storm code, you can input the storm name and year:\n\n```python\nfrom stormevents.nhc import VortexTrack\n\nVortexTrack.from_storm_name('irma', 2017)\n```\n\n```\nVortexTrack('AL112017', Timestamp('2017-08-30 00:00:00'), Timestamp('2017-09-13 12:00:00'), <ATCF_FileDeck.BEST: 'b'>, [<ATCF_Advisory.BEST: 'BEST'>], None)\n```\n\n##### specify storm track file deck\n\nBy default, `VortexTrack` retrieves data from the `BEST` track file deck (`b`). You can specify that you want\nthe `ADVISORY` (`a`) or `FIXED` (`f`) file decks with the `file_deck` parameter.\n\n```python\nfrom stormevents.nhc import VortexTrack\n\ntrack = VortexTrack('AL112017', file_deck='a')\ntrack.data\n```\n\n```\n      basin storm_number            datetime advisory_number  ... isowave_radius_for_SWQ extra_values                    geometry    track_start_time\n0        AL           11 2017-08-27 06:00:00              01  ...                    NaN         <NA>  POINT (-17.40000 11.70000) 2017-08-28 06:00:00\n1        AL           11 2017-08-27 12:00:00              01  ...                    NaN         <NA>  POINT (-17.90000 11.80000) 2017-08-28 06:00:00\n2        AL           11 2017-08-27 18:00:00              01  ...                    NaN         <NA>  POINT (-18.40000 11.90000) 2017-08-28 06:00:00\n3        AL           11 2017-08-28 00:00:00              01  ...                    NaN         <NA>  POINT (-19.00000 12.00000) 2017-08-28 06:00:00\n4        AL           11 2017-08-28 06:00:00              01  ...                    NaN         <NA>  POINT (-19.50000 12.00000) 2017-08-28 06:00:00\n...     ...          ...                 ...             ...  ...                    ...          ...                         ...                 ...\n10739    AL           11 2017-09-12 00:00:00              03  ...                    NaN         <NA>  POINT (-84.40000 31.90000) 2017-09-12 00:00:00\n10740    AL           11 2017-09-12 03:00:00              03  ...                    NaN         <NA>  POINT (-84.90000 32.40000) 2017-09-12 00:00:00\n10741    AL           11 2017-09-12 12:00:00              03  ...                    NaN         <NA>  POINT (-86.40000 33.80000) 2017-09-12 00:00:00\n10742    AL           11 2017-09-13 00:00:00              03  ...                    NaN         <NA>  POINT (-88.20000 35.20000) 2017-09-12 00:00:00\n10743    AL           11 2017-09-13 12:00:00              03  ...                    NaN         <NA>  POINT (-88.60000 36.40000) 2017-09-12 00:00:00\n\n[10434 rows x 38 columns]\n```\n\n##### read storm track from file\n\nIf you have an ATCF or `fort.22` file, use the corresponding methods:\n\n```python\nfrom stormevents.nhc import VortexTrack\n\nVortexTrack.from_file('tests/data/input/test_vortex_track_from_file/AL062018.dat')\n```\n\n```\nVortexTrack('AL062018', Timestamp('2018-08-30 06:00:00'), Timestamp('2018-09-18 12:00:00'), None, <ATCF_Mode.HISTORICAL: 'ARCHIVE'>, ['BEST', 'OFCL', 'OFCP', 'HMON', 'CARQ', 'HWRF'], PosixPath('/home/zrb/Projects/StormEvents/tests/data/input/test_vortex_track_from_file/AL062018.dat'))\n```\n\n```python\nfrom stormevents.nhc import VortexTrack\n\nVortexTrack.from_file('tests/data/input/test_vortex_track_from_file/irma2017_fort.22')\n```\n\n```\nVortexTrack('AL112017', Timestamp('2017-09-05 00:00:00'), Timestamp('2017-09-12 00:00:00'), None, <ATCF_Mode.HISTORICAL: 'ARCHIVE'>, ['BEST', 'OFCL', 'OFCP', 'HMON', 'CARQ', 'HWRF'], PosixPath('/home/zrb/Projects/StormEvents/tests/data/input/test_vortex_track_from_file/irma2017_fort.22'))\n```\n\n##### write storm track to `fort.22` file\n\n```python\nfrom stormevents.nhc import VortexTrack\n\ntrack = VortexTrack.from_storm_name('florence', 2018)\ntrack.to_file('fort.22')\n```\n\n#### high-water mark (HWM) surveys provided by the United States Geological Survey (USGS)\n\nThe [United States Geological Survey (USGS)](https://www.usgs.gov)\nconducts surveys of flooded areas following flood events to determine the highest level of water elevation, and provides the\nresults of these surveys via their API.\n\n##### list flood events defined by the USGS that have HWM surveys\n\n```python\nfrom stormevents.usgs import usgs_flood_events\n\nusgs_flood_events()\n```\n\n```\n                                            name  year                                        description  ... last_updated_by          start_date            end_date\nusgs_id                                                                                                    ...\n7                             FEMA 2013 exercise  2013                   Ardent/Sentry 2013 FEMA Exercise  ...             NaN 2013-05-15 04:00:00 2013-05-23 04:00:00\n8                                          Wilma  2005  Category 3 in west FL. \\nHurricane Wilma was t...  ...             NaN 2005-10-20 00:00:00 2005-10-31 00:00:00\n9                            Midwest Floods 2011  2011  Spring and summer 2011 flooding of the Mississ...  ...            35.0 2011-02-01 06:00:00 2011-08-30 05:00:00\n10                          2013 - June PA Flood  2013           Localized summer rain, small scale event  ...             NaN 2013-06-23 00:00:00 2013-07-01 00:00:00\n11               Colorado 2013 Front Range Flood  2013  A large prolonged precipitation event resulted...  ...            35.0 2013-09-12 05:00:00 2013-09-24 05:00:00\n...                                          ...   ...                                                ...  ...             ...                 ...                 ...\n312                    2021 Tropical Cyclone Ida  2021                                                NaN  ...           864.0 2021-08-27 05:00:00 2021-09-03 05:00:00\n313                Chesapeake Bay - October 2021  2021     Coastal-flooding event in the  Chesapeake Bay.  ...           406.0 2021-10-28 04:00:00                 NaT\n314      2021 November Flooding Washington State  2021                         Atmospheric River Flooding  ...           864.0 2021-11-08 06:00:00 2021-11-19 06:00:00\n315          Washington Coastal Winter 2021-2022  2021                                                NaN  ...           864.0 2021-11-01 05:00:00 2022-06-30 05:00:00\n317        2022 Hunga Tonga-Hunga Haapai tsunami  2022                                                     ...             1.0 2022-01-14 05:00:00 2022-01-18 05:00:00\n\n[293 rows x 11 columns]\n```\n\n##### retrieve HWM survey data for any flood event\n\n```python\nfrom stormevents.usgs import USGS_Event\n\nflood = USGS_Event(182)\nflood.high_water_marks()\n```\n\n```\n         latitude  longitude            eventName hwmTypeName  ... hwm_uncertainty                                          hwm_notes siteZone                    geometry\nhwm_id                                                         ...\n22602   31.170642 -81.428402  Irma September 2017      Debris  ...             NaN                                                NaN      NaN  POINT (-81.42840 31.17064)\n22605   31.453850 -81.362853  Irma September 2017   Seed line  ...             0.1                                                NaN      NaN  POINT (-81.36285 31.45385)\n22612   30.720000 -81.549440  Irma September 2017   Seed line  ...             NaN  There is a secondary peak around 5.5 ft, so th...      NaN  POINT (-81.54944 30.72000)\n22636   32.007730 -81.238270  Irma September 2017   Seed line  ...             0.1  Trimble R8 used to establish TBM. Levels ran f...      NaN  POINT (-81.23827 32.00773)\n22653   31.531078 -81.358894  Irma September 2017   Seed line  ...             NaN                                                NaN      NaN  POINT (-81.35889 31.53108)\n...           ...        ...                  ...         ...  ...             ...                                                ...      ...                         ...\n26171   18.470402 -66.246631  Irma September 2017      Debris  ...             0.5                                                NaN      NaN  POINT (-66.24663 18.47040)\n26173   18.470300 -66.449900  Irma September 2017      Debris  ...             0.5                                levels from GNSS BM      NaN  POINT (-66.44990 18.47030)\n26175   18.463954 -66.140869  Irma September 2017      Debris  ...             0.5                                levels from GNSS BM      NaN  POINT (-66.14087 18.46395)\n26177   18.488720 -66.392160  Irma September 2017      Debris  ...             0.5                                levels from GNSS BM      NaN  POINT (-66.39216 18.48872)\n26179   18.005607 -65.871768  Irma September 2017      Debris  ...             0.5                                levels from GNSS BM      NaN  POINT (-65.87177 18.00561)\n\n[506 rows x 53 columns]\n```\n\n```python\nfrom stormevents.usgs import USGS_Event\n\nflood = USGS_Event(182)\nflood.high_water_marks(quality=['EXCELLENT', 'GOOD'])\n```\n\n```\n         latitude  longitude            eventName hwmTypeName  ...                                          hwm_notes peak_summary_id siteZone                    geometry\nhwm_id                                                         ...\n22605   31.453850 -81.362853  Irma September 2017   Seed line  ...                                                NaN             NaN      NaN  POINT (-81.36285 31.45385)\n22612   30.720000 -81.549440  Irma September 2017   Seed line  ...  There is a secondary peak around 5.5 ft, so th...             NaN      NaN  POINT (-81.54944 30.72000)\n22636   32.007730 -81.238270  Irma September 2017   Seed line  ...  Trimble R8 used to establish TBM. Levels ran f...             NaN      NaN  POINT (-81.23827 32.00773)\n22674   32.030907 -80.900605  Irma September 2017   Seed line  ...                                                NaN          5042.0      NaN  POINT (-80.90061 32.03091)\n22849   30.741940 -81.687780  Irma September 2017      Debris  ...                                                NaN          4834.0      NaN  POINT (-81.68778 30.74194)\n...           ...        ...                  ...         ...  ...                                                ...             ...      ...                         ...\n25150   30.038222 -81.880928  Irma September 2017   Seed line  ...                              GNSS Level II survey.             NaN      NaN  POINT (-81.88093 30.03822)\n25151   30.118110 -81.760220  Irma September 2017   Seed line  ...                             GNSS Level III survey.             NaN      NaN  POINT (-81.76022 30.11811)\n25158   29.720560 -81.506110  Irma September 2017   Seed line  ...                              GNSS Level II survey.             NaN      NaN  POINT (-81.50611 29.72056)\n25159   30.097514 -81.794375  Irma September 2017   Seed line  ...                             GNSS Level III survey.             NaN      NaN  POINT (-81.79438 30.09751)\n25205   29.783890 -81.263060  Irma September 2017   Seed line  ...                              GNSS Level II survey.             NaN      NaN  POINT (-81.26306 29.78389)\n\n[277 rows x 53 columns]\n```\n\n#### data products from the Center for Operational Oceanographic Products and Services (CO-OPS)\n\nThe [Center for Operational Oceanographic Products and Services (CO-OPS)](https://tidesandcurrents.noaa.gov)\nmaintains and operates a large array of tidal buoys and oceanic weather stations that measure water and atmospheric variables\nacross the coastal United States. CO-OPS provides several [data products](https://tidesandcurrents.noaa.gov/products.html)\nincluding hourly water levels, tidal datums and predictions, and trends in sea level over time.\n\nA list of CO-OPS stations can be retrieved with the `coops_stations()` function.\n\n```python\nfrom stormevents.coops import coops_stations\n\ncoops_stations()\n```\n\n```\n        nws_id                         name state        status                                            removed                      geometry\nnos_id\n1600012  46125                    QREB buoy              active                                               <NA>    POINT (122.62500 37.75000)\n1619910  SNDP5  Sand Island, Midway Islands              active                                               <NA>   POINT (-177.37500 28.21875)\n1630000  APRP7            Apra Harbor, Guam              active                                               <NA>    POINT (144.62500 13.44531)\n1631428  PGBP7               Pago Bay, Guam              active                                               <NA>    POINT (144.75000 13.42969)\n1770000  NSTP6    Pago Pago, American Samoa              active                                               <NA>  POINT (-170.75000 -14.27344)\n...        ...                          ...   ...           ...                                                ...                           ...\n8423898  FTPN3                   Fort Point    NH  discontinued  2020-04-13 00:00:00,2014-08-05 00:00:00,2012-0...    POINT (-70.68750 43.06250)\n8726667  MCYF1           Mckay Bay Entrance    FL  discontinued  2020-05-20 00:00:00,2019-03-08 00:00:00,2017-0...    POINT (-82.43750 27.90625)\n8772447  FCGT2                     Freeport    TX  discontinued  2020-05-24 18:45:00,2018-10-10 21:50:00,2018-1...    POINT (-95.31250 28.93750)\n9087079  GBWW3                    Green Bay    WI  discontinued  2020-10-28 13:00:00,2007-08-06 23:59:00,2007-0...    POINT (-88.00000 44.53125)\n8770570  SBPT2            Sabine Pass North    TX  discontinued  2021-01-18 00:00:00,2020-09-30 15:45:00,2020-0...    POINT (-93.87500 29.73438)\n\n[435 rows x 6 columns]\n```\n\nAdditionally, you can use a Shapely `Polygon` or `MultiPolygon` to constrain the stations query to a specific region:\n\n```python\nfrom shapely.geometry import Polygon\nfrom stormevents.coops import coops_stations_within_region\n\nregion = Polygon(...)\n\ncoops_stations_within_region(region=region)\n```\n\n```\n        nws_id                               name state removed                    geometry\nnos_id                                                                                     \n8651370  DUKN7                               Duck    NC     NaT  POINT (-75.75000 36.18750)\n8652587  ORIN7                Oregon Inlet Marina    NC     NaT  POINT (-75.56250 35.78125)\n8654467  HCGN7              USCG Station Hatteras    NC     NaT  POINT (-75.68750 35.21875)\n8656483  BFTN7          Beaufort, Duke Marine Lab    NC     NaT  POINT (-76.68750 34.71875)\n8658120  WLON7                         Wilmington    NC     NaT  POINT (-77.93750 34.21875)\n8658163  JMPN7                 Wrightsville Beach    NC     NaT  POINT (-77.81250 34.21875)\n8661070  MROS1                    Springmaid Pier    SC     NaT  POINT (-78.93750 33.65625)\n8662245  NITS1   Oyster Landing (N Inlet Estuary)    SC     NaT  POINT (-79.18750 33.34375)\n8665530  CHTS1  Charleston, Cooper River Entrance    SC     NaT  POINT (-79.93750 32.78125)\n8670870  FPKG1                       Fort Pulaski    GA     NaT  POINT (-80.87500 32.03125)\n```\n\n##### retrieve CO-OPS data product for a specific station\n\n```python\nfrom datetime import datetime\nfrom stormevents.coops import COOPS_Station\n\nstation = COOPS_Station(8632200)\nstation.product('water_level', start_date=datetime(2018, 9, 13), end_date=datetime(2018, 9, 16, 12))\n```\n\n```\n<xarray.Dataset>\nDimensions:  (nos_id: 1, t: 841)\nCoordinates:\n  * nos_id   (nos_id) int64 8632200\n  * t        (t) datetime64[ns] 2018-09-13 ... 2018-09-16T12:00:00\n    nws_id   (nos_id) <U5 'KPTV2'\n    x        (nos_id) float64 -76.0\n    y        (nos_id) float64 37.16\nData variables:\n    v        (nos_id, t) float32 1.67 1.694 1.73 1.751 ... 1.597 1.607 1.605\n    s        (nos_id, t) float32 0.026 0.027 0.034 0.03 ... 0.018 0.019 0.021\n    f        (nos_id, t) object '0,0,0,0' '0,0,0,0' ... '0,0,0,0' '0,0,0,0'\n    q        (nos_id, t) object 'v' 'v' 'v' 'v' 'v' 'v' ... 'v' 'v' 'v' 'v' 'v'\n```\n\n##### retrieve CO-OPS data product from within a region and time interval\n\nTo retrieve data, you must provide three things:\n\n1. the **data product** of interest; one of\n    - `water_level` - Preliminary or verified water levels, depending on availability.\n    - `air_temperature` - Air temperature as measured at the station.\n    - `water_temperature` - Water temperature as measured at the station.\n    - `wind` - Wind speed, direction, and gusts as measured at the station.\n    - `air_pressure` - Barometric pressure as measured at the station.\n    - `air_gap` - Air Gap (distance between a bridge and the water's surface) at the station.\n    - `conductivity` - The water's conductivity as measured at the station.\n    - `visibility` - Visibility from the station's visibility sensor. A measure of atmospheric clarity.\n    - `humidity` - Relative humidity as measured at the station.\n    - `salinity` - Salinity and specific gravity data for the station.\n    - `hourly_height` - Verified hourly height water level data for the station.\n    - `high_low` - Verified high/low water level data for the station.\n    - `daily_mean` - Verified daily mean water level data for the station.\n    - `monthly_mean` - Verified monthly mean water level data for the station.\n    - `one_minute_water_level`  One minute water level data for the station.\n    - `predictions` - 6 minute predictions water level data for the station.*\n    - `datums` - datums data for the stations.\n    - `currents` - Currents data for currents stations.\n    - `currents_predictions` - Currents predictions data for currents predictions stations.\n2. a **region** within which to retrieve the data product\n3. a **time interval** within which to retrieve the data product\n\n```python\nfrom datetime import datetime, timedelta\n\nfrom shapely.geometry import Polygon\nfrom stormevents.coops import coops_product_within_region\n\npolygon = Polygon(...)\n\ncoops_product_within_region(\n    'water_level',\n    region=polygon,\n    start_date=datetime.now() - timedelta(hours=1),\n)\n```\n\n```\n<xarray.Dataset>\nDimensions:  (nos_id: 10, t: 11)\nCoordinates:\n  * nos_id   (nos_id) int64 8651370 8652587 8654467 ... 8662245 8665530 8670870\n  * t        (t) datetime64[ns] 2022-03-08T14:48:00 ... 2022-03-08T15:48:00\n    nws_id   (nos_id) <U5 'DUKN7' 'ORIN7' 'HCGN7' ... 'NITS1' 'CHTS1' 'FPKG1'\n    x        (nos_id) float64 -75.75 -75.56 -75.69 ... -79.19 -79.94 -80.88\n    y        (nos_id) float64 36.19 35.78 35.22 34.72 ... 33.34 32.78 32.03\nData variables:\n    v        (nos_id, t) float32 6.256 6.304 6.361 6.375 ... 2.633 2.659 2.686\n    s        (nos_id, t) float32 0.107 0.097 0.127 0.122 ... 0.005 0.004 0.004\n    f        (nos_id, t) object '1,0,0,0' '1,0,0,0' ... '1,0,0,0' '1,0,0,0'\n    q        (nos_id, t) object 'p' 'p' 'p' 'p' 'p' 'p' ... 'p' 'p' 'p' 'p' 'p'\n```\n\n### retrieve data surrounding a specific storm\n\nThe `StormEvent` class provides an interface to retrieve data within the time interval and spatial bounds of a specific storm\nevent.\n\nYou can create a new `StormEvent` object from a storm name and year,\n\n```python\nfrom stormevents import StormEvent\n\nStormEvent('FLORENCE', 2018)\n```\n\n```\nStormEvent(name='FLORENCE', year=2018, start_date=Timestamp('2018-08-30 06:00:00'), end_date=Timestamp('2018-09-18 12:00:00'))\n```\n\nor from a storm NHC code,\n\n```python\nfrom stormevents import StormEvent\n\nStormEvent.from_nhc_code('EP172016')\n```\n\n```\nStormEvent(name='PAINE', year=2016, start_date=Timestamp('2016-09-18 00:00:00'), end_date=Timestamp('2016-09-21 12:00:00'))\n```\n\nor from a USGS flood event ID.\n\n```python\nfrom stormevents import StormEvent\n\nStormEvent.from_usgs_id(310)\n```\n\n```\nStormEvent(name='HENRI', year=2021, start_date=Timestamp('2021-08-20 18:00:00'), end_date=Timestamp('2021-08-24 12:00:00'))\n```\n\nTo constrain the time interval, you can provide an absolute time range,\n\n```python\nfrom stormevents import StormEvent\nfrom datetime import datetime\n\nStormEvent('paine', 2016, start_date='2016-09-19', end_date=datetime(2016, 9, 19, 12))\n```\n\n```\nStormEvent(name='PAINE', year=2016, start_date=datetime.datetime(2016, 9, 19, 0, 0), end_date=datetime.datetime(2016, 9, 19, 12, 0))\n```\n\n```python\nfrom stormevents import StormEvent\nfrom datetime import datetime\n\nStormEvent('paine', 2016, end_date=datetime(2016, 9, 19, 12))\n```\n\n```\nStormEvent(name='PAINE', year=2016, start_date=Timestamp('2016-09-18 00:00:00'), end_date=datetime.datetime(2016, 9, 19, 12, 0))\n```\n\nor, alternatively, you can provide relative time deltas, which will be interpreted compared to the absolute time interval\nprovided by the NHC.\n\n```python\nfrom stormevents import StormEvent\nfrom datetime import timedelta\n\nStormEvent('florence', 2018, start_date=timedelta(days=2))  # <- start 2 days after NHC start time\n```\n\n```\nStormEvent(name='FLORENCE', year=2018, start_date=Timestamp('2018-09-01 06:00:00'), end_date=Timestamp('2018-09-18 12:00:00'))\n```\n\n```python\nfrom stormevents import StormEvent\nfrom datetime import timedelta\n\nStormEvent(\n    'henri',\n    2021,\n    start_date=timedelta(days=-3),  # <- start 3 days before NHC end time\n    end_date=timedelta(days=-2),  # <- end 2 days before NHC end time\n)\n```\n\n```\nStormEvent(name='HENRI', year=2021, start_date=Timestamp('2021-08-21 12:00:00'), end_date=Timestamp('2021-08-22 12:00:00'))\n```\n\n```python\nfrom stormevents import StormEvent\nfrom datetime import timedelta\n\nStormEvent('ida', 2021, end_date=timedelta(days=2))  # <- end 2 days after NHC start time \n```\n\n```\nStormEvent(name='IDA', year=2021, start_date=Timestamp('2021-08-27 18:00:00'), end_date=Timestamp('2021-08-29 18:00:00'))\n```\n\n#### retrieve data for a storm\n\nThe following methods are very similar to the data getter functions detailed above. However, these methods are tied to a\nspecific storm event, and will focus on retrieving data within the spatial region and time interval of their specific storm\nevent.\n\n##### track data from the National Hurricane Center (NHC)\n\n```python\nfrom stormevents import StormEvent\n\nstorm = StormEvent('florence', 2018)\nstorm.track()\n```\n\n```\nVortexTrack('AL062018', Timestamp('2018-08-30 06:00:00'), Timestamp('2018-09-18 12:00:00'), <ATCF_FileDeck.BEST: 'b'>, <ATCF_Mode.HISTORICAL: 'ARCHIVE'>, [<ATCF_Advisory.BEST: 'BEST'>], None)\n```\n\n```python\nfrom stormevents import StormEvent\n\nstorm = StormEvent('florence', 2018)\nstorm.track(file_deck='a')\n```\n\n```\nVortexTrack('AL062018', Timestamp('2018-08-30 06:00:00'), Timestamp('2018-09-18 12:00:00'), <ATCF_FileDeck.ADVISORY: 'a'>, <ATCF_Mode.HISTORICAL: 'ARCHIVE'>, ['OFCL', 'OFCP', 'HMON', 'CARQ', 'HWRF'], None)\n```\n\n##### high-water mark (HWM) surveys provided by the United States Geological Survey (USGS)\n\n```python\nfrom stormevents import StormEvent\n\nstorm = StormEvent('florence', 2018)\nflood = storm.flood_event\nflood.high_water_marks()\n```\n\n```\n         latitude  longitude          eventName  ... siteZone peak_summary_id                    geometry\nhwm_id                                           ...                                                     \n33496   37.298440 -80.007750  Florence Sep 2018  ...      NaN             NaN  POINT (-80.00775 37.29844)\n33497   33.699720 -78.936940  Florence Sep 2018  ...      NaN             NaN  POINT (-78.93694 33.69972)\n33498   33.758610 -78.792780  Florence Sep 2018  ...      NaN             NaN  POINT (-78.79278 33.75861)\n33499   33.641389 -78.947778  Florence Sep 2018  ...                      NaN  POINT (-78.94778 33.64139)\n33500   33.602500 -78.973889  Florence Sep 2018  ...                      NaN  POINT (-78.97389 33.60250)\n...           ...        ...                ...  ...      ...             ...                         ...\n34872   35.534641 -77.038183  Florence Sep 2018  ...      NaN             NaN  POINT (-77.03818 35.53464)\n34873   35.125000 -77.050044  Florence Sep 2018  ...      NaN             NaN  POINT (-77.05004 35.12500)\n34874   35.917467 -76.254367  Florence Sep 2018  ...      NaN             NaN  POINT (-76.25437 35.91747)\n34875   35.111000 -77.037851  Florence Sep 2018  ...      NaN             NaN  POINT (-77.03785 35.11100)\n34876   35.301135 -77.264727  Florence Sep 2018  ...      NaN             NaN  POINT (-77.26473 35.30114)\n\n[644 rows x 53 columns]\n```\n\n##### products from the Center for Operational Oceanographic Products and Services (CO-OPS)\n\n```python\nfrom stormevents import StormEvent\n\nstorm = StormEvent('florence', 2018)\nstorm.coops_product_within_isotach('water_level', wind_speed=34, start_date='2018-09-12 14:03:00', end_date='2018-09-14')\n```\n\n```\n<xarray.Dataset>\nDimensions:  (nos_id: 7, t: 340)\nCoordinates:\n  * nos_id   (nos_id) int64 8651370 8652587 8654467 ... 8658120 8658163 8661070\n  * t        (t) datetime64[ns] 2018-09-12T14:06:00 ... 2018-09-14\n    nws_id   (nos_id) <U5 'DUKN7' 'ORIN7' 'HCGN7' ... 'WLON7' 'JMPN7' 'MROS1'\n    x        (nos_id) float64 -75.75 -75.56 -75.69 -76.69 -77.94 -77.81 -78.94\n    y        (nos_id) float64 36.19 35.78 35.22 34.72 34.22 34.22 33.66\nData variables:\n    v        (nos_id, t) float32 7.181 7.199 7.144 7.156 ... 9.6 9.634 9.686\n    s        (nos_id, t) float32 0.317 0.36 0.31 0.318 ... 0.049 0.047 0.054\n    f        (nos_id, t) object '0,0,0,0' '0,0,0,0' ... '0,0,0,0' '0,0,0,0'\n    q        (nos_id, t) object 'v' 'v' 'v' 'v' 'v' 'v' ... 'v' 'v' 'v' 'v' 'v'\n```\n\n## Related Projects\n\n- `searvey` - https://github.com/oceanmodeling/searvey\n- `pyStorms` - https://github.com/brey/pyStorms\n- `tropycal` - https://tropycal.github.io/tropycal/index.html\n- `pyoos` - https://github.com/ioos/pyoos\n- `csdllib` - https://github.com/noaa-ocs-modeling/csdllib\n- `pyPoseidon` - https://github.com/ec-jrc/pyPoseidon\n- `Thalassa` - https://github.com/ec-jrc/Thalassa\n- `adcircpy` - https://github.com/noaa-ocs-modeling/adcircpy\n\n## Acknowledgements\n\nOriginal methodology for retrieving NHC storm tracks and CO-OPS tidal data was written\nby [@jreniel](https://github.com/jreniel) for [`adcircpy`](https://github.com/noaa-ocs-modeling/adcircpy).\n\nOriginal methodology for retrieving USGS high-water mark surveys and CO-OPS tidal station metadata was written\nby [@moghimis](https://github.com/moghimis) and [@grapesh](https://github.com/grapesh)\nfor [`csdllib`](https://github.com/noaa-ocs-modeling/csdllib).\n",
    'author': 'Zach Burnett',
    'author_email': 'zachary.r.burnett@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/oceanmodeling/StormEvents.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
