from howitzer.util.trading import *

from datetime import datetime, timezone, tzinfo
import unittest
import dateutil.parser as dateparse


class TestFoltillaTradingCandle(unittest.TestCase):
    def setUp(self):
        self.test_candle_01_inputs = [dateparse.parse("2021-01-01T00:00:00.000Z").timestamp(), 49, 55, 50, 51, 10000]
        self.test_candle_01 = Candle(self.test_candle_01_inputs)

    def test_createSimpleCandle_time(self):
        self.assertEqual(self.test_candle_01.time.timestamp(), 1609459200)

    def test_createSimpleCandle_getTImeString(self):
        self.assertEqual(self.test_candle_01.getTimeStr(),
                         "2021-01-01T00:00:00.000Z")

    def test_createSimpleCandle_open(self):
        self.assertEqual(self.test_candle_01.open,
                         self.test_candle_01_inputs[3])

    def test_createSimpleCandle_high(self):
        self.assertEqual(self.test_candle_01.high,
                         self.test_candle_01_inputs[2])

    def test_createSimpleCandle_low(self):
        self.assertEqual(self.test_candle_01.low,
                         self.test_candle_01_inputs[1])

    def test_createSimpleCandle_close(self):
        self.assertEqual(self.test_candle_01.close,
                         self.test_candle_01_inputs[4])

    def test_createSimpleCandle_volume(self):
        self.assertEqual(self.test_candle_01.volume,
                         self.test_candle_01_inputs[5])

    def test_createSimpleCandle_percent(self):
        self.assertEqual(self.test_candle_01.percent,  2)

    def test_createSimpleCandle_diff(self):
        self.assertEqual(self.test_candle_01.diff,  1)

    def test_createSimpleCandle_range(self):
        self.assertEqual(self.test_candle_01.range,  6)

    def test_createSimpleCandle_green(self):
        self.assertEqual(self.test_candle_01.green,  True)

    def test_createSimpleCandle_red(self):
        self.assertEqual(self.test_candle_01.red,  False)

    def test_createSimpleCandle_head(self):
        self.assertEqual(self.test_candle_01.head,  4)

    def test_createSimpleCandle_tail(self):
        self.assertEqual(self.test_candle_01.tail,  1)

    def test_createSimpleCandle_body(self):
        self.assertEqual(self.test_candle_01.body,  1)

    def test_createSimpleCandle_twap(self):
        self.assertAlmostEqual(self.test_candle_01.twap,  51.6666667)

    def test_createSimpleCandle_tr_empty(self):
        self.assertEqual(self.test_candle_01.tr,  6)

    def test_createSimpleCandle_twap_with_data(self):
        c = Candle(self.test_candle_01_inputs, previousClose=48)
        self.assertEqual(c.tr,  7)


class TestFoltillaTradingChart(unittest.TestCase):
    def setUp(self):
        self.test_chart_input = [[dateparse.parse("2021-01-01T05:00:00.000Z").timestamp(), 49, 55, 50, 51, 10000]]
        self.test_chart = Chart(self.test_chart_input)

    def test_chartMakesSingleCandle(self):
        self.assertEqual(len(self.test_chart.candles), 1)


class TestFoltillaTradingIndicators(unittest.TestCase):
    def setUp(self):
        self.test_chart_input = [{"time": "2021-02-07T00:00:00.000Z",
                                  "open": 39266.1, "high": 39737.83, "low": 37371.35, "close": 38871.42, "volume": 20297.87256102},
                                 {"time": "2021-02-08T00:00:00.000Z",
                                  "open": 38871.43, "high": 46750, "low": 38057.01, "close": 46448.1, "volume": 47934.0103883},
                                 {"time": "2021-02-09T00:00:00.000Z",
                                  "open": 46453.66, "high": 48200, "low": 45020, "close": 46514.73, "volume": 36418.79365116},
                                 {"time": "2021-02-10T00:00:00.000Z",
                                  "open": 46514.73, "high": 47349.99, "low": 43706.86, "close": 44850, "volume": 25818.01157857},
                                 {"time": "2021-02-11T00:00:00.000Z",
                                  "open": 44850, "high": 48690, "low": 44011, "close": 47997.91, "volume": 26222.60477552},
                                 {"time": "2021-02-12T00:00:00.000Z",
                                  "open": 47993.6, "high": 48912, "low": 46300, "close": 47408.34, "volume": 23503.15583529},
                                 {"time": "2021-02-13T00:00:00.000Z",
                                  "open": 47408.34, "high": 48199, "low": 46300, "close": 47232.45, "volume": 12020.8773325},
                                 {"time": "2021-02-14T00:00:00.000Z",
                                  "open": 47240.75, "high": 49700, "low": 47100.13, "close": 48680.69, "volume": 15146.55244091},
                                 {"time": "2021-02-15T00:00:00.000Z",
                                  "open": 48667.23, "high": 49093.44, "low": 45768.46, "close": 47936.37, "volume": 19432.15278111},
                                 {"time": "2021-02-16T00:00:00.000Z",
                                  "open": 47935.66, "high": 50645, "low": 47051.22, "close": 49158.71, "volume": 23053.20561135},
                                 {"time": "2021-02-17T00:00:00.000Z",
                                  "open": 49141.43, "high": 52640.35, "low": 48942.87, "close": 52170.01, "volume": 25675.171441},
                                 {"time": "2021-02-18T00:00:00.000Z",
                                  "open": 52170.01, "high": 52549.99, "low": 50875, "close": 51601.73, "volume": 15689.40275173},
                                 {"time": "2021-02-19T00:00:00.000Z",
                                  "open": 51596, "high": 56370.05, "low": 50725, "close": 55983.56, "volume": 27961.48197104},
                                 {"time": "2021-02-20T00:00:00.000Z",
                                  "open": 55983.57, "high": 57563.57, "low": 54000.1, "close": 55923.52, "volume": 22137.82194494},
                                 {"time": "2021-02-21T00:00:00.000Z",
                                  "open": 55921.44, "high": 58367, "low": 55531.23, "close": 57489.16, "volume": 13530.43444443},
                                 {"time": "2021-02-22T00:00:00.000Z",
                                  "open": 57489.16, "high": 57577.69, "low": 46616, "close": 54142.13, "volume": 49449.16575779},
                                 {"time": "2021-02-23T00:00:00.000Z",
                                  "open": 54125.67, "high": 54206.22, "low": 44888.08, "close": 48899.99, "volume": 72300.585728},
                                 {"time": "2021-02-24T00:00:00.000Z",
                                  "open": 48899.99, "high": 51415.41, "low": 47002, "close": 49737.82, "volume": 30732.14949499},
                                 {"time": "2021-02-25T00:00:00.000Z",
                                  "open": 49737.81, "high": 52107.83, "low": 46700, "close": 47058.48, "volume": 28011.20414657},
                                 {"time": "2021-02-26T00:00:00.000Z",
                                  "open": 47063.9, "high": 48464.64, "low": 44150, "close": 46326.2, "volume": 36048.6181927},
                                 {"time": "2021-02-27T00:00:00.000Z",
                                  "open": 46319.79, "high": 48356.86, "low": 45055, "close": 46180.75, "volume": 16290.28071728},
                                 {"time": "2021-02-28T00:00:00.000Z",
                                  "open": 46169.92, "high": 46656, "low": 43016, "close": 45231.75, "volume": 22745.60600179},
                                 {"time": "2021-03-01T00:00:00.000Z",
                                  "open": 45231.74, "high": 49829, "low": 45042.13, "close": 49639.4, "volume": 25068.25043716},
                                 {"time": "2021-03-02T00:00:00.000Z",
                                  "open": 49635.3, "high": 50250, "low": 47075.11, "close": 48511.6, "volume": 17063.56302004},
                                 {"time": "2021-03-03T00:00:00.000Z",
                                  "open": 48502.18, "high": 52666, "low": 48161.75, "close": 50360, "volume": 21301.47723933},
                                 {"time": "2021-03-04T00:00:00.000Z",
                                  "open": 50370.33, "high": 51798.33, "low": 47502, "close": 48368.52, "volume": 23386.97957577},
                                 {"time": "2021-03-05T00:00:00.000Z",
                                  "open": 48370.97, "high": 49473.99, "low": 46219.32, "close": 48769.47, "volume": 25468.18029576},
                                 {"time": "2021-03-06T00:00:00.000Z",
                                  "open": 48766.01, "high": 49210, "low": 47077, "close": 48909.84, "volume": 11877.83755909},
                                 {"time": "2021-03-07T00:00:00.000Z",
                                  "open": 48909.84, "high": 51459, "low": 48909.83, "close": 50978.61, "volume": 13965.15473114},
                                 {"time": "2021-03-08T00:00:00.000Z",
                                  "open": 50976.22, "high": 52425, "low": 49328.62, "close": 52415.23, "volume": 18856.28684439},
                                 {"time": "2021-03-09T00:00:00.000Z",
                                  "open": 52413.17, "high": 54936, "low": 51845.01, "close": 54916.38, "volume": 21177.14953983},
                                 {"time": "2021-03-10T00:00:00.000Z",
                                  "open": 54921.61, "high": 57402.14, "low": 53025, "close": 55890.69, "volume": 28326.07770941},
                                 {"time": "2021-03-11T00:00:00.000Z",
                                  "open": 55888.7, "high": 58113, "low": 54283, "close": 57815.47, "volume": 24369.21216789},
                                 {"time": "2021-03-12T00:00:00.000Z",
                                  "open": 57820.15, "high": 58065.92, "low": 55050, "close": 57225.91, "volume": 21760.53824103},
                                 {"time": "2021-03-13T00:00:00.000Z",
                                  "open": 57220.84, "high": 61788.45, "low": 56083.74, "close": 61178.5, "volume": 22301.28588901},
                                 {"time": "2021-03-14T00:00:00.000Z",
                                  "open": 61181.38, "high": 61680, "low": 58950, "close": 58972.7, "volume": 13581.96759058},
                                 {"time": "2021-03-15T00:00:00.000Z",
                                  "open": 58972.71, "high": 60601.07, "low": 54568, "close": 55636.15, "volume": 31651.39184131},
                                 {"time": "2021-03-16T00:00:00.000Z",
                                  "open": 55629.08, "high": 56970, "low": 53221, "close": 56927.11, "volume": 24432.37878722},
                                 {"time": "2021-03-17T00:00:00.000Z",
                                  "open": 56924.26, "high": 58996.38, "low": 54138.01, "close": 58925.54, "volume": 22687.57433043},
                                 {"time": "2021-03-18T00:00:00.000Z",
                                  "open": 58925.54, "high": 60100, "low": 56996.15, "close": 57644.95, "volume": 21459.4962399},
                                 {"time": "2021-03-19T00:00:00.000Z",
                                  "open": 57644.95, "high": 59461.99, "low": 56260, "close": 58039.66, "volume": 16429.629743},
                                 {"time": "2021-03-20T00:00:00.000Z",
                                  "open": 58026.91, "high": 59928, "low": 57827.97, "close": 58116.7, "volume": 9725.28123717},
                                 {"time": "2021-03-21T00:00:00.000Z",
                                  "open": 58126.34, "high": 58651.22, "low": 55543.89, "close": 57363.32, "volume": 11841.63877891},
                                 {"time": "2021-03-22T00:00:00.000Z",
                                  "open": 57377.29, "high": 58421.68, "low": 53739.5, "close": 54110.07, "volume": 19512.15647151},
                                 {"time": "2021-03-23T00:00:00.000Z",
                                  "open": 54111.49, "high": 55858, "low": 53000, "close": 54345.54, "volume": 19221.75674},
                                 {"time": "2021-03-24T00:00:00.000Z",
                                  "open": 54333.27, "high": 57209.97, "low": 51673.82, "close": 52276.16, "volume": 23775.21362623},
                                 {"time": "2021-03-25T00:00:00.000Z",
                                  "open": 52259.69, "high": 53238, "low": 50305, "close": 51325.01, "volume": 29642.04415588},
                                 {"time": "2021-03-26T00:00:00.000Z",
                                  "open": 51320.76, "high": 55125.13, "low": 51250, "close": 55072.44, "volume": 17843.00738205},
                                 {"time": "2021-03-27T00:00:00.000Z",
                                  "open": 55080.9, "high": 56640, "low": 53966, "close": 55856.53, "volume": 11333.33349523},
                                 {"time": "2021-03-28T00:00:00.000Z",
                                  "open": 55856.54, "high": 56587.08, "low": 54701, "close": 55778.82, "volume": 9235.80302347},
                                 {"time": "2021-03-29T00:00:00.000Z",
                                  "open": 55781.22, "high": 58400, "low": 54900, "close": 57613.48, "volume": 17552.36885101},
                                 {"time": "2021-03-30T00:00:00.000Z",
                                  "open": 57613.76, "high": 59397.48, "low": 57000, "close": 58786.46, "volume": 15105.07669026},
                                 {"time": "2021-03-31T00:00:00.000Z",
                                  "open": 58786.46, "high": 59800, "low": 56873.8, "close": 58800, "volume": 17376.62672814},
                                 {"time": "2021-04-01T00:00:00.000Z",
                                  "open": 58800.01, "high": 59474.94, "low": 57930, "close": 58726.48, "volume": 11096.33004907},
                                 {"time": "2021-04-02T00:00:00.000Z",
                                  "open": 58726.47, "high": 60055.02, "low": 58441.88, "close": 58981.04, "volume": 10966.93017846},
                                 {"time": "2021-04-03T00:00:00.000Z",
                                  "open": 58981.04, "high": 59752, "low": 56943, "close": 57094.34, "volume": 8405.26400491},
                                 {"time": "2021-04-04T00:00:00.000Z",
                                  "open": 57094.34, "high": 58500, "low": 56478.53, "close": 58215.94, "volume": 6301.43187883},
                                 {"time": "2021-04-05T00:00:00.000Z",
                                  "open": 58216.73, "high": 59251.76, "low": 56817.64, "close": 59123.02, "volume": 9204.86323941},
                                 {"time": "2021-04-06T00:00:00.000Z",
                                  "open": 59134.07, "high": 59475, "low": 57333.33, "close": 58019.98, "volume": 10354.33014603},
                                 {"time": "2021-04-07T00:00:00.000Z",
                                  "open": 58021.67, "high": 58630, "low": 55400, "close": 55955.75, "volume": 17787.29445084},
                                 {"time": "2021-04-08T00:00:00.000Z",
                                  "open": 55962.67, "high": 58152, "low": 55700, "close": 58083.1, "volume": 8073.59731731},
                                 {"time": "2021-04-09T00:00:00.000Z",
                                  "open": 58083.1, "high": 58869.69, "low": 57670.51, "close": 58092.68, "volume": 8094.74629652},
                                 {"time": "2021-04-10T00:00:00.000Z",
                                  "open": 58092.68, "high": 61218.97, "low": 57875.41, "close": 59778.6, "volume": 12817.81106734},
                                 {"time": "2021-04-11T00:00:00.000Z",
                                  "open": 59778.59, "high": 60658.89, "low": 59177.06, "close": 59985.26, "volume": 7278.96516665},
                                 {"time": "2021-04-12T00:00:00.000Z",
                                  "open": 59983.66, "high": 61199, "low": 59369, "close": 59839.82, "volume": 11467.72752999},
                                 {"time": "2021-04-13T00:00:00.000Z",
                                  "open": 59836.88, "high": 63774.39, "low": 59799.01, "close": 63588.22, "volume": 17897.76603088},
                                 {"time": "2021-04-14T00:00:00.000Z",
                                  "open": 63588.22, "high": 64899, "low": 61277.91, "close": 62971.8, "volume": 22570.84113444},
                                 {"time": "2021-04-15T00:00:00.000Z",
                                  "open": 62971.8, "high": 63831.82, "low": 62036.73, "close": 63229.04, "volume": 11209.4505281},
                                 {"time": "2021-04-16T00:00:00.000Z",
                                  "open": 63229.04, "high": 63604.34, "low": 60048.43, "close": 61427.27, "volume": 19867.41322269},
                                 {"time": "2021-04-17T00:00:00.000Z",
                                  "open": 61427.27, "high": 62572.48, "low": 59700, "close": 60058.86, "volume": 10847.76785762},
                                 {"time": "2021-04-18T00:00:00.000Z",
                                  "open": 60067.2, "high": 60437.97, "low": 51300, "close": 56274.41, "volume": 36891.70959047},
                                 {"time": "2021-04-19T00:00:00.000Z",
                                  "open": 56273.65, "high": 57600, "low": 54187.85, "close": 55696.83, "volume": 16838.39171629},
                                 {"time": "2021-04-20T00:00:00.000Z",
                                  "open": 55696.83, "high": 57111, "low": 53430.01, "close": 56499.29, "volume": 18608.27983583},
                                 {"time": "2021-04-21T00:00:00.000Z",
                                  "open": 56477.67, "high": 56810.56, "low": 53620.91, "close": 53800.86, "volume": 15584.08398293},
                                 {"time": "2021-04-22T00:00:00.000Z",
                                  "open": 53795.62, "high": 55469.98, "low": 50400, "close": 51701.59, "volume": 30427.89153907},
                                 {"time": "2021-04-23T00:00:00.000Z",
                                  "open": 51695.98, "high": 52130.58, "low": 47464.65, "close": 51187.27, "volume": 38794.85818376},
                                 {"time": "2021-04-24T00:00:00.000Z",
                                  "open": 51187.75, "high": 51260.61, "low": 48726.87, "close": 50089.15, "volume": 11839.14355893},
                                 {"time": "2021-04-25T00:00:00.000Z",
                                  "open": 50101.42, "high": 50591.38, "low": 47044.01, "close": 49121, "volume": 14633.51206544},
                                 {"time": "2021-04-26T00:00:00.000Z",
                                  "open": 49121, "high": 54400, "low": 48817.62, "close": 54053.6, "volume": 18005.22399425},
                                 {"time": "2021-04-27T00:00:00.000Z",
                                  "open": 54047.8, "high": 55509.39, "low": 53321, "close": 55069.62, "volume": 13957.08649512},
                                 {"time": "2021-04-28T00:00:00.000Z",
                                  "open": 55069.61, "high": 56476.17, "low": 53887, "close": 54894.03, "volume": 16484.33677655},
                                 {"time": "2021-04-29T00:00:00.000Z",
                                  "open": 54889.81, "high": 55226.86, "low": 52369.61, "close": 53580, "volume": 14592.31688772},
                                 {"time": "2021-04-30T00:00:00.000Z",
                                  "open": 53580, "high": 58075.01, "low": 53068.43, "close": 57798.77, "volume": 16536.33288912},
                                 {"time": "2021-05-01T00:00:00.000Z",
                                  "open": 57798.77, "high": 58550, "low": 57050.94, "close": 57859.28, "volume": 10787.04933216},
                                 {"time": "2021-05-02T00:00:00.000Z",
                                  "open": 57859.28, "high": 57972.26, "low": 56072, "close": 56625.2, "volume": 7647.62578087},
                                 {"time": "2021-05-03T00:00:00.000Z",
                                  "open": 56625.21, "high": 58986, "low": 56500, "close": 57212.73, "volume": 15165.00648795},
                                 {"time": "2021-05-04T00:00:00.000Z",
                                  "open": 57212.73, "high": 57247.24, "low": 53057.7, "close": 53241.92, "volume": 23492.0586849},
                                 {"time": "2021-05-05T00:00:00.000Z",
                                  "open": 53241.91, "high": 57980.86, "low": 52913.02, "close": 57515.69, "volume": 22918.90458572},
                                 {"time": "2021-05-06T00:00:00.000Z",
                                  "open": 57515.69, "high": 58400, "low": 55288, "close": 56440.66, "volume": 18244.13605211},
                                 {"time": "2021-05-07T00:00:00.000Z",
                                  "open": 56444.82, "high": 58735.95, "low": 55300, "close": 57380.39, "volume": 18323.8958022},
                                 {"time": "2021-05-08T00:00:00.000Z",
                                  "open": 57380.39, "high": 59559.84, "low": 56980.01, "close": 58958.05, "volume": 15374.94442964},
                                 {"time": "2021-05-09T00:00:00.000Z",
                                  "open": 58958.06, "high": 59300.52, "low": 56265.3, "close": 58312.57, "volume": 17169.72318523},
                                 {"time": "2021-05-10T00:00:00.000Z",
                                  "open": 58313.84, "high": 59592.2, "low": 53600, "close": 55866.41, "volume": 24647.23540718},
                                 {"time": "2021-05-11T00:00:00.000Z",
                                  "open": 55866.38, "high": 56962.07, "low": 54511.48, "close": 56753.19, "volume": 15446.63591676},
                                 {"time": "2021-05-12T00:00:00.000Z",
                                  "open": 56753.19, "high": 58041, "low": 48500, "close": 49498.77, "volume": 29660.84273553},
                                 {"time": "2021-05-13T00:00:00.000Z",
                                  "open": 49498.76, "high": 52500, "low": 46000, "close": 49690.11, "volume": 42250.08740789},
                                 {"time": "2021-05-14T00:00:00.000Z",
                                  "open": 49682.28, "high": 51569.56, "low": 48895, "close": 49893.48, "volume": 21987.79362448},
                                 {"time": "2021-05-15T00:00:00.000Z",
                                  "open": 49893.48, "high": 50730.58, "low": 46573, "close": 46775.51, "volume": 21936.11454712},
                                 {"time": "2021-05-16T00:00:00.000Z",
                                  "open": 46771.01, "high": 49808.28, "low": 43825, "close": 46450.79, "volume": 30655.33064729},
                                 {"time": "2021-05-17T00:00:00.000Z",
                                  "open": 46450.78, "high": 46646.15, "low": 42101, "close": 43580.5, "volume": 43660.22208427},
                                 {"time": "2021-05-18T00:00:00.000Z",
                                  "open": 43571.89, "high": 45851, "low": 42300, "close": 42857.15, "volume": 28210.30024018},
                                 {"time": "2021-05-19T00:00:00.000Z",
                                  "open": 42865.05, "high": 43591.7, "low": 30000, "close": 36731.75, "volume": 109537.24233282},
                                 {"time": "2021-05-20T00:00:00.000Z",
                                  "open": 36735.44, "high": 42605, "low": 35000, "close": 40623.33, "volume": 51445.79735691},
                                 {"time": "2021-05-21T00:00:00.000Z",
                                  "open": 40623.24, "high": 42275.99, "low": 33501.54, "close": 37340.77, "volume": 55961.90006046},
                                 {"time": "2021-05-22T00:00:00.000Z",
                                  "open": 37333.09, "high": 38889.76, "low": 35257.52, "close": 37476.83, "volume": 29239.02738254},
                                 {"time": "2021-05-23T00:00:00.000Z",
                                  "open": 37476.84, "high": 38319.36, "low": 31111, "close": 34758.67, "volume": 53433.30296532},
                                 {"time": "2021-05-24T00:00:00.000Z",
                                  "open": 34758.88, "high": 39966.87, "low": 34407.17, "close": 38878.56, "volume": 44419.45827713},
                                 {"time": "2021-05-25T00:00:00.000Z",
                                  "open": 38874.53, "high": 39878.51, "low": 36481.26, "close": 38361.81, "volume": 26107.55687287},
                                 {"time": "2021-05-26T00:00:00.000Z",
                                  "open": 38361.81, "high": 40900, "low": 37837.36, "close": 39293.23, "volume": 25107.68631485},
                                 {"time": "2021-05-27T00:00:00.000Z",
                                  "open": 39293.23, "high": 40440.95, "low": 37191.99, "close": 38556.88, "volume": 19390.89447264},
                                 {"time": "2021-05-28T00:00:00.000Z",
                                  "open": 38556.88, "high": 38900, "low": 34688, "close": 35680.47, "volume": 36231.72333985},
                                 {"time": "2021-05-29T00:00:00.000Z",
                                  "open": 35680.48, "high": 37325, "low": 33650.86, "close": 34627.82, "volume": 27999.15073256},
                                 {"time": "2021-05-30T00:00:00.000Z",
                                  "open": 34627.81, "high": 36500, "low": 33333, "close": 35669.44, "volume": 16038.94658082},
                                 {"time": "2021-05-31T00:00:00.000Z",
                                  "open": 35669.44, "high": 37534.09, "low": 34183, "close": 37279.31, "volume": 19527.11600162},
                                 {"time": "2021-06-01T00:00:00.000Z",
                                  "open": 37276.23, "high": 37918.97, "low": 35669.14, "close": 36685, "volume": 15713.10281689},
                                 {"time": "2021-06-02T00:00:00.000Z",
                                  "open": 36684.99, "high": 38237.37, "low": 35920, "close": 37577.91, "volume": 13952.35891376},
                                 {"time": "2021-06-03T00:00:00.000Z",
                                  "open": 37581.83, "high": 39489.82, "low": 37184.88, "close": 39248.55, "volume": 14325.7560692},
                                 {"time": "2021-06-04T00:00:00.000Z",
                                  "open": 39248.54, "high": 39291.24, "low": 35593.22, "close": 36856.52, "volume": 17917.63251152},
                                 {"time": "2021-06-05T00:00:00.000Z",
                                  "open": 36856.53, "high": 37921.35, "low": 34832.17, "close": 35539.49, "volume": 13654.69033381},
                                 {"time": "2021-06-06T00:00:00.000Z",
                                  "open": 35539.97, "high": 36477.86, "low": 35258, "close": 35800.48, "volume": 8184.50324318},
                                 {"time": "2021-06-07T00:00:00.000Z",
                                  "open": 35800.48, "high": 36812.39, "low": 33333, "close": 33575.91, "volume": 18997.71626987},
                                 {"time": "2021-06-08T00:00:00.000Z",
                                  "open": 33567.26, "high": 34069, "low": 31004.95, "close": 33402.13, "volume": 34937.55754915},
                                 {"time": "2021-06-09T00:00:00.000Z",
                                  "open": 33402.13, "high": 37573.99, "low": 32408.53, "close": 37403.86, "volume": 26040.89793254},
                                 {"time": "2021-06-10T00:00:00.000Z",
                                  "open": 37404.75, "high": 38425.67, "low": 35800, "close": 36694.05, "volume": 19853.81418911},
                                 {"time": "2021-06-11T00:00:00.000Z",
                                  "open": 36694.91, "high": 37695, "low": 35944, "close": 37338.44, "volume": 14142.19349961},
                                 {"time": "2021-06-12T00:00:00.000Z",
                                  "open": 37340.08, "high": 37448, "low": 34635.47, "close": 35557.33, "volume": 14516.54556279},
                                 {"time": "2021-06-13T00:00:00.000Z",
                                  "open": 35557.32, "high": 39396, "low": 34780.57, "close": 39015.24, "volume": 15748.96203951},
                                 {"time": "2021-06-14T00:00:00.000Z",
                                  "open": 39015.24, "high": 41076.03, "low": 38744.83, "close": 40539.47, "volume": 21916.84415956},
                                 {"time": "2021-06-15T00:00:00.000Z",
                                  "open": 40537.93, "high": 41322.55, "low": 39510.98, "close": 40162.37, "volume": 17020.70299457},
                                 {"time": "2021-06-16T00:00:00.000Z",
                                  "open": 40158.06, "high": 40499, "low": 38105, "close": 38351, "volume": 17238.57541183},
                                 {"time": "2021-06-17T00:00:00.000Z",
                                  "open": 38351, "high": 39554.98, "low": 37365.63, "close": 38095.17, "volume": 15970.35031651},
                                 {"time": "2021-06-18T00:00:00.000Z",
                                  "open": 38103.94, "high": 38216.22, "low": 35153.16, "close": 35841.81, "volume": 23472.37724375},
                                 {"time": "2021-06-19T00:00:00.000Z",
                                  "open": 35840.96, "high": 36457.18, "low": 34833.26, "close": 35484.79, "volume": 16801.08670642},
                                 {"time": "2021-06-20T00:00:00.000Z",
                                  "open": 35484.8, "high": 36139.73, "low": 33347.05, "close": 35585.78, "volume": 19400.07454207},
                                 {"time": "2021-06-21T00:00:00.000Z",
                                  "open": 35585.79, "high": 35741.54, "low": 31259, "close": 31609.82, "volume": 41304.39903519},
                                 {"time": "2021-06-22T00:00:00.000Z",
                                  "open": 31609.82, "high": 33352, "low": 28800, "close": 32538.37, "volume": 48409.17333487},
                                 {"time": "2021-06-23T00:00:00.000Z",
                                  "open": 32538.9, "high": 34881.02, "low": 31708.03, "close": 33688.35, "volume": 23124.81248294},
                                 {"time": "2021-06-24T00:00:00.000Z",
                                  "open": 33688.51, "high": 35295.67, "low": 32300, "close": 34654.58, "volume": 16868.94084747},
                                 {"time": "2021-06-25T00:00:00.000Z",
                                  "open": 34654.87, "high": 35500, "low": 31300, "close": 31594.63, "volume": 26505.19266978},
                                 {"time": "2021-06-26T00:00:00.000Z",
                                  "open": 31594.62, "high": 32711.68, "low": 30173.49, "close": 32275.19, "volume": 18299.38702942},
                                 {"time": "2021-06-27T00:00:00.000Z",
                                  "open": 32267.59, "high": 34749, "low": 32002.21, "close": 34709.23, "volume": 14428.92978925},
                                 {"time": "2021-06-28T00:00:00.000Z",
                                  "open": 34708.53, "high": 35301.43, "low": 33864.99, "close": 34493.22, "volume": 14920.2647883},
                                 {"time": "2021-06-29T00:00:00.000Z",
                                  "open": 34493.22, "high": 36675, "low": 34233.35, "close": 35904.28, "volume": 17917.59826258},
                                 {"time": "2021-06-30T00:00:00.000Z",
                                  "open": 35906.17, "high": 36094.42, "low": 34033, "close": 35060, "volume": 16700.72309396},
                                 {"time": "2021-07-01T00:00:00.000Z",
                                  "open": 35060, "high": 35066.54, "low": 32703.48, "close": 33516.11, "volume": 13210.9509733},
                                 {"time": "2021-07-02T00:00:00.000Z",
                                  "open": 33510.93, "high": 33972.06, "low": 32704.73, "close": 33805.02, "volume": 11765.65389836},
                                 {"time": "2021-07-03T00:00:00.000Z",
                                  "open": 33805.01, "high": 34953.07, "low": 33320, "close": 34682.16, "volume": 6197.80112922},
                                 {"time": "2021-07-04T00:00:00.000Z",
                                  "open": 34685.03, "high": 35951, "low": 34390.53, "close": 35284.05, "volume": 6224.03057033},
                                 {"time": "2021-07-05T00:00:00.000Z",
                                  "open": 35284.06, "high": 35290.82, "low": 33156.86, "close": 33697.78, "volume": 9607.60619891},
                                 {"time": "2021-07-06T00:00:00.000Z",
                                  "open": 33697.78, "high": 35100, "low": 33300, "close": 34225.73, "volume": 10313.01368118},
                                 {"time": "2021-07-07T00:00:00.000Z",
                                  "open": 34225.72, "high": 35077.46, "low": 33770.01, "close": 33878.56, "volume": 9404.8979867},
                                 {"time": "2021-07-08T00:00:00.000Z",
                                  "open": 33879.5, "high": 33934.62, "low": 32111, "close": 32875.95, "volume": 14147.3525248},
                                 {"time": "2021-07-09T00:00:00.000Z",
                                  "open": 32875.95, "high": 34100, "low": 32255.24, "close": 33824.26, "volume": 8320.90649257},
                                 {"time": "2021-07-10T00:00:00.000Z",
                                  "open": 33824.25, "high": 34267.14, "low": 33027.83, "close": 33515.35, "volume": 4906.24582215},
                                 {"time": "2021-07-11T00:00:00.000Z",
                                  "open": 33515.35, "high": 34607.37, "low": 33333.33, "close": 34259.23, "volume": 7023.06500489},
                                 {"time": "2021-07-12T00:00:00.000Z",
                                  "open": 34259.22, "high": 34670, "low": 32665, "close": 33091.1, "volume": 10145.24046995},
                                 {"time": "2021-07-13T00:00:00.000Z",
                                  "open": 33077.43, "high": 33337.61, "low": 32201.12, "close": 32734.14, "volume": 9478.04218884},
                                 {"time": "2021-07-14T00:00:00.000Z",
                                  "open": 32734.14, "high": 33125.55, "low": 31600, "close": 32816.39, "volume": 10355.3186961},
                                 {"time": "2021-07-15T00:00:00.000Z",
                                  "open": 32815.75, "high": 33187.6, "low": 31064.77, "close": 31868.68, "volume": 11273.61174025},
                                 {"time": "2021-07-16T00:00:00.000Z",
                                  "open": 31872.49, "high": 32259.16, "low": 31025.42, "close": 31388.06, "volume": 10006.47793721},
                                 {"time": "2021-07-17T00:00:00.000Z",
                                  "open": 31389.57, "high": 31949.99, "low": 31179.01, "close": 31533.91, "volume": 5695.57411588},
                                 {"time": "2021-07-18T00:00:00.000Z",
                                  "open": 31533.9, "high": 32450, "low": 31125.6, "close": 31788.25, "volume": 5006.4831051},
                                 {"time": "2021-07-19T00:00:00.000Z",
                                  "open": 31786.37, "high": 31887.82, "low": 30429.99, "close": 30842.03, "volume": 11027.00528966},
                                 {"time": "2021-07-20T00:00:00.000Z",
                                  "open": 30842.04, "high": 31052.65, "low": 29301.56, "close": 29796.16, "volume": 18114.15286104},
                                 {"time": "2021-07-21T00:00:00.000Z",
                                  "open": 29796.15, "high": 32825, "low": 29501.02, "close": 32135.19, "volume": 15598.47830878},
                                 {"time": "2021-07-22T00:00:00.000Z",
                                  "open": 32152.68, "high": 32611.84, "low": 31729.68, "close": 32287.74, "volume": 7704.6404628},
                                 {"time": "2021-07-23T00:00:00.000Z",
                                  "open": 32287.75, "high": 33647.27, "low": 32000, "close": 33647.26, "volume": 9004.84262627},
                                 {"time": "2021-07-24T00:00:00.000Z",
                                  "open": 33647.27, "high": 34525, "low": 33424.7, "close": 34283.01, "volume": 9864.75243703},
                                 {"time": "2021-07-25T00:00:00.000Z",
                                  "open": 34291.67, "high": 35451.04, "low": 33888.89, "close": 35428.26, "volume": 9434.63002297},
                                 {"time": "2021-07-26T00:00:00.000Z",
                                  "open": 35428.25, "high": 40593.93, "low": 35251.54, "close": 37262.77, "volume": 40354.54555841},
                                 {"time": "2021-07-27T00:00:00.000Z",
                                  "open": 37262.76, "high": 39550, "low": 36413.26, "close": 39470.48, "volume": 24709.10440908},
                                 {"time": "2021-07-28T00:00:00.000Z",
                                  "open": 39475.99, "high": 40925.56, "low": 38803.34, "close": 40035.1, "volume": 27198.33358111},
                                 {"time": "2021-07-29T00:00:00.000Z",
                                  "open": 40035.1, "high": 40644.85, "low": 39229.47, "close": 40039.44, "volume": 11022.42512817},
                                 {"time": "2021-07-30T00:00:00.000Z",
                                  "open": 40039.44, "high": 42320.57, "low": 38342, "close": 42237.95, "volume": 15063.2344059},
                                 {"time": "2021-07-31T00:00:00.000Z",
                                  "open": 42233.66, "high": 42414.61, "low": 41064.11, "close": 41495.01, "volume": 9351.94288576},
                                 {"time": "2021-08-01T00:00:00.000Z",
                                  "open": 41499, "high": 42605.64, "low": 39426, "close": 39865.41, "volume": 10617.67293456},
                                 {"time": "2021-08-02T00:00:00.000Z",
                                  "open": 39859.18, "high": 40459.68, "low": 38690, "close": 39149.59, "volume": 12379.41216528},
                                 {"time": "2021-08-03T00:00:00.000Z",
                                  "open": 39149.59, "high": 39785.3, "low": 37627.21, "close": 38191.44, "volume": 15094.81739624},
                                 {"time": "2021-08-04T00:00:00.000Z",
                                  "open": 38189.4, "high": 39965.1, "low": 37509, "close": 39722.41, "volume": 15327.67624669},
                                 {"time": "2021-08-05T00:00:00.000Z",
                                  "open": 39718.9, "high": 41447.89, "low": 37300, "close": 40888.74, "volume": 28507.85532682},
                                 {"time": "2021-08-06T00:00:00.000Z",
                                  "open": 40888.73, "high": 43399.26, "low": 39876.74, "close": 42869.58, "volume": 20425.1145444},
                                 {"time": "2021-08-07T00:00:00.000Z",
                                  "open": 42869.59, "high": 44750, "low": 42457.27, "close": 44637.34, "volume": 19634.79475606},
                                 {"time": "2021-08-08T00:00:00.000Z",
                                  "open": 44628.11, "high": 45363.83, "low": 43131.97, "close": 43829.14, "volume": 18712.27158173},
                                 {"time": "2021-08-09T00:00:00.000Z",
                                  "open": 43829.15, "high": 46497.42, "low": 42821.3, "close": 46285.48, "volume": 19464.72493503},
                                 {"time": "2021-08-10T00:00:00.000Z",
                                  "open": 46280, "high": 46718.44, "low": 44650.27, "close": 45595.66, "volume": 13756.50066875},
                                 {"time": "2021-08-11T00:00:00.000Z",
                                  "open": 45601.82, "high": 46781.09, "low": 45350.74, "close": 45553.49, "volume": 11628.32043954},
                                 {"time": "2021-08-12T00:00:00.000Z",
                                  "open": 45553.1, "high": 46223.95, "low": 43714.36, "close": 44422.14, "volume": 13169.78176994},
                                 {"time": "2021-08-13T00:00:00.000Z",
                                  "open": 44422.15, "high": 47953.9, "low": 44249.64, "close": 47833.64, "volume": 11794.94196257},
                                 {"time": "2021-08-14T00:00:00.000Z",
                                  "open": 47833.64, "high": 48176.52, "low": 46034.93, "close": 47109.65, "volume": 8130.36769388},
                                 {"time": "2021-08-15T00:00:00.000Z",
                                  "open": 47109.64, "high": 47400, "low": 45525.19, "close": 47014.49, "volume": 7745.70200802},
                                 {"time": "2021-08-16T00:00:00.000Z",
                                  "open": 47014.38, "high": 48076.52, "low": 45676.91, "close": 45902.46, "volume": 10467.66031475},
                                 {"time": "2021-08-17T00:00:00.000Z",
                                  "open": 45903.22, "high": 47174.62, "low": 44405.39, "close": 44671.58, "volume": 12756.14777945},
                                 {"time": "2021-08-18T00:00:00.000Z",
                                  "open": 44675.34, "high": 46035.27, "low": 44216.47, "close": 44707.98, "volume": 10327.49887365},
                                 {"time": "2021-08-19T00:00:00.000Z",
                                  "open": 44709.1, "high": 47088.08, "low": 43955, "close": 46765.87, "volume": 11938.12185013},
                                 {"time": "2021-08-20T00:00:00.000Z",
                                  "open": 46769.02, "high": 49400, "low": 46644.95, "close": 49342.4, "volume": 13184.67048641},
                                 {"time": "2021-08-21T00:00:00.000Z",
                                  "open": 49349.99, "high": 49821.92, "low": 48285.28, "close": 48867.02, "volume": 9574.83615164},
                                 {"time": "2021-08-22T00:00:00.000Z",
                                  "open": 48870.21, "high": 49526.13, "low": 48102.89, "close": 49284.63, "volume": 6243.39643384},
                                 {"time": "2021-08-23T00:00:00.000Z",
                                  "open": 49284.38, "high": 50505, "low": 49012.84, "close": 49506.5, "volume": 10824.26838095},
                                 {"time": "2021-08-24T00:00:00.000Z",
                                  "open": 49504.53, "high": 49875.9, "low": 47600, "close": 47682.45, "volume": 12068.33823315},
                                 {"time": "2021-08-25T00:00:00.000Z",
                                  "open": 47682.45, "high": 49277, "low": 47122.02, "close": 48987.33, "volume": 9289.23177109},
                                 {"time": "2021-08-26T00:00:00.000Z",
                                  "open": 48991.57, "high": 49365.1, "low": 46315.48, "close": 46851.47, "volume": 11651.34373853},
                                 {"time": "2021-08-27T00:00:00.000Z",
                                  "open": 46851.47, "high": 49185.12, "low": 46361.01, "close": 49077.57, "volume": 8924.12247655},
                                 {"time": "2021-08-28T00:00:00.000Z",
                                  "open": 49081.08, "high": 49309.83, "low": 48370, "close": 48942.94, "volume": 4895.70191319},
                                 {"time": "2021-08-29T00:00:00.000Z",
                                  "open": 48942.89, "high": 49667.1, "low": 47801, "close": 48802.58, "volume": 6935.09511006},
                                 {"time": "2021-08-30T00:00:00.000Z",
                                  "open": 48802.58, "high": 48906.51, "low": 46866, "close": 46993.71, "volume": 12249.23741495},
                                 {"time": "2021-08-31T00:00:00.000Z",
                                  "open": 46996.8, "high": 48259.68, "low": 46708.93, "close": 47112.5, "volume": 12919.43970212},
                                 {"time": "2021-09-01T00:00:00.000Z",
                                  "open": 47110.33, "high": 49125, "low": 46537.62, "close": 48839.86, "volume": 13010.83839478},
                                 {"time": "2021-09-02T00:00:00.000Z",
                                  "open": 48834.27, "high": 50392.19, "low": 48620.73, "close": 49279.37, "volume": 14120.3826407},
                                 {"time": "2021-09-03T00:00:00.000Z",
                                  "open": 49279.38, "high": 51064.44, "low": 48349.66, "close": 50025, "volume": 14577.22621879},
                                 {"time": "2021-09-04T00:00:00.000Z",
                                  "open": 50025, "high": 50558.75, "low": 49400, "close": 49942.98, "volume": 7595.48816499},
                                 {"time": "2021-09-05T00:00:00.000Z",
                                  "open": 49944.89, "high": 51907.08, "low": 49500, "close": 51789.17, "volume": 7743.93472308},
                                 {"time": "2021-09-06T00:00:00.000Z",
                                  "open": 51789.17, "high": 52802.03, "low": 51020.08, "close": 52698.81, "volume": 9571.75471136},
                                 {"time": "2021-09-07T00:00:00.000Z",
                                  "open": 52698.8, "high": 52944.96, "low": 42830.77, "close": 46894.5, "volume": 29022.815351},
                                 {"time": "2021-09-08T00:00:00.000Z",
                                  "open": 46894.49, "high": 47381.47, "low": 44423.49, "close": 46060.36, "volume": 19624.55093117},
                                 {"time": "2021-09-09T00:00:00.000Z",
                                  "open": 46059.94, "high": 47400, "low": 45511.82, "close": 46400, "volume": 13594.64633403},
                                 {"time": "2021-09-10T00:00:00.000Z",
                                  "open": 46396.26, "high": 47040.76, "low": 44140.48, "close": 44851.45, "volume": 14905.54795341},
                                 {"time": "2021-09-11T00:00:00.000Z",
                                  "open": 44850.37, "high": 45989.94, "low": 44730.29, "close": 45171.83, "volume": 6207.38337004},
                                 {"time": "2021-09-12T00:00:00.000Z",
                                  "open": 45173.66, "high": 46462.98, "low": 44754.31, "close": 46024.23, "volume": 6499.85821749},
                                 {"time": "2021-09-13T00:00:00.000Z",
                                  "open": 46027.8, "high": 46900, "low": 43465, "close": 44947.72, "volume": 15015.30289712},
                                 {"time": "2021-09-14T00:00:00.000Z",
                                  "open": 44953.23, "high": 47274.92, "low": 44679.92, "close": 47127.22, "volume": 12644.16435358},
                                 {"time": "2021-09-15T00:00:00.000Z",
                                  "open": 47127.81, "high": 48475, "low": 46705, "close": 48148.12, "volume": 12139.84206628},
                                 {"time": "2021-09-16T00:00:00.000Z",
                                  "open": 48144.03, "high": 48499.99, "low": 47020.82, "close": 47753.16, "volume": 11739.27966629},
                                 {"time": "2021-09-17T00:00:00.000Z",
                                  "open": 47753.15, "high": 48176.64, "low": 46752.67, "close": 47303.5, "volume": 9088.56956453},
                                 {"time": "2021-09-18T00:00:00.000Z",
                                  "open": 47306.87, "high": 48825.62, "low": 47050, "close": 48314.56, "volume": 7491.03449237},
                                 {"time": "2021-09-19T00:00:00.000Z",
                                  "open": 48309.86, "high": 48379.19, "low": 46850.87, "close": 47255.92, "volume": 7086.34626236},
                                 {"time": "2021-09-20T00:00:00.000Z",
                                  "open": 47253.71, "high": 47358.92, "low": 42500, "close": 43012.97, "volume": 27706.67636162},
                                 {"time": "2021-09-21T00:00:00.000Z",
                                  "open": 43007.69, "high": 43639.88, "low": 39600, "close": 40719.6, "volume": 27275.33818249},
                                 {"time": "2021-09-22T00:00:00.000Z",
                                  "open": 40736.81, "high": 44035.44, "low": 40570.42, "close": 43575.1, "volume": 19462.53865157},
                                 {"time": "2021-09-23T00:00:00.000Z",
                                  "open": 43575.19, "high": 45000, "low": 43096.73, "close": 44897.59, "volume": 13446.87468684},
                                 {"time": "2021-09-24T00:00:00.000Z",
                                  "open": 44896.55, "high": 45200, "low": 40683.29, "close": 42848.92, "volume": 22451.57711462},
                                 {"time": "2021-09-25T00:00:00.000Z",
                                  "open": 42848.44, "high": 42998.76, "low": 41675, "close": 42705.51, "volume": 7383.32390261},
                                 {"time": "2021-09-26T00:00:00.000Z",
                                  "open": 42705.51, "high": 43937, "low": 40803, "close": 43178.02, "volume": 9971.64545213},
                                 {"time": "2021-09-27T00:00:00.000Z",
                                  "open": 43172.17, "high": 44366.96, "low": 42111, "close": 42171.76, "volume": 10094.77099806},
                                 {"time": "2021-09-28T00:00:00.000Z",
                                  "open": 42166.31, "high": 42780, "low": 40897.7, "close": 41026.07, "volume": 10491.81363045},
                                 {"time": "2021-09-29T00:00:00.000Z",
                                  "open": 41030.19, "high": 42638.99, "low": 40750.12, "close": 41522.16, "volume": 10303.30152312},
                                 {"time": "2021-09-30T00:00:00.000Z",
                                  "open": 41519.11, "high": 44110.17, "low": 41409.67, "close": 43824.43, "volume": 12835.9838101},
                                 {"time": "2021-10-01T00:00:00.000Z",
                                  "open": 43828.89, "high": 48500, "low": 43287.44, "close": 48165.76, "volume": 20095.33155568},
                                 {"time": "2021-10-02T00:00:00.000Z",
                                  "open": 48165.76, "high": 48359.33, "low": 47451, "close": 47657.69, "volume": 7262.63442143},
                                 {"time": "2021-10-03T00:00:00.000Z",
                                  "open": 47663.74, "high": 49300, "low": 47120.11, "close": 48233.99, "volume": 7989.25925892},
                                 {"time": "2021-10-04T00:00:00.000Z",
                                  "open": 48233.99, "high": 49505, "low": 46916.7, "close": 49245.54, "volume": 13938.64182971},
                                 {"time": "2021-10-05T00:00:00.000Z",
                                  "open": 49244.13, "high": 51906.23, "low": 49057.18, "close": 51493.99, "volume": 17828.59726658},
                                 {"time": "2021-10-06T00:00:00.000Z",
                                  "open": 51499.77, "high": 55757.11, "low": 50416.01, "close": 55339.48, "volume": 25869.64925998},
                                 {"time": "2021-10-07T00:00:00.000Z",
                                  "open": 55346.95, "high": 55356.85, "low": 53379, "close": 53797.82, "volume": 15820.83417633},
                                 {"time": "2021-10-08T00:00:00.000Z",
                                  "open": 53805.46, "high": 56113, "low": 53634.41, "close": 53963.82, "volume": 12416.9019906},
                                 {"time": "2021-10-09T00:00:00.000Z",
                                  "open": 53965.18, "high": 55500, "low": 53675, "close": 54962.29, "volume": 7123.05512404},
                                 {"time": "2021-10-10T00:00:00.000Z",
                                  "open": 54963.29, "high": 56545.24, "low": 54112.95, "close": 54690.53, "volume": 9370.1912382},
                                 {"time": "2021-10-11T00:00:00.000Z",
                                  "open": 54683.09, "high": 57833.23, "low": 54411.54, "close": 57487.44, "volume": 13422.42081815},
                                 {"time": "2021-10-12T00:00:00.000Z",
                                  "open": 57485.97, "high": 57674.55, "low": 53873.78, "close": 56005.1, "volume": 15457.00896085},
                                 {"time": "2021-10-13T00:00:00.000Z",
                                  "open": 56005.1, "high": 57771.33, "low": 54236.09, "close": 57367.32, "volume": 14691.01338574},
                                 {"time": "2021-10-14T00:00:00.000Z",
                                  "open": 57367.32, "high": 58520.71, "low": 56832.32, "close": 57359.51, "volume": 12198.02457938},
                                 {"time": "2021-10-15T00:00:00.000Z",
                                  "open": 57359.51, "high": 62910, "low": 56867.11, "close": 61695.39, "volume": 27105.19764492},
                                 {"time": "2021-10-16T00:00:00.000Z",
                                  "open": 61690.32, "high": 62350, "low": 60139.01, "close": 60877.42, "volume": 10161.25819825},
                                 {"time": "2021-10-17T00:00:00.000Z",
                                  "open": 60867.05, "high": 61744.17, "low": 58943.83, "close": 61527.11, "volume": 9153.32273471},
                                 {"time": "2021-10-18T00:00:00.000Z",
                                  "open": 61530.07, "high": 62675.19, "low": 59887.74, "close": 62042.41, "volume": 19798.1489024},
                                 {"time": "2021-10-19T00:00:00.000Z",
                                  "open": 62042.4, "high": 64500, "low": 61333, "close": 64303.14, "volume": 17380.30486176},
                                 {"time": "2021-10-20T00:00:00.000Z",
                                  "open": 64303.14, "high": 66999, "low": 63525, "close": 66026.54, "volume": 19194.45410878},
                                 {"time": "2021-10-21T00:00:00.000Z",
                                  "open": 66021.26, "high": 66650.85, "low": 62050, "close": 62204.02, "volume": 23474.39133313},
                                 {"time": "2021-10-22T00:00:00.000Z",
                                  "open": 62204.01, "high": 63750, "low": 60000, "close": 60687.64, "volume": 17217.70765017},
                                 {"time": "2021-10-23T00:00:00.000Z",
                                  "open": 60682.88, "high": 61750, "low": 59650, "close": 61300.01, "volume": 6708.15013802},
                                 {"time": "2021-10-24T00:00:00.000Z",
                                  "open": 61300.02, "high": 61491.96, "low": 59522.89, "close": 60854.48, "volume": 8573.72719199},
                                 {"time": "2021-10-25T00:00:00.000Z",
                                  "open": 60847.91, "high": 63726.58, "low": 60653.5, "close": 63083.54, "volume": 10496.58058683},
                                 {"time": "2021-10-26T00:00:00.000Z",
                                  "open": 63081.96, "high": 63290.49, "low": 59837.07, "close": 60337.62, "volume": 12181.48144997},
                                 {"time": "2021-10-27T00:00:00.000Z",
                                  "open": 60333.19, "high": 61488, "low": 58100, "close": 58455.47, "volume": 18856.68565412},
                                 {"time": "2021-10-28T00:00:00.000Z",
                                  "open": 58462.73, "high": 62477.47, "low": 57653.88, "close": 60591.65, "volume": 19354.49897509},
                                 {"time": "2021-10-29T00:00:00.000Z",
                                  "open": 60591.63, "high": 62974, "low": 60188.73, "close": 62276.72, "volume": 15497.01101939},
                                 {"time": "2021-10-30T00:00:00.000Z",
                                  "open": 62287.95, "high": 62379.2, "low": 60725.01, "close": 61892.41, "volume": 6793.59698552},
                                 {"time": "2021-10-31T00:00:00.000Z",
                                  "open": 61896.38, "high": 62427.02, "low": 60001, "close": 61343.68, "volume": 8371.1698899},
                                 {"time": "2021-11-01T00:00:00.000Z",
                                  "open": 61346.17, "high": 62500, "low": 59500, "close": 60949.54, "volume": 11724.34161877},
                                 {"time": "2021-11-02T00:00:00.000Z",
                                  "open": 60956.4, "high": 64300, "low": 60672.16, "close": 63266.51, "volume": 13694.11539462},
                                 {"time": "2021-11-03T00:00:00.000Z",
                                  "open": 63266.5, "high": 63568.36, "low": 60070, "close": 62935.41, "volume": 12951.87181247},
                                 {"time": "2021-11-04T00:00:00.000Z",
                                  "open": 62935.4, "high": 63114.14, "low": 60728.77, "close": 61444.5, "volume": 11051.75853921},
                                 {"time": "2021-11-05T00:00:00.000Z",
                                  "open": 61444.49, "high": 62638.59, "low": 60777, "close": 61006.15, "volume": 10598.79620286},
                                 {"time": "2021-11-06T00:00:00.000Z",
                                  "open": 61006.14, "high": 61599.25, "low": 60125, "close": 61539.31, "volume": 6366.43859189},
                                 {"time": "2021-11-07T00:00:00.000Z",
                                  "open": 61539.3, "high": 63327.96, "low": 61397.86, "close": 63309.13, "volume": 5596.02460734},
                                 {"time": "2021-11-08T00:00:00.000Z",
                                  "open": 63309.12, "high": 67792.77, "low": 63309.12, "close": 67554.84, "volume": 17661.87297159},
                                 {"time": "2021-11-09T00:00:00.000Z",
                                  "open": 67554.13, "high": 68568.85, "low": 66261.79, "close": 66944.66, "volume": 14996.62508818},
                                 {"time": "2021-11-10T00:00:00.000Z",
                                  "open": 66938.76, "high": 69000, "low": 62800, "close": 64912.2, "volume": 19030.24517744},
                                 {"time": "2021-11-11T00:00:00.000Z",
                                  "open": 64912.2, "high": 65600, "low": 64133.78, "close": 64807.59, "volume": 10259.51472415},
                                 {"time": "2021-11-12T00:00:00.000Z",
                                  "open": 64807.73, "high": 65477.06, "low": 62300, "close": 64147.9, "volume": 13658.626579},
                                 {"time": "2021-11-13T00:00:00.000Z",
                                  "open": 64144.42, "high": 64990, "low": 63394.17, "close": 64400.01, "volume": 5968.10354833},
                                 {"time": "2021-11-14T00:00:00.000Z",
                                  "open": 64400.01, "high": 65525, "low": 63596.9, "close": 65505.02, "volume": 6081.57225975},
                                 {"time": "2021-11-15T00:00:00.000Z",
                                  "open": 65505.03, "high": 66339.9, "low": 63371, "close": 63624.59, "volume": 10426.55522646},
                                 {"time": "2021-11-16T00:00:00.000Z",
                                  "open": 63621.05, "high": 63624.6, "low": 58638, "close": 60107.98, "volume": 24910.15267457},
                                 {"time": "2021-11-17T00:00:00.000Z",
                                  "open": 60101.56, "high": 60824.38, "low": 58380, "close": 60351.51, "volume": 20907.21545076},
                                 {"time": "2021-11-18T00:00:00.000Z",
                                  "open": 60351.51, "high": 60977.21, "low": 56514.13, "close": 56898, "volume": 26749.21203012},
                                 {"time": "2021-11-19T00:00:00.000Z",
                                  "open": 56898.46, "high": 58412.12, "low": 55625, "close": 58122.16, "volume": 18554.56064192},
                                 {"time": "2021-11-20T00:00:00.000Z",
                                  "open": 58122.15, "high": 59900, "low": 57423.35, "close": 59760.76, "volume": 7584.65241653},
                                 {"time": "2021-11-21T00:00:00.000Z",
                                  "open": 59760.76, "high": 60070, "low": 58509.69, "close": 58671.21, "volume": 6661.36635434},
                                 {"time": "2021-11-22T00:00:00.000Z",
                                  "open": 58671.22, "high": 59526.51, "low": 55641.03, "close": 56280.81, "volume": 15889.22078935},
                                 {"time": "2021-11-23T00:00:00.000Z",
                                  "open": 56280.81, "high": 57886.03, "low": 55377, "close": 57566.85, "volume": 14019.78167156},
                                 {"time": "2021-11-24T00:00:00.000Z",
                                  "open": 57562.65, "high": 57607.97, "low": 55875.01, "close": 57162.66, "volume": 12509.25971944},
                                 {"time": "2021-11-25T00:00:00.000Z",
                                  "open": 57158.58, "high": 59445.99, "low": 57037.04, "close": 58987.27, "volume": 10668.67680723},
                                 {"time": "2021-11-26T00:00:00.000Z",
                                  "open": 58994.72, "high": 59194, "low": 53533, "close": 53757.67, "volume": 25946.30174683},
                                 {"time": "2021-11-27T00:00:00.000Z",
                                  "open": 53757.67, "high": 55316.67, "low": 53642.58, "close": 54759.05, "volume": 7899.30349126},
                                 {"time": "2021-11-28T00:00:00.000Z",
                                  "open": 54759.04, "high": 57490, "low": 53327, "close": 57317.17, "volume": 8222.42493442},
                                 {"time": "2021-11-29T00:00:00.000Z",
                                  "open": 57318.51, "high": 58908.27, "low": 56730, "close": 57838.06, "volume": 13454.23335496},
                                 {"time": "2021-11-30T00:00:00.000Z",
                                  "open": 57838.06, "high": 59249.77, "low": 55910.33, "close": 56987.97, "volume": 18038.78770838}, ]

        self.test_chart_input.reverse()
        self.raw_data = []

        for data in self.test_chart_input:
            self.raw_data.append([dateparse.parse(data["time"]).timestamp(), data["low"], data["high"], data["open"], data["close"], data["volume"]])

        self.test_chart = Chart(self.raw_data)

    def test_chartSma(self):
        self.assertEqual(
            round(self.test_chart.SMA(9, 0, "close"), 2), 56739.72)

    def test_chartIndexOfHighest(self):
        c = Chart(self.raw_data[:30])
        self.assertEqual(c.IndexOfHighest(), 20)

    def test_chartIndexOfLowest(self):
        c = Chart(self.raw_data[:30])
        self.assertEqual(c.IndexOfLowest(), 2)

    def test_ChartAroon(self):
        aroon = self.test_chart.Aroon(14)
        self.assertAlmostEqual(aroon.up, 0.00, 2)
        self.assertAlmostEqual(aroon.down, 85.71, 2)
        self.assertAlmostEqual(aroon.oscillator, -85.71, 2)

    def test_ChartEma(self):
        self.assertEqual(
            round(self.test_chart.EMA(9), 2), 57259.26)
