"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerCondition = void 0;
/**
 * ListenerCondition providers definition.
 *
 * @stability stable
 */
class ListenerCondition {
    /**
     * Create a host-header listener rule condition.
     *
     * @param values Hosts for host headers.
     * @stability stable
     */
    static hostHeaders(values) {
        return new HostHeaderListenerCondition(values);
    }
    /**
     * Create a http-header listener rule condition.
     *
     * @param name HTTP header name.
     * @param values HTTP header values.
     * @stability stable
     */
    static httpHeader(name, values) {
        return new HttpHeaderListenerCondition(name, values);
    }
    /**
     * Create a http-request-method listener rule condition.
     *
     * @param values HTTP request methods.
     * @stability stable
     */
    static httpRequestMethods(values) {
        return new HttpRequestMethodListenerCondition(values);
    }
    /**
     * Create a path-pattern listener rule condition.
     *
     * @param values Path patterns.
     * @stability stable
     */
    static pathPatterns(values) {
        return new PathPatternListenerCondition(values);
    }
    /**
     * Create a query-string listener rule condition.
     *
     * @param values Query string key/value pairs.
     * @stability stable
     */
    static queryStrings(values) {
        return new QueryStringListenerCondition(values);
    }
    /**
     * Create a source-ip listener rule condition.
     *
     * @param values Source ips.
     * @stability stable
     */
    static sourceIps(values) {
        return new SourceIpListenerCondition(values);
    }
}
exports.ListenerCondition = ListenerCondition;
/**
 * Host header config of the listener rule condition
 */
class HostHeaderListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'host-header',
            hostHeaderConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * HTTP header config of the listener rule condition
 */
class HttpHeaderListenerCondition extends ListenerCondition {
    constructor(name, values) {
        super();
        this.name = name;
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-header',
            httpHeaderConfig: {
                httpHeaderName: this.name,
                values: this.values,
            },
        };
    }
}
/**
 * HTTP reqeust method config of the listener rule condition
 */
class HttpRequestMethodListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-request-method',
            httpRequestMethodConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Path pattern config of the listener rule condition
 */
class PathPatternListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
        if (values && values.length > 5) {
            throw new Error("A rule can only have '5' condition values");
        }
    }
    renderRawCondition() {
        return {
            field: 'path-pattern',
            pathPatternConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Query string config of the listener rule condition
 */
class QueryStringListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'query-string',
            queryStringConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Source ip config of the listener rule condition
 */
class SourceIpListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'source-ip',
            sourceIpConfig: {
                values: this.values,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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