"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancer = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_canned_metrics_generated_1 = require("../elasticloadbalancingv2-canned-metrics.generated");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * Define a new network load balancer.
 *
 * @stability stable
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    /**
     * Looks up the network load balancer.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        const props = base_load_balancer_1.BaseLoadBalancer._queryContextProvider(scope, {
            userOptions: options,
            loadBalancerType: cxschema.LoadBalancerType.NETWORK,
        });
        return new LookedUpNetworkLoadBalancer(scope, id, props);
    }
    /**
     * @stability stable
     */
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props,
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'network',
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * Add a listener to this load balancer.
     *
     * @returns The newly created listener
     * @stability stable
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * This is extending the BaseLoadBalancer.logAccessLogs method to match the bucket permissions described
     * at https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-access-logs.html#access-logging-bucket-requirements
     *
     * @stability stable
     */
    logAccessLogs(bucket, prefix) {
        super.logAccessLogs(bucket, prefix);
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
    }
    /**
     * Return the given named metric for this Network Load Balancer.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricActiveFlowCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.activeFlowCountSum, {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricConsumedLCUs(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.consumedLcUsAverage, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * (deprecated) The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricHealthyHostCount`` instead
     */
    metricHealthyHostCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.healthyHostCountAverage, props);
    }
    /**
     * (deprecated) The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @deprecated use ``NetworkTargetGroup.metricUnHealthyHostCount`` instead
     */
    metricUnHealthyHostCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.unHealthyHostCountAverage, props);
    }
    /**
     * The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricNewFlowCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.newFlowCountSum, props);
    }
    /**
     * The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricProcessedBytes(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.processedBytesSum, props);
    }
    /**
     * The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpClientResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpClientResetCountSum, props);
    }
    /**
     * The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpElbResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpElbResetCountSum, props);
    }
    /**
     * The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTcpTargetResetCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.NetworkELBMetrics.tcpTargetResetCountSum, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ LoadBalancer: this.loadBalancerFullName }),
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
class LookedUpNetworkLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.loadBalancerArn = props.loadBalancerArn;
        this.loadBalancerCanonicalHostedZoneId = props.loadBalancerCanonicalHostedZoneId;
        this.loadBalancerDnsName = props.loadBalancerDnsName;
        this.vpc = ec2.Vpc.fromLookup(this, 'Vpc', {
            vpcId: props.vpcId,
        });
    }
    addListener(lid, props) {
        return new network_listener_1.NetworkListener(this, lid, {
            loadBalancer: this,
            ...props,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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