"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Define the target of a load balancer.
 *
 * @stability stable
 */
class TargetGroupBase extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment.
         *
         * @stability stable
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.string({ produce: () => this.targetType }),
            targets: cdk.Lazy.any({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.string({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.any({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.any({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer.
     *
     * @stability stable
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check.
     *
     * @stability stable
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     * @stability stable
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group.
     *
     * @stability stable
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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