"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapTagMapToCxschema = exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
/**
 * Helper to map a map of tags to cxschema tag format.
 * @internal
 */
function mapTagMapToCxschema(tagMap) {
    return Object.entries(tagMap)
        .map(([key, value]) => ({ key, value }));
}
exports.mapTagMapToCxschema = mapTagMapToCxschema;
//# sourceMappingURL=data:application/json;base64,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