"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddTarget = exports.CreateLambdaTargetGroup = exports.AddListener = exports.ObtainAlb = void 0;
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const elbt = require("aws-cdk-lib/aws-elasticloadbalancingv2-targets");
const utils_1 = require("./utils");
const alb_defaults_1 = require("./alb-defaults");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
//  Returns the correct ALB Load Balancer to use in this construct, either an existing
//  one provided as an argument or create  new one otherwise.
function ObtainAlb(scope, id, vpc, publicApi, existingLoadBalancerInterface, loadBalancerProps, logAccessLogs, loggingBucketProps) {
    let loadBalancer;
    if (existingLoadBalancerInterface) {
        loadBalancer = existingLoadBalancerInterface;
    }
    else {
        const consolidatedProps = loadBalancerProps
            ? utils_1.overrideProps(loadBalancerProps, { vpc, internetFacing: publicApi })
            : { vpc, internetFacing: publicApi };
        loadBalancer = new elb.ApplicationLoadBalancer(scope, `${id}-alb`, consolidatedProps);
        if (logAccessLogs === undefined || logAccessLogs === true) {
            const consolidatedLoggingBucketProps = loggingBucketProps
                ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultLoggingBucketProps(), loggingBucketProps)
                : s3_bucket_defaults_1.DefaultLoggingBucketProps();
            const loggingBucket = s3_bucket_helper_1.createAlbLoggingBucket(scope, id, consolidatedLoggingBucketProps);
            loadBalancer.logAccessLogs(loggingBucket);
        }
    }
    return loadBalancer;
}
exports.ObtainAlb = ObtainAlb;
function AddListener(scope, loadBalancer, targetGroup, listenerProps) {
    var _a;
    let consolidatedListenerProps;
    consolidatedListenerProps = utils_1.overrideProps(alb_defaults_1.DefaultListenerProps(loadBalancer), listenerProps);
    //   create the listener
    const listener = new elb.ApplicationListener(scope, "listener", consolidatedListenerProps);
    loadBalancer.listeners.push(listener);
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTP) {
        // This will use core.printWarning in the actual construct
        utils_1.printWarning("AWS recommends encrypting traffic to an Application Load Balancer using HTTPS.");
        if (((_a = listenerProps.certificates) === null || _a === void 0 ? void 0 : _a.length) > 0) {
            throw new Error("HTTP listeners cannot use a certificate");
        }
    }
    else {
        if (!listenerProps.certificates || listenerProps.certificates.length === 0) {
            throw new Error("A listener using HTTPS protocol requires a certificate");
        }
        listener.addCertificates("listener-cert-add", listenerProps.certificates);
    }
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTPS) {
        const opt = {
            port: "443",
            protocol: "HTTPS",
        };
        const httpListener = new elb.ApplicationListener(scope, "redirect-listener", {
            loadBalancer,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect(opt),
        });
        loadBalancer.listeners.push(httpListener);
    }
    AddTarget(scope, targetGroup, listener);
    return listener;
}
exports.AddListener = AddListener;
function CreateLambdaTargetGroup(scope, id, lambdaFunction, targetProps) {
    const lambdaTarget = new elbt.LambdaTarget(lambdaFunction);
    return new elb.ApplicationTargetGroup(scope, `${id}-tg`, {
        targets: [lambdaTarget],
        targetGroupName: targetProps ? targetProps.targetGroupName : undefined,
        healthCheck: targetProps ? targetProps.healthCheck : undefined
    });
}
exports.CreateLambdaTargetGroup = CreateLambdaTargetGroup;
function AddTarget(scope, targetGroup, listener, ruleProps) {
    // AddRuleProps includes conditions and priority, combine that with targetGroups and
    // we can assemble AddApplicationTargetGroupProps
    if (ruleProps) {
        const consolidatedTargetProps = utils_1.overrideProps(ruleProps, { targetGroups: [targetGroup] });
        listener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    }
    else {
        listener.addTargetGroups("targets", {
            targetGroups: [targetGroup],
        });
    }
    return;
}
exports.AddTarget = AddTarget;
//# sourceMappingURL=data:application/json;base64,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