"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("aws-cdk-lib");
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, { ...apiGatewayProps, cloudWatchRole: false });
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    let cwRole;
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, { ...apiGatewayProps, cloudWatchRole: false });
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    let cwRole;
    // Configure API access logging
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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