"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupCognitoForElasticSearch = exports.buildIdentityPool = exports.buildUserPoolClient = exports.buildUserPool = void 0;
const cognito = require("aws-cdk-lib/aws-cognito");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    if (userPoolProps) {
        cognitoUserPoolProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    }
    else {
        cognitoUserPoolProps = cognito_defaults_1.DefaultUserPoolProps;
    }
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    // Add Cfn Nag suppress for the cognito SMS role policy
    const userPoolSmsRole = userPool.node.tryFindChild('smsRole');
    if (userPoolSmsRole) {
        utils_1.addCfnSuppressRules(userPool, [
            {
                id: 'W11',
                reason: `Allowing * resource on permissions policy since its used by Cognito to send SMS messages via sns:Publish`
            }
        ]);
    }
    return userPool;
}
exports.buildUserPool = buildUserPool;
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    if (cognitoUserPoolClientProps) {
        userPoolClientProps = utils_1.overrideProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    }
    else {
        userPoolClientProps = cognito_defaults_1.DefaultUserPoolClientProps(userPool);
    }
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    if (identityPoolProps) {
        cognitoIdentityPoolProps = utils_1.overrideProps(cognitoIdentityPoolProps, identityPoolProps);
    }
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
function setupCognitoForElasticSearch(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependsOn(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForElasticSearch = setupCognitoForElasticSearch;
//# sourceMappingURL=data:application/json;base64,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