"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("aws-cdk-lib/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    if (cfnDomainProps) {
        _cfnDomainProps = utils_1.overrideProps(_cfnDomainProps, cfnDomainProps);
    }
    const esDomain = new elasticsearch.CfnDomain(scope, "ElasticsearchDomain", _cfnDomainProps);
    utils_1.addCfnSuppressRules(esDomain, [
        {
            id: "W28",
            reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return [esDomain, cognitoKibanaConfigureRole];
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace',
            statistic: 'Minimum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,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