"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    if (props.kinesisStreamProps) {
        // If property overrides have been provided, incorporate them and deploy
        kinesisStreamProps = utils_1.overrideProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        kinesisStreamProps = kinesis_streams_defaults_1.DefaultStreamProps;
    }
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds',
            statistic: 'Maximum',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 2592000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded',
            statistic: 'Average',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 0,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,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