"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.createAlbLoggingBucket = exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("aws-cdk-lib/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
function buildS3Bucket(scope, props, bucketId) {
    return s3BucketWithLogging(scope, props.bucketProps, bucketId, props.loggingBucketProps);
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`,
            `${s3Bucket.bucketArn}`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function createAlbLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: "This is a log bucket for an Application Load Balancer"
        }
    ]);
    return loggingBucket;
}
exports.createAlbLoggingBucket = createAlbLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId, userLoggingBucketProps) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: aws_s3_1.StorageClass.GLACIER,
                    transitionAfter: aws_cdk_lib_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        let loggingBucketProps;
        if (userLoggingBucketProps) { // User provided logging bucket props
            loggingBucketProps = utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), userLoggingBucketProps);
        }
        else if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.removalPolicy) { // Deletes logging bucket only if it is empty
            loggingBucketProps = utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), { removalPolicy: s3BucketProps.removalPolicy });
        }
        else { // Default S3 bucket props
            loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
        }
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, loggingBucketProps);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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