"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(aws_cdk_lib_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": aws_cdk_lib_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        let snsTopicProps;
        if (props.topicProps) {
            // If property overrides have been provided, incorporate them and deploy
            snsTopicProps = utils_1.overrideProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            snsTopicProps = sns_defaults_1.DefaultSnsTopicProps;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey === undefined || props.enableEncryptionWithCustomerManagedKey === false) {
            // Retrieve SNS managed key to encrypt the SNS Topic
            const awsManagedKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
            snsTopicProps.masterKey = awsManagedKey;
        }
        else {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                snsTopicProps.masterKey = props.encryptionKey;
            }
            else {
                // Create a new Customer Managed KMS key
                snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return [topic, snsTopicProps.masterKey];
    }
    else {
        return [props.existingTopicObj];
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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