"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            const checkFifo = props.queueProps.fifo ? true : undefined;
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), { ...props.queueProps, fifo: checkFifo });
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const [dlq] = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: dlq
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(aws_cdk_lib_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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