"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test ObtainAlb with existing ALB', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, existingLoadBalancer);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "unique-name",
    });
});
test('Test ObtainAlb for new ALB with provided props', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, undefined, {
        loadBalancerName: 'new-loadbalancer',
        vpc,
        internetFacing: true
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "new-loadbalancer",
        Scheme: "internet-facing",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: "internal",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda"
    });
});
test('Test ObtainAlb for new ALB with custom props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
        Name: 'test-target-group'
    });
});
test('Test Add Target without ruleProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const testListener = new elb.ApplicationListener(stack, 'test-listener', {
        loadBalancer: existingLoadBalancer,
        protocol: elb.ApplicationProtocol.HTTP
    });
    defaults.AddTarget(stack, targetGroup, testListener);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        DefaultActions: [
            {
                TargetGroupArn: {
                    Ref: "testtargettgB2EE41CA"
                },
                Type: "forward"
            }
        ],
    });
});
test('Test Add Target with ruleProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    const secondTargetGroup = defaults.CreateLambdaTargetGroup(stack, 'second-target', testFunction, {
        targetGroupName: 'second-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const testListener = new elb.ApplicationListener(stack, 'test-listener', {
        loadBalancer: existingLoadBalancer,
        protocol: elb.ApplicationProtocol.HTTP
    });
    // The first target is default and can't have rules, so
    // we need to add 2 targets
    defaults.AddTarget(stack, targetGroup, testListener);
    defaults.AddTarget(stack, secondTargetGroup, testListener, {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Actions: [
            {
                TargetGroupArn: {
                    Ref: "secondtargettg0CE37E1F"
                },
                Type: "forward"
            }
        ],
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ]
    });
});
test('Test AddListener with defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const cert = acm.Certificate.fromCertificateArn(stack, 'not-really-a-cert', "arn:aws:acm:us-east-1:123456789012:certificate/85c52dc8-1b37-4afd-a7aa-f03aac2db0cc");
    defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
        certificates: [cert],
    });
    // This should create 2 listeners, HTTPS plus redirect of HTTP
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
});
test('Test AddListener with no cert for an HTTPS listener', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const app = () => {
        defaults.AddListener(stack, existingLoadBalancer, targetGroup, {});
    };
    expect(app).toThrowError();
});
test('Test AddListener error for HTTP with a cert', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    const cert = acm.Certificate.fromCertificateArn(stack, 'not-really-a-cert', "arn:aws:acm:us-east-1:123456789012:certificate/85c52dc8-1b37-4afd-a7aa-f03aac2db0cc");
    const app = () => {
        defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
            certificates: [cert],
            protocol: elb.ApplicationProtocol.HTTP,
        });
    };
    expect(app).toThrowError();
});
test('Test AddListener for HTTP Listener', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
    const targetGroup = defaults.CreateLambdaTargetGroup(stack, 'test-target', testFunction, {
        targetGroupName: 'test-target-group'
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.AddListener(stack, existingLoadBalancer, targetGroup, {
        protocol: elb.ApplicationProtocol.HTTP,
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
    expect(stack).not.toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
});
test('Test with custom logging bucket props', () => {
    // Creating ALB logging requires a region and account (but
    // these can be fake in unit tests)
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const testName = 'test-name';
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, true, { bucketName: testName });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketName: testName
    });
});
test('Test with no logging', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, false);
    expect(stack).not.toHaveResourceLike('AWS::S3::Bucket', {});
});
//# sourceMappingURL=data:application/json;base64,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