"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
const s3_bucket_helper_1 = require("../lib/s3-bucket-helper");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const acm = require("aws-cdk-lib/aws-certificatemanager");
test('check bucket policy metadata', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "F16",
                        reason: "Public website bucket policy requires a wildcard principal"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('check bucket metadata', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W35",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('test cloudfront check bucket policy', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": ["S3Bucket07682993", "Arn"],
                                    },
                                    "/*",
                                ],
                            ],
                        },
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('test cloudfront with no security headers ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {}, false);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override cloudfront logging bucket ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const logBucket = new aws_s3_1.Bucket(stack, 'loggingbucket');
    const myprops = {
        enableLogging: true,
        logBucket
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "SetHttpSecurityHeadersEE936115",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "loggingbucket6D73BD53",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const props = {
        defaultBehavior: {
            origin: new origins.S3Origin(sourceBucket),
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
            cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS
        },
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PUT",
                    "PATCH",
                    "POST",
                    "DELETE"
                ],
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "SetHttpSecurityHeadersEE936115",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront with custom cloudfront function', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom cloudfront function
    const cloudfrontFunction = new cloudfront.Function(stack, "MyFunction", {
        code: cloudfront.FunctionCode.fromInline("exports.handler = (event, context, callback) => {}")
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {
        defaultBehavior: {
            functionAssociations: [
                {
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                    function: cloudfrontFunction
                }
            ],
        }
    });
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "MyFunction3BAA72D1",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront replace custom lambda@edge', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom lambda@edg function
    const handler = new lambda.Function(stack, 'SomeHandler', {
        functionName: 'SomeHandler',
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const handlerVersion = new lambda.Version(stack, 'SomeHandlerVersion', {
        lambda: handler,
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {
        defaultBehavior: {
            edgeLambdas: [
                {
                    eventType: aws_cloudfront_1.LambdaEdgeEventType.VIEWER_REQUEST,
                    includeBody: false,
                    functionVersion: handlerVersion,
                }
            ]
        }
    }, false);
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "viewer-request",
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: "SomeHandlerVersionDA986E41"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override cloudfront custom domain names ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012');
    const myprops = {
        domainNames: ['mydomains'],
        certificate
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            Aliases: [
                "mydomains"
            ],
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRmcm9udC1kaXN0cmlidXRpb24tczMtaGVscGVyLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1zMy1oZWxwZXIudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsNENBQStDO0FBQy9DLDZDQUFvQztBQUNwQyx5REFBeUQ7QUFDekQsaURBQWlEO0FBQ2pELDBGQUFvRjtBQUNwRiw4REFBd0Q7QUFDeEQsZ0NBQThCO0FBQzlCLCtDQUE0QztBQUM1Qyw4REFBOEQ7QUFDOUQsK0RBQWlFO0FBQ2pFLDBEQUEwRDtBQUUxRCxJQUFJLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxnQ0FBYSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNoRCw0REFBMkIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFDakQsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRTtRQUNwRCxRQUFRLEVBQUU7WUFDUixPQUFPLEVBQUU7Z0JBQ1AsaUJBQWlCLEVBQUU7b0JBQ2pCO3dCQUNFLEVBQUUsRUFBRSxLQUFLO3dCQUNULE1BQU0sRUFBRSw0REFBNEQ7cUJBQ3JFO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0FBQ3RDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtJQUNqQyxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsZ0NBQWEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDaEQsNERBQTJCLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQ2pELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsaUJBQWlCLEVBQUU7UUFDOUMsUUFBUSxFQUFFO1lBQ1IsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQixFQUFFO29CQUNqQjt3QkFDRSxFQUFFLEVBQUUsS0FBSzt3QkFDVCxNQUFNLEVBQUUsaUZBQWlGO3FCQUMxRjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQztBQUN0QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7SUFDL0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFHLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2hELDREQUEyQixDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUVqRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsdUJBQXVCLEVBQUU7UUFDeEQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRSxHQUFHO29CQUNYLFNBQVMsRUFBRTt3QkFDVCxJQUFJLEVBQUU7NEJBQ0oscUJBQXFCLEVBQUUsT0FBTzt5QkFDL0I7cUJBQ0Y7b0JBQ0QsTUFBTSxFQUFFLE1BQU07b0JBQ2QsU0FBUyxFQUFFO3dCQUNULEdBQUcsRUFBRSxHQUFHO3FCQUNUO29CQUNELFFBQVEsRUFBRTt3QkFDUjs0QkFDRSxVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRTt3Q0FDRSxZQUFZLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLENBQUM7cUNBQzFDO29DQUNELElBQUk7aUNBQ0w7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsWUFBWSxFQUFFO2dDQUNaLGtCQUFrQjtnQ0FDbEIsS0FBSzs2QkFDTjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1lBQ0QsT0FBTyxFQUFFLFlBQVk7U0FDdEI7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywyQ0FBMkMsRUFBRSxHQUFHLEVBQUU7SUFDckQsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFHLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRWhELDREQUEyQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTVELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtRQUNoRSxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsY0FBYyxFQUFFLHVDQUF1QztnQkFDdkQsb0JBQW9CLEVBQUUsbUJBQW1CO2FBQzFDO1lBQ0QsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSTtZQUNiLFdBQVcsRUFBRSxPQUFPO1lBQ3BCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUU7b0JBQ04sWUFBWSxFQUFFO3dCQUNaLGlDQUFpQzt3QkFDakMsb0JBQW9CO3FCQUNyQjtpQkFDRjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQO29CQUNFLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osa0JBQWtCOzRCQUNsQixvQkFBb0I7eUJBQ3JCO3FCQUNGO29CQUNELEVBQUUsRUFBRSx1Q0FBdUM7b0JBQzNDLGNBQWMsRUFBRTt3QkFDZCxvQkFBb0IsRUFBRTs0QkFDcEIsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0Usb0NBQW9DO29DQUNwQzt3Q0FDRSxHQUFHLEVBQUUsK0NBQStDO3FDQUNyRDtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxREFBcUQsRUFBRSxHQUFHLEVBQUU7SUFDL0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFHLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2hELE1BQU0sU0FBUyxHQUFHLElBQUksZUFBTSxDQUFDLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztJQUVyRCxNQUFNLE9BQU8sR0FBRztRQUNkLGFBQWEsRUFBRSxJQUFJO1FBQ25CLFNBQVM7S0FDVixDQUFDO0lBRUYsNERBQTJCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUUxRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7UUFDaEUsa0JBQWtCLEVBQUU7WUFDbEIsb0JBQW9CLEVBQUU7Z0JBQ3BCLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELFFBQVEsRUFBRSxJQUFJO2dCQUNkLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxTQUFTLEVBQUUsaUJBQWlCO3dCQUM1QixXQUFXLEVBQUU7NEJBQ1gsWUFBWSxFQUFFO2dDQUNaLGdDQUFnQztnQ0FDaEMsYUFBYTs2QkFDZDt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCxjQUFjLEVBQUUsdUNBQXVDO2dCQUN2RCxvQkFBb0IsRUFBRSxtQkFBbUI7YUFDMUM7WUFDRCxpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRTtvQkFDTixZQUFZLEVBQUU7d0JBQ1osdUJBQXVCO3dCQUN2QixvQkFBb0I7cUJBQ3JCO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1A7b0JBQ0UsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixrQkFBa0I7NEJBQ2xCLG9CQUFvQjt5QkFDckI7cUJBQ0Y7b0JBQ0QsRUFBRSxFQUFFLHVDQUF1QztvQkFDM0MsY0FBYyxFQUFFO3dCQUNkLG9CQUFvQixFQUFFOzRCQUNwQixVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSxvQ0FBb0M7b0NBQ3BDO3dDQUNFLEdBQUcsRUFBRSwrQ0FBK0M7cUNBQ3JEO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsZ0NBQWEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDaEQsTUFBTSxLQUFLLEdBQWlDO1FBQzFDLGVBQWUsRUFBRTtZQUNmLE1BQU0sRUFBRSxJQUFJLE9BQU8sQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO1lBQzFDLG9CQUFvQixFQUFFLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUI7WUFDdkUsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjLENBQUMsU0FBUztZQUNuRCxhQUFhLEVBQUUsVUFBVSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0I7U0FDL0Q7S0FDRixDQUFDO0lBRUYsNERBQTJCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV4RCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7UUFDaEUsa0JBQWtCLEVBQUU7WUFDbEIsb0JBQW9CLEVBQUU7Z0JBQ3BCLGNBQWMsRUFBRTtvQkFDZCxLQUFLO29CQUNMLE1BQU07b0JBQ04sU0FBUztvQkFDVCxLQUFLO29CQUNMLE9BQU87b0JBQ1AsTUFBTTtvQkFDTixRQUFRO2lCQUNUO2dCQUNELGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRTtvQkFDYixLQUFLO29CQUNMLE1BQU07b0JBQ04sU0FBUztpQkFDVjtnQkFDRCxRQUFRLEVBQUUsSUFBSTtnQkFDZCxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsU0FBUyxFQUFFLGlCQUFpQjt3QkFDNUIsV0FBVyxFQUFFOzRCQUNYLFlBQVksRUFBRTtnQ0FDWixnQ0FBZ0M7Z0NBQ2hDLGFBQWE7NkJBQ2Q7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsY0FBYyxFQUFFLHVDQUF1QztnQkFDdkQsb0JBQW9CLEVBQUUsbUJBQW1CO2FBQzFDO1lBQ0QsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSTtZQUNiLFdBQVcsRUFBRSxPQUFPO1lBQ3BCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUU7b0JBQ04sWUFBWSxFQUFFO3dCQUNaLGlDQUFpQzt3QkFDakMsb0JBQW9CO3FCQUNyQjtpQkFDRjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQO29CQUNFLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osa0JBQWtCOzRCQUNsQixvQkFBb0I7eUJBQ3JCO3FCQUNGO29CQUNELEVBQUUsRUFBRSx1Q0FBdUM7b0JBQzNDLGNBQWMsRUFBRTt3QkFDZCxvQkFBb0IsRUFBRTs0QkFDcEIsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0Usb0NBQW9DO29DQUNwQzt3Q0FDRSxHQUFHLEVBQUUsK0NBQStDO3FDQUNyRDtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7SUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFHLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRWhELDZCQUE2QjtJQUM3QixNQUFNLGtCQUFrQixHQUFHLElBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1FBQ3RFLElBQUksRUFBRSxVQUFVLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxvREFBb0QsQ0FBQztLQUMvRixDQUFDLENBQUM7SUFFSCw0REFBMkIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1FBQy9DLGVBQWUsRUFBRTtZQUNmLG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxTQUFTLEVBQUUsVUFBVSxDQUFDLGlCQUFpQixDQUFDLGVBQWU7b0JBQ3ZELFFBQVEsRUFBRSxrQkFBa0I7aUJBQzdCO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsK0JBQStCLEVBQUU7UUFDNUQsa0JBQWtCLEVBQUU7WUFDbEIsb0JBQW9CLEVBQUU7Z0JBQ3BCLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELFFBQVEsRUFBRSxJQUFJO2dCQUNkLG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxTQUFTLEVBQUUsaUJBQWlCO3dCQUM1QixXQUFXLEVBQUU7NEJBQ1gsWUFBWSxFQUFFO2dDQUNaLG9CQUFvQjtnQ0FDcEIsYUFBYTs2QkFDZDt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCxjQUFjLEVBQUUsdUNBQXVDO2dCQUN2RCxvQkFBb0IsRUFBRSxtQkFBbUI7YUFDMUM7WUFDRCxpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRTtvQkFDTixZQUFZLEVBQUU7d0JBQ1osaUNBQWlDO3dCQUNqQyxvQkFBb0I7cUJBQ3JCO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1A7b0JBQ0UsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixrQkFBa0I7NEJBQ2xCLG9CQUFvQjt5QkFDckI7cUJBQ0Y7b0JBQ0QsRUFBRSxFQUFFLHVDQUF1QztvQkFDM0MsY0FBYyxFQUFFO3dCQUNkLG9CQUFvQixFQUFFOzRCQUNwQixVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSxvQ0FBb0M7b0NBQ3BDO3dDQUNFLEdBQUcsRUFBRSwrQ0FBK0M7cUNBQ3JEO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsZ0NBQWEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFaEQsNkJBQTZCO0lBQzdCLE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1FBQ3hELFlBQVksRUFBRSxhQUFhO1FBQzNCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7UUFDeEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7S0FDbkQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxjQUFjLEdBQUcsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtRQUNyRSxNQUFNLEVBQUUsT0FBTztLQUNoQixDQUFDLENBQUM7SUFFSCw0REFBMkIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1FBQy9DLGVBQWUsRUFBRTtZQUNmLFdBQVcsRUFBRTtnQkFDWDtvQkFDRSxTQUFTLEVBQUUsb0NBQW1CLENBQUMsY0FBYztvQkFDN0MsV0FBVyxFQUFFLEtBQUs7b0JBQ2xCLGVBQWUsRUFBRSxjQUFjO2lCQUNoQzthQUNGO1NBQ0Y7S0FDRixFQUNELEtBQUssQ0FBQyxDQUFDO0lBRVAsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtRQUM1RCxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsMEJBQTBCLEVBQUU7b0JBQzFCO3dCQUNFLFNBQVMsRUFBRSxnQkFBZ0I7d0JBQzNCLFdBQVcsRUFBRSxLQUFLO3dCQUNsQixpQkFBaUIsRUFBRTs0QkFDakIsR0FBRyxFQUFFLDRCQUE0Qjt5QkFDbEM7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsY0FBYyxFQUFFLHVDQUF1QztnQkFDdkQsb0JBQW9CLEVBQUUsbUJBQW1CO2FBQzFDO1lBQ0QsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSTtZQUNiLFdBQVcsRUFBRSxPQUFPO1lBQ3BCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUU7b0JBQ04sWUFBWSxFQUFFO3dCQUNaLGlDQUFpQzt3QkFDakMsb0JBQW9CO3FCQUNyQjtpQkFDRjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQO29CQUNFLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osa0JBQWtCOzRCQUNsQixvQkFBb0I7eUJBQ3JCO3FCQUNGO29CQUNELEVBQUUsRUFBRSx1Q0FBdUM7b0JBQzNDLGNBQWMsRUFBRTt3QkFDZCxvQkFBb0IsRUFBRTs0QkFDcEIsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0Usb0NBQW9DO29DQUNwQzt3Q0FDRSxHQUFHLEVBQUUsK0NBQStDO3FDQUNyRDtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7SUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFHLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2hELE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxxRkFBcUYsQ0FBQyxDQUFDO0lBRTdKLE1BQU0sT0FBTyxHQUFHO1FBQ2QsV0FBVyxFQUFFLENBQUMsV0FBVyxDQUFDO1FBQzFCLFdBQVc7S0FDWixDQUFDO0lBRUYsNERBQTJCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUUxRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7UUFDaEUsa0JBQWtCLEVBQUU7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLFdBQVc7YUFDWjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IFJlc291cmNlUGFydCB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzIH0gZnJvbSAnLi4vbGliL2Nsb3VkZnJvbnQtZGlzdHJpYnV0aW9uLWhlbHBlcic7XG5pbXBvcnQgeyBidWlsZFMzQnVja2V0IH0gZnJvbSAnLi4vbGliL3MzLWJ1Y2tldC1oZWxwZXInO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBCdWNrZXQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0ICogYXMgb3JpZ2lucyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWRmcm9udC1vcmlnaW5zJztcbmltcG9ydCB7IExhbWJkYUVkZ2VFdmVudFR5cGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWRmcm9udCc7XG5pbXBvcnQgKiBhcyBhY20gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNlcnRpZmljYXRlbWFuYWdlcic7XG5cbnRlc3QoJ2NoZWNrIGJ1Y2tldCBwb2xpY3kgbWV0YWRhdGEnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IFtzb3VyY2VCdWNrZXRdID0gYnVpbGRTM0J1Y2tldChzdGFjaywge30pO1xuICBDbG91ZEZyb250RGlzdHJpYnV0aW9uRm9yUzMoc3RhY2ssIHNvdXJjZUJ1Y2tldCk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6UzM6OkJ1Y2tldFBvbGljeScsIHtcbiAgICBNZXRhZGF0YToge1xuICAgICAgY2ZuX25hZzoge1xuICAgICAgICBydWxlc190b19zdXBwcmVzczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiBcIkYxNlwiLFxuICAgICAgICAgICAgcmVhc29uOiBcIlB1YmxpYyB3ZWJzaXRlIGJ1Y2tldCBwb2xpY3kgcmVxdWlyZXMgYSB3aWxkY2FyZCBwcmluY2lwYWxcIlxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfVxuICAgIH1cbiAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbik7XG59KTtcblxudGVzdCgnY2hlY2sgYnVja2V0IG1ldGFkYXRhJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBbc291cmNlQnVja2V0XSA9IGJ1aWxkUzNCdWNrZXQoc3RhY2ssIHt9KTtcbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzKHN0YWNrLCBzb3VyY2VCdWNrZXQpO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlMzOjpCdWNrZXQnLCB7XG4gICAgTWV0YWRhdGE6IHtcbiAgICAgIGNmbl9uYWc6IHtcbiAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBpZDogXCJXMzVcIixcbiAgICAgICAgICAgIHJlYXNvbjogXCJUaGlzIFMzIGJ1Y2tldCBpcyB1c2VkIGFzIHRoZSBhY2Nlc3MgbG9nZ2luZyBidWNrZXQgZm9yIENsb3VkRnJvbnQgRGlzdHJpYnV0aW9uXCJcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH1cbiAgICB9XG4gIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pO1xufSk7XG5cbnRlc3QoJ3Rlc3QgY2xvdWRmcm9udCBjaGVjayBidWNrZXQgcG9saWN5JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBbc291cmNlQnVja2V0XSA9IGJ1aWxkUzNCdWNrZXQoc3RhY2ssIHt9KTtcbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzKHN0YWNrLCBzb3VyY2VCdWNrZXQpO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTMzo6QnVja2V0UG9saWN5XCIsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFwiKlwiLFxuICAgICAgICAgIENvbmRpdGlvbjoge1xuICAgICAgICAgICAgQm9vbDoge1xuICAgICAgICAgICAgICBcImF3czpTZWN1cmVUcmFuc3BvcnRcIjogXCJmYWxzZVwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBFZmZlY3Q6IFwiRGVueVwiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgQVdTOiBcIipcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXCJTM0J1Y2tldDA3NjgyOTkzXCIsIFwiQXJuXCJdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiLypcIixcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcIlMzQnVja2V0MDc2ODI5OTNcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBjbG91ZGZyb250IHdpdGggbm8gc2VjdXJpdHkgaGVhZGVycyAnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IFtzb3VyY2VCdWNrZXRdID0gYnVpbGRTM0J1Y2tldChzdGFjaywge30pO1xuXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JTMyhzdGFjaywgc291cmNlQnVja2V0LCB7fSwgZmFsc2UpO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb25cIiwge1xuICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgQ2FjaGVQb2xpY3lJZDogXCI2NTgzMjdlYS1mODlkLTRmYWItYTYzZC03ZTg4NjM5ZTU4ZjZcIixcbiAgICAgICAgQ29tcHJlc3M6IHRydWUsXG4gICAgICAgIFRhcmdldE9yaWdpbklkOiBcIkNsb3VkRnJvbnREaXN0cmlidXRpb25PcmlnaW4xNzZFQzNBMTJcIixcbiAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6IFwicmVkaXJlY3QtdG8taHR0cHNcIlxuICAgICAgfSxcbiAgICAgIERlZmF1bHRSb290T2JqZWN0OiBcImluZGV4Lmh0bWxcIixcbiAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICBIdHRwVmVyc2lvbjogXCJodHRwMlwiLFxuICAgICAgSVBWNkVuYWJsZWQ6IHRydWUsXG4gICAgICBMb2dnaW5nOiB7XG4gICAgICAgIEJ1Y2tldDoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIkNsb3VkZnJvbnRMb2dnaW5nQnVja2V0M0MzRUZBQTdcIixcbiAgICAgICAgICAgIFwiUmVnaW9uYWxEb21haW5OYW1lXCJcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBPcmlnaW5zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBEb21haW5OYW1lOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIlMzQnVja2V0MDc2ODI5OTNcIixcbiAgICAgICAgICAgICAgXCJSZWdpb25hbERvbWFpbk5hbWVcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgSWQ6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjE3NkVDM0ExMlwiLFxuICAgICAgICAgIFMzT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgICBPcmlnaW5BY2Nlc3NJZGVudGl0eToge1xuICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgIFwib3JpZ2luLWFjY2Vzcy1pZGVudGl0eS9jbG91ZGZyb250L1wiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjFTM09yaWdpbjNEOUNBMEU5XCJcbiAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ3Rlc3QgY2xvdWRmcm9udCBvdmVycmlkZSBjbG91ZGZyb250IGxvZ2dpbmcgYnVja2V0ICcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgW3NvdXJjZUJ1Y2tldF0gPSBidWlsZFMzQnVja2V0KHN0YWNrLCB7fSk7XG4gIGNvbnN0IGxvZ0J1Y2tldCA9IG5ldyBCdWNrZXQoc3RhY2ssICdsb2dnaW5nYnVja2V0Jyk7XG5cbiAgY29uc3QgbXlwcm9wcyA9IHtcbiAgICBlbmFibGVMb2dnaW5nOiB0cnVlLFxuICAgIGxvZ0J1Y2tldFxuICB9O1xuXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JTMyhzdGFjaywgc291cmNlQnVja2V0LCBteXByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIENhY2hlUG9saWN5SWQ6IFwiNjU4MzI3ZWEtZjg5ZC00ZmFiLWE2M2QtN2U4ODYzOWU1OGY2XCIsXG4gICAgICAgIENvbXByZXNzOiB0cnVlLFxuICAgICAgICBGdW5jdGlvbkFzc29jaWF0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEV2ZW50VHlwZTogXCJ2aWV3ZXItcmVzcG9uc2VcIixcbiAgICAgICAgICAgIEZ1bmN0aW9uQVJOOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJTZXRIdHRwU2VjdXJpdHlIZWFkZXJzRUU5MzYxMTVcIixcbiAgICAgICAgICAgICAgICBcIkZ1bmN0aW9uQVJOXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjE3NkVDM0ExMlwiLFxuICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogXCJyZWRpcmVjdC10by1odHRwc1wiXG4gICAgICB9LFxuICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgIEh0dHBWZXJzaW9uOiBcImh0dHAyXCIsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIExvZ2dpbmc6IHtcbiAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwibG9nZ2luZ2J1Y2tldDZENzNCRDUzXCIsXG4gICAgICAgICAgICBcIlJlZ2lvbmFsRG9tYWluTmFtZVwiXG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgT3JpZ2luczogW1xuICAgICAgICB7XG4gICAgICAgICAgRG9tYWluTmFtZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJTM0J1Y2tldDA3NjgyOTkzXCIsXG4gICAgICAgICAgICAgIFwiUmVnaW9uYWxEb21haW5OYW1lXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIElkOiBcIkNsb3VkRnJvbnREaXN0cmlidXRpb25PcmlnaW4xNzZFQzNBMTJcIixcbiAgICAgICAgICBTM09yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgT3JpZ2luQWNjZXNzSWRlbnRpdHk6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICBcIm9yaWdpbi1hY2Nlc3MtaWRlbnRpdHkvY2xvdWRmcm9udC9cIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkNsb3VkRnJvbnREaXN0cmlidXRpb25PcmlnaW4xUzNPcmlnaW4zRDlDQTBFOVwiXG4gICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCd0ZXN0IGNsb3VkZnJvbnQgb3ZlcnJpZGUgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgW3NvdXJjZUJ1Y2tldF0gPSBidWlsZFMzQnVja2V0KHN0YWNrLCB7fSk7XG4gIGNvbnN0IHByb3BzOiBjbG91ZGZyb250LkRpc3RyaWJ1dGlvblByb3BzID0ge1xuICAgIGRlZmF1bHRCZWhhdmlvcjoge1xuICAgICAgb3JpZ2luOiBuZXcgb3JpZ2lucy5TM09yaWdpbihzb3VyY2VCdWNrZXQpLFxuICAgICAgdmlld2VyUHJvdG9jb2xQb2xpY3k6IGNsb3VkZnJvbnQuVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgICBhbGxvd2VkTWV0aG9kczogY2xvdWRmcm9udC5BbGxvd2VkTWV0aG9kcy5BTExPV19BTEwsXG4gICAgICBjYWNoZWRNZXRob2RzOiBjbG91ZGZyb250LkNhY2hlZE1ldGhvZHMuQ0FDSEVfR0VUX0hFQURfT1BUSU9OU1xuICAgIH0sXG4gIH07XG5cbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzKHN0YWNrLCBzb3VyY2VCdWNrZXQsIHByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEFsbG93ZWRNZXRob2RzOiBbXG4gICAgICAgICAgXCJHRVRcIixcbiAgICAgICAgICBcIkhFQURcIixcbiAgICAgICAgICBcIk9QVElPTlNcIixcbiAgICAgICAgICBcIlBVVFwiLFxuICAgICAgICAgIFwiUEFUQ0hcIixcbiAgICAgICAgICBcIlBPU1RcIixcbiAgICAgICAgICBcIkRFTEVURVwiXG4gICAgICAgIF0sXG4gICAgICAgIENhY2hlUG9saWN5SWQ6IFwiNjU4MzI3ZWEtZjg5ZC00ZmFiLWE2M2QtN2U4ODYzOWU1OGY2XCIsXG4gICAgICAgIENhY2hlZE1ldGhvZHM6IFtcbiAgICAgICAgICBcIkdFVFwiLFxuICAgICAgICAgIFwiSEVBRFwiLFxuICAgICAgICAgIFwiT1BUSU9OU1wiXG4gICAgICAgIF0sXG4gICAgICAgIENvbXByZXNzOiB0cnVlLFxuICAgICAgICBGdW5jdGlvbkFzc29jaWF0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEV2ZW50VHlwZTogXCJ2aWV3ZXItcmVzcG9uc2VcIixcbiAgICAgICAgICAgIEZ1bmN0aW9uQVJOOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJTZXRIdHRwU2VjdXJpdHlIZWFkZXJzRUU5MzYxMTVcIixcbiAgICAgICAgICAgICAgICBcIkZ1bmN0aW9uQVJOXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjE3NkVDM0ExMlwiLFxuICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogXCJyZWRpcmVjdC10by1odHRwc1wiXG4gICAgICB9LFxuICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgIEh0dHBWZXJzaW9uOiBcImh0dHAyXCIsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIExvZ2dpbmc6IHtcbiAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQzQzNFRkFBN1wiLFxuICAgICAgICAgICAgXCJSZWdpb25hbERvbWFpbk5hbWVcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIERvbWFpbk5hbWU6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiUzNCdWNrZXQwNzY4Mjk5M1wiLFxuICAgICAgICAgICAgICBcIlJlZ2lvbmFsRG9tYWluTmFtZVwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBJZDogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMTc2RUMzQTEyXCIsXG4gICAgICAgICAgUzNPcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpbkFjY2Vzc0lkZW50aXR5OiB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJvcmlnaW4tYWNjZXNzLWlkZW50aXR5L2Nsb3VkZnJvbnQvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMVMzT3JpZ2luM0Q5Q0EwRTlcIlxuICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBvdmVycmlkZSBjbG91ZGZyb250IHdpdGggY3VzdG9tIGNsb3VkZnJvbnQgZnVuY3Rpb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IFtzb3VyY2VCdWNrZXRdID0gYnVpbGRTM0J1Y2tldChzdGFjaywge30pO1xuXG4gIC8vIGN1c3RvbSBjbG91ZGZyb250IGZ1bmN0aW9uXG4gIGNvbnN0IGNsb3VkZnJvbnRGdW5jdGlvbiA9IG5ldyBjbG91ZGZyb250LkZ1bmN0aW9uKHN0YWNrLCBcIk15RnVuY3Rpb25cIiwge1xuICAgIGNvZGU6IGNsb3VkZnJvbnQuRnVuY3Rpb25Db2RlLmZyb21JbmxpbmUoXCJleHBvcnRzLmhhbmRsZXIgPSAoZXZlbnQsIGNvbnRleHQsIGNhbGxiYWNrKSA9PiB7fVwiKVxuICB9KTtcblxuICBDbG91ZEZyb250RGlzdHJpYnV0aW9uRm9yUzMoc3RhY2ssIHNvdXJjZUJ1Y2tldCwge1xuICAgIGRlZmF1bHRCZWhhdmlvcjoge1xuICAgICAgZnVuY3Rpb25Bc3NvY2lhdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIGV2ZW50VHlwZTogY2xvdWRmcm9udC5GdW5jdGlvbkV2ZW50VHlwZS5WSUVXRVJfUkVTUE9OU0UsXG4gICAgICAgICAgZnVuY3Rpb246IGNsb3VkZnJvbnRGdW5jdGlvblxuICAgICAgICB9XG4gICAgICBdLFxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIENhY2hlUG9saWN5SWQ6IFwiNjU4MzI3ZWEtZjg5ZC00ZmFiLWE2M2QtN2U4ODYzOWU1OGY2XCIsXG4gICAgICAgIENvbXByZXNzOiB0cnVlLFxuICAgICAgICBGdW5jdGlvbkFzc29jaWF0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEV2ZW50VHlwZTogXCJ2aWV3ZXItcmVzcG9uc2VcIixcbiAgICAgICAgICAgIEZ1bmN0aW9uQVJOOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJNeUZ1bmN0aW9uM0JBQTcyRDFcIixcbiAgICAgICAgICAgICAgICBcIkZ1bmN0aW9uQVJOXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjE3NkVDM0ExMlwiLFxuICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogXCJyZWRpcmVjdC10by1odHRwc1wiXG4gICAgICB9LFxuICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgIEh0dHBWZXJzaW9uOiBcImh0dHAyXCIsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIExvZ2dpbmc6IHtcbiAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQzQzNFRkFBN1wiLFxuICAgICAgICAgICAgXCJSZWdpb25hbERvbWFpbk5hbWVcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIERvbWFpbk5hbWU6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiUzNCdWNrZXQwNzY4Mjk5M1wiLFxuICAgICAgICAgICAgICBcIlJlZ2lvbmFsRG9tYWluTmFtZVwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBJZDogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMTc2RUMzQTEyXCIsXG4gICAgICAgICAgUzNPcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpbkFjY2Vzc0lkZW50aXR5OiB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJvcmlnaW4tYWNjZXNzLWlkZW50aXR5L2Nsb3VkZnJvbnQvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMVMzT3JpZ2luM0Q5Q0EwRTlcIlxuICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBvdmVycmlkZSBjbG91ZGZyb250IHJlcGxhY2UgY3VzdG9tIGxhbWJkYUBlZGdlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBbc291cmNlQnVja2V0XSA9IGJ1aWxkUzNCdWNrZXQoc3RhY2ssIHt9KTtcblxuICAvLyBjdXN0b20gbGFtYmRhQGVkZyBmdW5jdGlvblxuICBjb25zdCBoYW5kbGVyID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ1NvbWVIYW5kbGVyJywge1xuICAgIGZ1bmN0aW9uTmFtZTogJ1NvbWVIYW5kbGVyJyxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gIH0pO1xuXG4gIGNvbnN0IGhhbmRsZXJWZXJzaW9uID0gbmV3IGxhbWJkYS5WZXJzaW9uKHN0YWNrLCAnU29tZUhhbmRsZXJWZXJzaW9uJywge1xuICAgIGxhbWJkYTogaGFuZGxlcixcbiAgfSk7XG5cbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzKHN0YWNrLCBzb3VyY2VCdWNrZXQsIHtcbiAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgIGVkZ2VMYW1iZGFzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBldmVudFR5cGU6IExhbWJkYUVkZ2VFdmVudFR5cGUuVklFV0VSX1JFUVVFU1QsXG4gICAgICAgICAgaW5jbHVkZUJvZHk6IGZhbHNlLFxuICAgICAgICAgIGZ1bmN0aW9uVmVyc2lvbjogaGFuZGxlclZlcnNpb24sXG4gICAgICAgIH1cbiAgICAgIF1cbiAgICB9XG4gIH0sXG4gIGZhbHNlKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKFwiQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb25cIiwge1xuICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgQ2FjaGVQb2xpY3lJZDogXCI2NTgzMjdlYS1mODlkLTRmYWItYTYzZC03ZTg4NjM5ZTU4ZjZcIixcbiAgICAgICAgQ29tcHJlc3M6IHRydWUsXG4gICAgICAgIExhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25zOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgRXZlbnRUeXBlOiBcInZpZXdlci1yZXF1ZXN0XCIsXG4gICAgICAgICAgICBJbmNsdWRlQm9keTogZmFsc2UsXG4gICAgICAgICAgICBMYW1iZGFGdW5jdGlvbkFSTjoge1xuICAgICAgICAgICAgICBSZWY6IFwiU29tZUhhbmRsZXJWZXJzaW9uREE5ODZFNDFcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6IFwiQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk9yaWdpbjE3NkVDM0ExMlwiLFxuICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogXCJyZWRpcmVjdC10by1odHRwc1wiXG4gICAgICB9LFxuICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgIEh0dHBWZXJzaW9uOiBcImh0dHAyXCIsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIExvZ2dpbmc6IHtcbiAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQzQzNFRkFBN1wiLFxuICAgICAgICAgICAgXCJSZWdpb25hbERvbWFpbk5hbWVcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIERvbWFpbk5hbWU6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiUzNCdWNrZXQwNzY4Mjk5M1wiLFxuICAgICAgICAgICAgICBcIlJlZ2lvbmFsRG9tYWluTmFtZVwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBJZDogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMTc2RUMzQTEyXCIsXG4gICAgICAgICAgUzNPcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpbkFjY2Vzc0lkZW50aXR5OiB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJvcmlnaW4tYWNjZXNzLWlkZW50aXR5L2Nsb3VkZnJvbnQvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJDbG91ZEZyb250RGlzdHJpYnV0aW9uT3JpZ2luMVMzT3JpZ2luM0Q5Q0EwRTlcIlxuICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBjbG91ZGZyb250IG92ZXJyaWRlIGNsb3VkZnJvbnQgY3VzdG9tIGRvbWFpbiBuYW1lcyAnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IFtzb3VyY2VCdWNrZXRdID0gYnVpbGRTM0J1Y2tldChzdGFjaywge30pO1xuICBjb25zdCBjZXJ0aWZpY2F0ZSA9IGFjbS5DZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oc3RhY2ssICdDZXJ0JywgJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTExMTIyMjItMzMzMy0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyk7XG5cbiAgY29uc3QgbXlwcm9wcyA9IHtcbiAgICBkb21haW5OYW1lczogWydteWRvbWFpbnMnXSxcbiAgICBjZXJ0aWZpY2F0ZVxuICB9O1xuXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JTMyhzdGFjaywgc291cmNlQnVja2V0LCBteXByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIEFsaWFzZXM6IFtcbiAgICAgICAgXCJteWRvbWFpbnNcIlxuICAgICAgXSxcbiAgICB9XG4gIH0pO1xufSk7XG4iXX0=