"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployES(stack, domainName, cfnDomainProps, lambdaRoleARN) {
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    const cognitoAuthorizedRole = defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    if (lambdaRoleARN) {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
            serviceRoleARN: lambdaRoleARN
        }, cfnDomainProps);
    }
    else {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn
        }, cfnDomainProps);
    }
}
test('Test override SnapshotOptions for buildElasticSearch', () => {
    const stack = new aws_cdk_lib_1.Stack();
    deployES(stack, 'test-domain', {
        snapshotOptions: {
            automatedSnapshotStartHour: 5
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 5
        }
    });
});
test('Test override ES version for buildElasticSearch', () => {
    const stack = new aws_cdk_lib_1.Stack();
    deployES(stack, 'test-domain', {
        elasticsearchVersion: '7.0'
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "7.0",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Test ES with lambdaRoleARN', () => {
    const stack = new aws_cdk_lib_1.Stack();
    deployES(stack, 'test-domain', {}, 'arn:aws:us-east-1:mylambdaRoleARN');
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: [
                            {
                                "Fn::GetAtt": [
                                    "CognitoAuthorizedRole14E74FE0",
                                    "Arn"
                                ]
                            },
                            "arn:aws:us-east-1:mylambdaRoleARN"
                        ]
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Count ES CW Alarms', () => {
    const stack = new aws_cdk_lib_1.Stack();
    deployES(stack, 'test-domain');
    const cwList = defaults.buildElasticSearchCWAlarms(stack);
    expect(cwList.length).toEqual(9);
});
//# sourceMappingURL=data:application/json;base64,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