"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const glue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const lambda = require("aws-cdk-lib/aws-lambda");
const mediastore = require("aws-cdk-lib/aws-mediastore");
const s3 = require("aws-cdk-lib/aws-s3");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const mediastore_defaults_1 = require("../lib/mediastore-defaults");
const sagemaker_helper_1 = require("../lib/sagemaker-helper");
const test_helper_1 = require("./test-helper");
test('Test with valid props', () => {
    const props = {};
    defaults.CheckProps(props);
});
test('Test fail DynamoDB table check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingTableObj: new dynamodb.Table(stack, 'placeholder', defaults.DefaultTableProps),
        dynamoTableProps: defaults.DefaultTableProps,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
test("Test fail Lambda function check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
test("Test fail SQS Queue check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'placeholder', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps or existingQueueObj, but not both.\n');
});
test('Test fail Dead Letter Queue check', () => {
    const props = {
        deployDeadLetterQueue: false,
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and undefined deadLetterQueueProps', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and deadLetterQueueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false and deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test("Test fail MediaStore container check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, "placeholder", mediastore_defaults_1.MediaStoreContainerProps());
    const props = {
        mediaStoreContainerProps: mediastore_defaults_1.MediaStoreContainerProps(),
        existingMediaStoreContainerObj: mediaStoreContainer,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n");
});
test('Test fail Kinesis stream check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const stream = new kinesis.Stream(stack, 'placeholder', {});
    const props = {
        existingStreamObj: stream,
        kinesisStreamProps: {}
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
test('Test fail S3 check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingBucketObj: test_helper_1.CreateScrapBucket(stack, {}),
        bucketProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
test('Test fail SNS topic check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check with bad topic attribute name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail Glue job check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const _jobRole = new iam.Role(stack, 'CustomETLJobRole', {
        assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
    });
    const jobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: new s3.Bucket(stack, 'ScriptBucket').bucketArn,
        },
        role: new iam.Role(stack, 'JobRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
        }).roleArn
    }, 'testETLJob', {});
    const job = new glue.CfnJob(stack, 'placeholder', jobProps);
    const props = {
        glueJobProps: jobProps,
        existingGlueJob: job
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide glueJobProps or existingGlueJob, but not both.\n');
});
test('Test fail SageMaker endpoint check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // Build Sagemaker Inference Endpoint
    const modelProps = {
        primaryContainer: {
            image: "<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest",
            modelDataUrl: "s3://<bucket-name>/<prefix>/model.tar.gz",
        },
    };
    const [endpoint] = sagemaker_helper_1.BuildSagemakerEndpoint(stack, { modelProps });
    const props = {
        existingSagemakerEndpointObj: endpoint,
        endpointProps: {
            endpointConfigName: 'placeholder'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
test('Test fail Secret check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail encryption key check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        enableKeyRotation: false
    });
    const props = {
        encryptionKey: key,
        encryptionKeyProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Vpc check with deployVpc', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deployVpc: true,
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail Vpc check with vpcProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        vpcProps: defaults.DefaultPublicPrivateVpcProps(),
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail multiple failures message', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n' +
        'Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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