"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const kms = require("aws-cdk-lib/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test('test s3Bucket override versioningConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        versioned: false
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-verioning', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    });
});
test('test s3Bucket override bucketEncryption', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        encryption: s3.BucketEncryption.KMS,
        encryptionKey: new kms.Key(stack, 'mykeyid')
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-encryption', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            "Fn::GetAtt": [
                                "mykeyidFA4203B0",
                                "Arn"
                            ]
                        },
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        },
    });
});
test('test s3Bucket override publicAccessBlockConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        blockPublicAccess: s3.BlockPublicAccess.BLOCK_ACLS
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-publicAccessBlock', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            IgnorePublicAcls: true
        },
    });
});
test('test s3Bucket add lifecycleConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        lifecycleRules: [{
                expiration: aws_cdk_lib_1.Duration.days(365)
            }]
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-lifecycle', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    Status: 'Enabled',
                    ExpirationInDays: 365,
                }
            ]
        }
    });
});
test('test s3Bucket override serverAccessLogsBucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myLoggingBucket = new s3.Bucket(stack, 'MyS3LoggingBucket', defaults.DefaultS3Props());
    const myS3Props = defaults.DefaultS3Props(myLoggingBucket);
    defaults.buildS3Bucket(stack, {
        bucketProps: myS3Props
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "MyS3LoggingBucket119BE896"
            }
        }
    });
});
test('test createAlbLoggingBucket()', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.createAlbLoggingBucket(stack, 'test-bucket', {
        bucketName: 'test-name'
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketName: 'test-name'
    });
});
//# sourceMappingURL=data:application/json;base64,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