"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateIntegStackName = exports.CreateScrapBucket = void 0;
// Imports
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const utils_1 = require("../lib/utils");
const path = require("path");
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
//# sourceMappingURL=data:application/json;base64,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