"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
// --------------------------------------------------------------
// Test minimal Isolated deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultIsolatedVpcProps(),
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::EC2::Subnet', 2);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::EC2::InternetGateway', 0);
});
// --------------------------------------------------------------
// Test deployment w/ user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/16',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/16',
        EnableDnsHostnames: false,
        EnableDnsSupport: false,
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ construct provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/16',
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct and user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ construct and user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '10.0.0.0/16',
        },
        constructVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/16',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/16',
        EnableDnsHostnames: false,
        EnableDnsSupport: false,
    });
});
// --------------------------------------------------------------
// Test priority of default, user and construct properties
// --------------------------------------------------------------
test('Test deployment w/ construct and user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const v = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '10.0.0.0/16',
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'isolated',
                    subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
                },
            ],
        },
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, v, defaults.ServiceEndpointTypes.SQS);
    // Expect 2 isolated subnets (usual error condition is 2 public/2 private)
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
});
// --------------------------------------------------------------
// Test deployment w/ existing VPC provided
// --------------------------------------------------------------
test('Test deployment w/ existing VPC provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testExistingVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/16',
        },
    });
    const newVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        existingVpc: testExistingVpc,
    });
    expect(newVpc).toBe(testExistingVpc);
});
// --------------------------------------------------------------
// Test adding Gateway Endpoint
// --------------------------------------------------------------
test('Test adding Gateway Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.DYNAMODB);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SQS);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    expect(stack).toCountResources('AWS::EC2::VPCEndpoint', 3);
});
// --------------------------------------------------------------
// Test adding Interface Endpoint
// --------------------------------------------------------------
test('Test adding Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
// --------------------------------------------------------------
// Test adding SAGEMAKER_RUNTIME Interface Endpoint
// --------------------------------------------------------------
test('Test adding SAGEMAKER_RUNTIME Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
// --------------------------------------------------------------
// Test adding a second Endpoint of same service
// --------------------------------------------------------------
test('Test adding a second Endpoint of same service', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toCountResources('AWS::EC2::VPCEndpoint', 1);
});
// --------------------------------------------------------------
// Test adding bad Endpoint
// --------------------------------------------------------------
test('Test adding bad Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    const app = () => {
        vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, 'string');
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test adding Events Interface Endpoint
// --------------------------------------------------------------
test('Test adding Events Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.EVENTS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
//# sourceMappingURL=data:application/json;base64,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