"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("..");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const __1 = require("..");
// --------------------------------------------------------------
// Test construct with default props
// --------------------------------------------------------------
test('Test construct with default props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    defaults.buildWebacl(stack, 'REGIONAL', {});
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        Scope: "REGIONAL",
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesBotControlRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesBotControlRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesBotControlRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesKnownBadInputsRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesKnownBadInputsRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesKnownBadInputsRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 2,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAnonymousIpList",
                OverrideAction: {
                    None: {}
                },
                Priority: 3,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAnonymousIpList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAnonymousIpList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAmazonIpReputationList",
                OverrideAction: {
                    None: {}
                },
                Priority: 4,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAmazonIpReputationList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAmazonIpReputationList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAdminProtectionRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 5,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAdminProtectionRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAdminProtectionRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesSQLiRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 6,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesSQLiRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesSQLiRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::WAFv2::WebACL', 1);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::WAFv2::WebACLAssociation', 0);
});
// --------------------------------------------------------------
// Test deployment w/ user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    const props = {
        scope: 'CLOUDFRONT',
        defaultAction: {
            allow: {}
        },
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
        rules: [
            defaults.wrapManagedRuleSet("AWSManagedRulesCommonRuleSet", "AWS", 0),
            defaults.wrapManagedRuleSet("AWSManagedRulesWordPressRuleSet", "AWS", 1),
        ]
    };
    defaults.buildWebacl(stack, 'CLOUDFRONT', {
        webaclProps: props
    });
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        Scope: "CLOUDFRONT",
        VisibilityConfig: {
            CloudWatchMetricsEnabled: false,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesWordPressRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesWordPressRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesWordPressRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing WAF web ACL provided
// --------------------------------------------------------------
test('Test deployment w/ existing WAF web ACL provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    const testWaf = __1.buildWebacl(stack, 'CLOUDFRONT', {});
    const newWaf = defaults.buildWebacl(stack, 'CLOUDFRONT', {
        existingWebaclObj: testWaf
    });
    expect(newWaf).toBe(testWaf);
});
//# sourceMappingURL=data:application/json;base64,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