"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ECRDeployment = exports.S3ArchiveName = exports.DockerImageName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const child_process = require("child_process");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
function getCode() {
    if (!(process.env.CI || process.env.NO_PREBUILT_LAMBDA)) {
        try {
            console.log('Try to get prebuilt lambda');
            const installScript = path.join(__dirname, '../lambda/install.js');
            const prebuiltPath = path.join(__dirname, '../lambda/out');
            child_process.execSync(`${process.argv0} ${installScript} ${prebuiltPath}`);
            return aws_cdk_lib_1.aws_lambda.Code.fromAsset(prebuiltPath);
        }
        catch (err) {
            console.warn(`Can not get prebuilt lambda: ${err}`);
        }
    }
    console.log('Build lambda from scratch');
    return aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(path.join(__dirname, '../lambda'));
}
/**
 * @stability stable
 */
class DockerImageName {
    /**
     * @stability stable
     */
    constructor(name, creds) {
        this.name = name;
        this.creds = creds;
    }
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri() { return `docker://${this.name}`; }
}
exports.DockerImageName = DockerImageName;
_a = JSII_RTTI_SYMBOL_1;
DockerImageName[_a] = { fqn: "cdk-ecr-deployment.DockerImageName", version: "2.1.0" };
/**
 * @stability stable
 */
class S3ArchiveName {
    /**
     * @stability stable
     */
    constructor(p, ref, creds) {
        this.creds = creds;
        this.name = p;
        if (ref) {
            this.name += ':' + ref;
        }
    }
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri() { return `s3://${this.name}`; }
}
exports.S3ArchiveName = S3ArchiveName;
_b = JSII_RTTI_SYMBOL_1;
S3ArchiveName[_b] = { fqn: "cdk-ecr-deployment.S3ArchiveName", version: "2.1.0" };
/**
 * @stability stable
 */
class ECRDeployment extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id);
        const memoryLimit = (_d = props.memoryLimit) !== null && _d !== void 0 ? _d : 512;
        const handler = new aws_cdk_lib_1.aws_lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(memoryLimit),
            code: getCode(),
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.GO_1_X,
            handler: 'main',
            environment: props.environment,
            lambdaPurpose: 'Custom::CDKECRDeployment',
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: props.role,
            memorySize: memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        handlerRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:GetRepositoryPolicy',
                'ecr:DescribeRepositories',
                'ecr:ListImages',
                'ecr:DescribeImages',
                'ecr:BatchGetImage',
                'ecr:ListTagsForResource',
                'ecr:DescribeImageScanFindings',
                'ecr:InitiateLayerUpload',
                'ecr:UploadLayerPart',
                'ecr:CompleteLayerUpload',
                'ecr:PutImage',
            ],
            resources: ['*'],
        }));
        handlerRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
            ],
            resources: ['*'],
        }));
        new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SrcImage: props.src.uri,
                SrcCreds: props.src.creds,
                DestImage: props.dest.uri,
                DestCreds: props.dest.creds,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = 'bd07c930-edb9-4112-a20f-03f096f53666';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (aws_cdk_lib_1.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.ECRDeployment = ECRDeployment;
_c = JSII_RTTI_SYMBOL_1;
ECRDeployment[_c] = { fqn: "cdk-ecr-deployment.ECRDeployment", version: "2.1.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDO0FBR3RDLCtDQUErQztBQUMvQyw2QkFBNkI7QUFDN0IsNkNBQW9IO0FBQ3BILDJDQUF1QztBQWlDdkMsU0FBUyxPQUFPO0lBQ2QsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1FBQ3ZELElBQUk7WUFDRixPQUFPLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7WUFFMUMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUNuRSxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUMzRCxhQUFhLENBQUMsUUFBUSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssSUFBSSxhQUFhLElBQUksWUFBWSxFQUFFLENBQUMsQ0FBQztZQUU1RSxPQUFPLHdCQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM1QztRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1osT0FBTyxDQUFDLElBQUksQ0FBQyxnQ0FBZ0MsR0FBRyxFQUFFLENBQUMsQ0FBQztTQUNyRDtLQUNGO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBRXpDLE9BQU8sd0JBQU0sQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUM7QUFDeEUsQ0FBQzs7OztBQUVELE1BQWEsZUFBZTs7OztJQUMxQixZQUEyQixJQUFZLEVBQVMsS0FBYztRQUFuQyxTQUFJLEdBQUosSUFBSSxDQUFRO1FBQVMsVUFBSyxHQUFMLEtBQUssQ0FBUztJQUFJLENBQUM7Ozs7Ozs7O0lBQ25FLElBQVcsR0FBRyxLQUFhLE9BQU8sWUFBWSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDOztBQUY5RCwwQ0FHQzs7Ozs7O0FBRUQsTUFBYSxhQUFhOzs7O0lBRXhCLFlBQW1CLENBQVMsRUFBRSxHQUFZLEVBQVMsS0FBYztRQUFkLFVBQUssR0FBTCxLQUFLLENBQVM7UUFDL0QsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7UUFDZCxJQUFJLEdBQUcsRUFBRTtZQUNQLElBQUksQ0FBQyxJQUFJLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQztTQUN4QjtJQUNILENBQUM7Ozs7Ozs7O0lBQ0QsSUFBVyxHQUFHLEtBQWEsT0FBTyxRQUFRLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7O0FBUjFELHNDQVNDOzs7Ozs7QUFFRCxNQUFhLGFBQWMsU0FBUSxzQkFBUzs7OztJQUMxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCOztRQUNqRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sV0FBVyxTQUFHLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEdBQUcsQ0FBQztRQUM3QyxNQUFNLE9BQU8sR0FBRyxJQUFJLHdCQUFNLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO1lBQzFFLElBQUksRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsV0FBVyxDQUFDO1lBQzNDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDZixPQUFPLEVBQUUsd0JBQU0sQ0FBQyxPQUFPLENBQUMsTUFBTTtZQUM5QixPQUFPLEVBQUUsTUFBTTtZQUNmLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixhQUFhLEVBQUUsMEJBQTBCO1lBQ3pDLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLFVBQVUsRUFBRSxXQUFXO1lBQ3ZCLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtTQUM3QixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FBRTtRQUU3RixXQUFXLENBQUMsb0JBQW9CLENBQzlCLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLDJCQUEyQjtnQkFDM0IsaUNBQWlDO2dCQUNqQyw0QkFBNEI7Z0JBQzVCLHlCQUF5QjtnQkFDekIsMEJBQTBCO2dCQUMxQixnQkFBZ0I7Z0JBQ2hCLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQix5QkFBeUI7Z0JBQ3pCLCtCQUErQjtnQkFDL0IseUJBQXlCO2dCQUN6QixxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIsY0FBYzthQUNmO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBQ04sV0FBVyxDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkQsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLGNBQWM7YUFDZjtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksNEJBQWMsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDekMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxXQUFXO1lBQ2pDLFlBQVksRUFBRSw2QkFBNkI7WUFDM0MsVUFBVSxFQUFFO2dCQUNWLFFBQVEsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLEdBQUc7Z0JBQ3ZCLFFBQVEsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLEtBQUs7Z0JBQ3pCLFNBQVMsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUc7Z0JBQ3pCLFNBQVMsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUs7YUFDNUI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sbUJBQW1CLENBQUMsV0FBb0I7UUFDOUMsSUFBSSxJQUFJLEdBQUcsc0NBQXNDLENBQUM7UUFFbEQsMEVBQTBFO1FBQzFFLDJFQUEyRTtRQUMzRSw0Q0FBNEM7UUFDNUMsSUFBSSxXQUFXLEVBQUU7WUFDZixJQUFJLG1CQUFLLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxFQUFFO2dCQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7YUFDdEk7WUFFRCxJQUFJLElBQUksSUFBSSxXQUFXLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQztTQUN6QztRQUVELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7QUE3RUgsc0NBOEVDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuXG5cbmltcG9ydCAqIGFzIGNoaWxkX3Byb2Nlc3MgZnJvbSAnY2hpbGRfcHJvY2Vzcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgYXdzX2VjMiBhcyBlYzIsIGF3c19pYW0gYXMgaWFtLCBhd3NfbGFtYmRhIGFzIGxhbWJkYSwgRHVyYXRpb24sIEN1c3RvbVJlc291cmNlLCBUb2tlbiB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEVDUkRlcGxveW1lbnRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3JjOiBJSW1hZ2VOYW1lO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXN0OiBJSW1hZ2VOYW1lO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtZW1vcnlMaW1pdD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSUltYWdlTmFtZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHVyaTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY3JlZHM/OiBzdHJpbmc7XG59XG5cbmZ1bmN0aW9uIGdldENvZGUoKTogbGFtYmRhLkFzc2V0Q29kZSB7XG4gIGlmICghKHByb2Nlc3MuZW52LkNJIHx8IHByb2Nlc3MuZW52Lk5PX1BSRUJVSUxUX0xBTUJEQSkpIHtcbiAgICB0cnkge1xuICAgICAgY29uc29sZS5sb2coJ1RyeSB0byBnZXQgcHJlYnVpbHQgbGFtYmRhJyk7XG5cbiAgICAgIGNvbnN0IGluc3RhbGxTY3JpcHQgPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGFtYmRhL2luc3RhbGwuanMnKTtcbiAgICAgIGNvbnN0IHByZWJ1aWx0UGF0aCA9IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9sYW1iZGEvb3V0Jyk7XG4gICAgICBjaGlsZF9wcm9jZXNzLmV4ZWNTeW5jKGAke3Byb2Nlc3MuYXJndjB9ICR7aW5zdGFsbFNjcmlwdH0gJHtwcmVidWlsdFBhdGh9YCk7XG5cbiAgICAgIHJldHVybiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocHJlYnVpbHRQYXRoKTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUud2FybihgQ2FuIG5vdCBnZXQgcHJlYnVpbHQgbGFtYmRhOiAke2Vycn1gKTtcbiAgICB9XG4gIH1cblxuICBjb25zb2xlLmxvZygnQnVpbGQgbGFtYmRhIGZyb20gc2NyYXRjaCcpO1xuXG4gIHJldHVybiBsYW1iZGEuQ29kZS5mcm9tRG9ja2VyQnVpbGQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xhbWJkYScpKTtcbn1cblxuZXhwb3J0IGNsYXNzIERvY2tlckltYWdlTmFtZSBpbXBsZW1lbnRzIElJbWFnZU5hbWUge1xuICBwdWJsaWMgY29uc3RydWN0b3IocHJpdmF0ZSBuYW1lOiBzdHJpbmcsIHB1YmxpYyBjcmVkcz86IHN0cmluZykgeyB9XG4gIHB1YmxpYyBnZXQgdXJpKCk6IHN0cmluZyB7IHJldHVybiBgZG9ja2VyOi8vJHt0aGlzLm5hbWV9YDsgfVxufVxuXG5leHBvcnQgY2xhc3MgUzNBcmNoaXZlTmFtZSBpbXBsZW1lbnRzIElJbWFnZU5hbWUge1xuICBwcml2YXRlIG5hbWU6IHN0cmluZztcbiAgcHVibGljIGNvbnN0cnVjdG9yKHA6IHN0cmluZywgcmVmPzogc3RyaW5nLCBwdWJsaWMgY3JlZHM/OiBzdHJpbmcpIHtcbiAgICB0aGlzLm5hbWUgPSBwO1xuICAgIGlmIChyZWYpIHtcbiAgICAgIHRoaXMubmFtZSArPSAnOicgKyByZWY7XG4gICAgfVxuICB9XG4gIHB1YmxpYyBnZXQgdXJpKCk6IHN0cmluZyB7IHJldHVybiBgczM6Ly8ke3RoaXMubmFtZX1gOyB9XG59XG5cbmV4cG9ydCBjbGFzcyBFQ1JEZXBsb3ltZW50IGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEVDUkRlcGxveW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgY29uc3QgbWVtb3J5TGltaXQgPSBwcm9wcy5tZW1vcnlMaW1pdCA/PyA1MTI7XG4gICAgY29uc3QgaGFuZGxlciA9IG5ldyBsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24odGhpcywgJ0N1c3RvbVJlc291cmNlSGFuZGxlcicsIHtcbiAgICAgIHV1aWQ6IHRoaXMucmVuZGVyU2luZ2xldG9uVXVpZChtZW1vcnlMaW1pdCksXG4gICAgICBjb2RlOiBnZXRDb2RlKCksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5HT18xX1gsXG4gICAgICBoYW5kbGVyOiAnbWFpbicsXG4gICAgICBlbnZpcm9ubWVudDogcHJvcHMuZW52aXJvbm1lbnQsXG4gICAgICBsYW1iZGFQdXJwb3NlOiAnQ3VzdG9tOjpDREtFQ1JEZXBsb3ltZW50JyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgcm9sZTogcHJvcHMucm9sZSxcbiAgICAgIG1lbW9yeVNpemU6IG1lbW9yeUxpbWl0LFxuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy52cGNTdWJuZXRzLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaGFuZGxlclJvbGUgPSBoYW5kbGVyLnJvbGU7XG4gICAgaWYgKCFoYW5kbGVyUm9sZSkgeyB0aHJvdyBuZXcgRXJyb3IoJ2xhbWJkYS5TaW5nbGV0b25GdW5jdGlvbiBzaG91bGQgaGF2ZSBjcmVhdGVkIGEgUm9sZScpOyB9XG5cbiAgICBoYW5kbGVyUm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nLFxuICAgICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgICAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLFxuICAgICAgICAgICdlY3I6R2V0UmVwb3NpdG9yeVBvbGljeScsXG4gICAgICAgICAgJ2VjcjpEZXNjcmliZVJlcG9zaXRvcmllcycsXG4gICAgICAgICAgJ2VjcjpMaXN0SW1hZ2VzJyxcbiAgICAgICAgICAnZWNyOkRlc2NyaWJlSW1hZ2VzJyxcbiAgICAgICAgICAnZWNyOkJhdGNoR2V0SW1hZ2UnLFxuICAgICAgICAgICdlY3I6TGlzdFRhZ3NGb3JSZXNvdXJjZScsXG4gICAgICAgICAgJ2VjcjpEZXNjcmliZUltYWdlU2NhbkZpbmRpbmdzJyxcbiAgICAgICAgICAnZWNyOkluaXRpYXRlTGF5ZXJVcGxvYWQnLFxuICAgICAgICAgICdlY3I6VXBsb2FkTGF5ZXJQYXJ0JyxcbiAgICAgICAgICAnZWNyOkNvbXBsZXRlTGF5ZXJVcGxvYWQnLFxuICAgICAgICAgICdlY3I6UHV0SW1hZ2UnLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgfSkpO1xuICAgIGhhbmRsZXJSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3MzOkdldE9iamVjdCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0N1c3RvbVJlc291cmNlJywge1xuICAgICAgc2VydmljZVRva2VuOiBoYW5kbGVyLmZ1bmN0aW9uQXJuLFxuICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50JyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgU3JjSW1hZ2U6IHByb3BzLnNyYy51cmksXG4gICAgICAgIFNyY0NyZWRzOiBwcm9wcy5zcmMuY3JlZHMsXG4gICAgICAgIERlc3RJbWFnZTogcHJvcHMuZGVzdC51cmksXG4gICAgICAgIERlc3RDcmVkczogcHJvcHMuZGVzdC5jcmVkcyxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclNpbmdsZXRvblV1aWQobWVtb3J5TGltaXQ/OiBudW1iZXIpIHtcbiAgICBsZXQgdXVpZCA9ICdiZDA3YzkzMC1lZGI5LTQxMTItYTIwZi0wM2YwOTZmNTM2NjYnO1xuXG4gICAgLy8gaWYgdXNlciBzcGVjaWZ5IGEgY3VzdG9tIG1lbW9yeSBsaW1pdCwgZGVmaW5lIGFub3RoZXIgc2luZ2xldG9uIGhhbmRsZXJcbiAgICAvLyB3aXRoIHRoaXMgY29uZmlndXJhdGlvbi4gb3RoZXJ3aXNlLCBpdCB3b24ndCBiZSBwb3NzaWJsZSB0byB1c2UgbXVsdGlwbGVcbiAgICAvLyBjb25maWd1cmF0aW9ucyBzaW5jZSB3ZSBoYXZlIGEgc2luZ2xldG9uLlxuICAgIGlmIChtZW1vcnlMaW1pdCkge1xuICAgICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChtZW1vcnlMaW1pdCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5cXCd0IHVzZSB0b2tlbnMgd2hlbiBzcGVjaWZ5aW5nIFwibWVtb3J5TGltaXRcIiBzaW5jZSB3ZSB1c2UgaXQgdG8gaWRlbnRpZnkgdGhlIHNpbmdsZXRvbiBjdXN0b20gcmVzb3VyY2UgaGFuZGxlcicpO1xuICAgICAgfVxuXG4gICAgICB1dWlkICs9IGAtJHttZW1vcnlMaW1pdC50b1N0cmluZygpfU1pQmA7XG4gICAgfVxuXG4gICAgcmV0dXJuIHV1aWQ7XG4gIH1cbn1cbiJdfQ==