"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const lib_1 = require("../lib");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A Private DNS configuration for a VPC endpoint service.
 */
class VpcEndpointServiceDomainName extends core_2.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_VpcEndpointServiceDomainNameProps(props);
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        this.domainName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = this.domainName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, this.domainName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retrieveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retrieveNameValuePairAction,
            onUpdate: retrieveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
_a = JSII_RTTI_SYMBOL_1;
VpcEndpointServiceDomainName[_a] = { fqn: "@aws-cdk/aws-route53.VpcEndpointServiceDomainName", version: "1.148.0" };
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
/**
 * Hash a string
 */
function hashcode(s) {
    const hash = crypto.createHash('md5');
    hash.update(s);
    return hash.digest('hex');
}
;
//# sourceMappingURL=data:application/json;base64,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