#!/usr/bin/env python3
# --------------------------------------------------------------------------------------
# SPDX-FileCopyrightText: 2021 Magenta ApS <https://magenta.dk>
# SPDX-License-Identifier: MPL-2.0
# --------------------------------------------------------------------------------------
# --------------------------------------------------------------------------------------
# Imports
# --------------------------------------------------------------------------------------
from typing import Literal
from typing import Optional
from uuid import UUID

from pydantic import Field

from .._shared import EngagementRef
from .._shared import LeaveType
from .._shared import MOBase
from .._shared import PersonRef
from .._shared import Validity

# --------------------------------------------------------------------------------------
# Role
# --------------------------------------------------------------------------------------


class LeaveBase(MOBase):
    """A MO leave object."""

    type_: str = Field("leave", alias="type", description="The object type.")
    validity: Validity = Field(description="Validity of the leave object.")


class LeaveRead(LeaveBase):
    """A MO leave read object."""

    person: UUID = Field(description="UUID of the person related to the leave.")
    leave_type: UUID = Field(description="UUID of the leave type klasse.")
    engagement: Optional[UUID] = Field(
        description="UUID of the engagement related to the leave."
    )


class LeaveWrite(LeaveBase):
    """A MO leave write object."""

    leave_type: LeaveType = Field(description="Reference to the leave type klasse.")
    person: PersonRef = Field(
        description="Reference to the person for which the leave should be created."
    )
    engagement: Optional[EngagementRef] = Field(
        description="Reference to the engagement for which the leave should be "
        "created."
    )


class Leave(MOBase):
    type_: Literal["leave"] = Field(
        "leave", alias="type", description="The object type."
    )
    leave_type: LeaveType = Field(description="Reference to the leave type facet")
    person: PersonRef = Field(
        description="Reference to the person object for which the role should "
        "be created."
    )
    engagement: Optional[EngagementRef] = Field(
        description="Reference to the engagement for which the role should "
        "be created."
    )
    validity: Validity = Field(description="Validity of the created role object.")
