import asyncio
import os

from inferelator_prior.processor.utils import file_path_abs
from inferelator_prior import KALLISTO_EXECUTABLE_PATH

KALLISTO_OUT_FILE = "abundance.tsv"

KALLISTO_COUNT_COL = "est_counts"
KALLISTO_TPM_COL = "tpm"


def kallisto_align_fastqs(srr_ids, fastq_file_names, reference_genome, output_path, num_workers=5):
    """
    Take a set of FASTQ files and align them with the Kallisto pseudoaligner

    :param srr_ids: list(str)
        NCBI SRR ID string
    :param fastq_file_names: list(list(str))
        A list of complete FASTQ file names that were unpacked from the SRR file (including path)
    :param reference_genome: str
        A path to the Kallisto reference genome that was preassembled
    :param output_path: str
        The path to put the output alignment files
    :param num_workers: int
        Number of separate simultaneous jobs to run
    :return sam_file_names: list(str)
        The SAM alignment files generated by STAR (including path)
    """

    sem = asyncio.Semaphore(num_workers)

    # Build output paths for STAR from SRR ids
    output_paths = list(map(lambda x: os.path.join(output_path, x, ''), srr_ids))

    # Build STAR tasks
    tasks = [_kallisto_align(sid, fqfn, reference_genome, sout, sem)
             for sid, fqfn, sout in zip(srr_ids, fastq_file_names, output_paths)]

    # Run and return STAR tasks
    return asyncio.get_event_loop().run_until_complete(asyncio.gather(*tasks))


async def _kallisto_align(srr_id, fastq_file_names, reference_genome, output_path, semaphore):
    """
    Align an individual set of FASTQs from an SRA to the reference genome

    :param srr_id: str
        NCBI SRR ID string
    :param fastq_file_names: list(str)
        A list of FASTQ files for the SRR ID
    :param reference_genome: str
        A path to the Kallisto reference genome
    :param output_path: str
        A path to the output
    :param semaphore: asyncio.Semaphore
        Semaphore for resource utilization
    :return output_file: str
        The path to the output files
    """

    async with semaphore:

        if fastq_file_names[0] is None:
            return None

        try:
            os.makedirs(output_path)
        except FileExistsError:
            pass

        output_file = os.path.join(file_path_abs(output_path), KALLISTO_OUT_FILE)

        if os.path.exists(output_file):
            print("{id} Kallisto alignment file exists ({path})".format(id=srr_id, path=output_path))
            return output_file

        # Build the Kallisto executable call
        kall_call = [KALLISTO_EXECUTABLE_PATH, "quant",
                     "-i", reference_genome,
                     "-o", output_path,
                     *fastq_file_names]

        process = await asyncio.create_subprocess_exec(*kall_call)
        code = await process.wait()

        if int(code) != 0:
            print("Kallisto failed for {id} ({files})".format(id=srr_id, files=" ".join(fastq_file_names)))
            print(" ".join(kall_call))
            return None

        return output_file
