import math
from dataclasses import dataclass

import numpy as np

slen1_tab = [0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4]
slen2_tab = [0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3]

scale_fact_band_index = [
    # MPEG-I
    # Table B.8.b: 44.1 kHz
    [0, 4, 8, 12, 16, 20, 24, 30, 36, 44, 52, 62, 74, 90, 110, 134, 162, 196, 238, 288, 342, 418, 576],
    # Table B.8.c: 48 kHz
    [0, 4, 8, 12, 16, 20, 24, 30, 36, 42, 50, 60, 72, 88, 106, 128, 156, 190, 230, 276, 330, 384, 576],
    # Table B.8.a: 32 kHz
    [0, 4, 8, 12, 16, 20, 24, 30, 36, 44, 54, 66, 82, 102, 126, 156, 194, 240, 296, 364, 448, 550, 576],
    # MPEG-II
    # Table B.2.b: 22.05 kHz
    [0, 6, 12, 18, 24, 30, 36, 44, 54, 66, 80, 96, 116, 140, 168, 200, 238, 284, 336, 396, 464, 522, 576],
    # Table B.2.c: 24 kHz
    [0, 6, 12, 18, 24, 30, 36, 44, 54, 66, 80, 96, 114, 136, 162, 194, 232, 278, 330, 394, 464, 540, 576],
    # Table B.2.a: 16 kHz
    [0, 6, 12, 18, 24, 30, 36, 44, 45, 66, 80, 96, 116, 140, 168, 200, 238, 248, 336, 396, 464, 522, 576],

    # MPEG-2.5
    # 11.025 kHz
    [0, 6, 12, 18, 24, 30, 36, 44, 54, 66, 80, 96, 116, 140, 168, 200, 238, 284, 336, 396, 464, 522, 576],
    # 12 kHz
    [0, 6, 12, 18, 24, 30, 36, 44, 54, 66, 80, 96, 116, 140, 168, 200, 238, 284, 336, 396, 464, 522, 576],
    # MPEG-2.5 8 kHz
    [0, 12, 24, 36, 48, 60, 72, 88, 108, 132, 160, 192, 232, 280, 336, 400, 476, 566, 568, 570, 572, 574, 576],
]

enwindow = np.array(
    [0, 0, 0, 0, 0, 0, 0, -2147, -2147, -2147, -2147, -2147, -2147, -4294, -4294, -4294, -4294, -6442, -6442,
     -6442, -8589, -8589, -10737, -10737, -12884, -15032, -17179, -17179, -19327, -21474, -23622, -25769, -30064,
     -32212, -36507, -38654, -42949, -45097, -49392, -53687, -60129, -64424, -68719, -75161, -81604, -88046,
     -92341, -98784, -107374, -113816, -120259, -128849, -135291, -141733, -150323, -156766, -165356, -173946,
     -180388, -186831, -195421, -199715, -206158, -212600, 219043, 223338, 227633, 229780, 231928, 234075,
     234075, 231928, 229780, 225485, 221190, 212600, 204010, 193273, 180388, 167503, 150323, 130996, 109521,
     85899, 57982, 30064, -2147, -36507, -73014, -113816, -156766, -201863, -249108, -300647, -354334, -410169,
     -470298, -530428, -594852, -661424, -727996, -796716, -867583, -940597, -1015759, -1088774, -1163936,
     -1239098, -1314259, -1389421, -1462436, -1533303, -1604170, -1672889, -1739461, -1801738, -1859720,
     -1915555, -1964947, -2010044, -2048699, -2080911, -2106681, -2123861, -2134598, -2136746, -2130303,
     -2113123, 2085206, 2048699, 1999307, 1939177, 1866163, 1780263, 1683627, 1571958, 1447403, 1309965, 1157493,
     994284, 813896, 618475, 412316, 188978, -45097, -294205, -558345, -833223, -1120986, -1421634, -1733019,
     -2055141, -2385854, -2727304, -3075196, -3431678, -3794603, -4159675, -4531190, -4902705, -5276367,
     -5650029, -6019396, -6386616, -6747393, -7101728, -7445325, -7780333, -8100308, -8405250, -8695161,
     -8965744, -9214852, -9440338, -9642201, -9814000, -9960029, -10073845, -10153302, -10198399, -10204842,
     -10172630, -10099615, -9983651, -9822590, -9614284, -9358733, -9051643, -8695161, -8287139, -7823282,
     -7305739, 6732361, 6101001, 5415953, 4670776, 3867618, 3006477, 2085206, 1108101, 70866, -1022202, -2173253,
     -3380139, -4642859, -5957119, -7325066, -8744553, -10213432, -11729555, -13290776, -14897094, -16542066,
     -18225693, -19945828, -21698174, -23478438, -25286619, -27118423, -28967406, -30835717, -32714765,
     -34602404, -36494337, -38388417, -40280350, -42163693, -44038447, -45896020, -47736413, -49551037,
     -51339891, -53096533, -54818815, -56502442, -58140972, -59732257, -61274150, -62760209, -64188286,
     -65556233, -66857608, -68090263, -69252052, -70340826, -71352291, -72284299, -73134703, -73901354,
     -74582107, -75176960, -75681618, -76096083, -76420353, -76652281, -76791867, 76839112, 76791867, 76652281,
     76420353, 76096083, 75681618, 75176960, 74582107, 73901354, 73134703, 72284299, 71352291, 70340826,
     69252052, 68090263, 66857608, 65556233, 64188286, 62760209, 61274150, 59732257, 58140972, 56502442,
     54818815, 53096533, 51339891, 49551037, 47736413, 45896020, 44038447, 42163693, 40280350, 38388417,
     36494337, 34602404, 32714765, 30835717, 28967406, 27118423, 25286619, 23478438, 21698174, 19945828,
     18225693, 16542066, 14897094, 13290776, 11729555, 10213432, 8744553, 7325066, 5957119, 4642859, 3380139,
     2173253, 1022202, -70866, -1108101, -2085206, -3006477, -3867618, -4670776, -5415953, -6101001, 6732361,
     7305739, 7823282, 8287139, 8695161, 9051643, 9358733, 9614284, 9822590, 9983651, 10099615, 10172630,
     10204842, 10198399, 10153302, 10073845, 9960029, 9814000, 9642201, 9440338, 9214852, 8965744, 8695161,
     8405250, 8100308, 7780333, 7445325, 7101728, 6747393, 6386616, 6019396, 5650029, 5276367, 4902705, 4531190,
     4159675, 3794603, 3431678, 3075196, 2727304, 2385854, 2055141, 1733019, 1421634, 1120986, 833223, 558345,
     294205, 45097, -188978, -412316, -618475, -813896, -994284, -1157493, -1309965, -1447403, -1571958,
     -1683627, -1780263, -1866163, -1939177, -1999307, -2048699, 2085206, 2113123, 2130303, 2136746, 2134598,
     2123861, 2106681, 2080911, 2048699, 2010044, 1964947, 1915555, 1859720, 1801738, 1739461, 1672889, 1604170,
     1533303, 1462436, 1389421, 1314259, 1239098, 1163936, 1088774, 1015759, 940597, 867583, 796716, 727996,
     661424, 594852, 530428, 470298, 410169, 354334, 300647, 249108, 201863, 156766, 113816, 73014, 36507, 2147,
     -30064, -57982, -85899, -109521, -130996, -150323, -167503, -180388, -193273, -204010, -212600, -221190,
     -225485, -229780, -231928, -234075, -234075, -231928, -229780, -227633, -223338, 219043, 212600, 206158,
     199715, 195421, 186831, 180388, 173946, 165356, 156766, 150323, 141733, 135291, 128849, 120259, 113816,
     107374, 98784, 92341, 88046, 81604, 75161, 68719, 64424, 60129, 53687, 49392, 45097, 42949, 38654, 36507,
     32212, 30064, 25769, 23622, 21474, 19327, 17179, 17179, 15032, 12884, 10737, 10737, 8589, 8589, 6442, 6442,
     6442, 4294, 4294, 4294, 4294, 2147, 2147, 2147, 2147, 2147, 2147, 0, 0, 0, 0, 0, 0])

t1HB = [1, 1, 1, 0]
t2HB = [1, 2, 1, 3, 1, 1, 3, 2, 0]
t3HB = [3, 2, 1, 1, 1, 1, 3, 2, 0]
t5HB = [1, 2, 6, 5, 3, 1, 4, 4, 7, 5, 7, 1, 6, 1, 1, 0]
t6HB = [7, 3, 5, 1, 6, 2, 3, 2, 5, 4, 4, 1, 3, 3, 2, 0]
t7HB = [1, 2, 10, 19, 16, 10, 3, 3, 7, 10, 5, 3, 11, 4, 13, 17, 8, 4, 12, 11, 18, 15, 11, 2, 7,
        6, 9, 14, 3, 1, 6, 4, 5, 3, 2, 0]
t8HB = [3, 4, 6, 18, 12, 5, 5, 1, 2, 16, 9, 3, 7, 3, 5, 14, 7, 3, 19, 17, 15, 13, 10, 4, 13, 5,
        8, 11, 5, 1, 12, 4, 4, 1, 1, 0]
t9HB = [7, 5, 9, 14, 15, 7, 6, 4, 5, 5, 6, 7, 7, 6, 8, 8, 8, 5, 15, 6, 9, 10, 5, 1, 11, 7, 9, 6,
        4, 1, 14, 4, 6, 2, 6, 0]
t10HB = [1, 2, 10, 23, 35, 30, 12, 17, 3, 3, 8, 12, 18, 21, 12, 7, 11, 9, 15, 21, 32, 40, 19, 6,
         14, 13, 22, 34, 46, 23, 18, 7, 20, 19, 33, 47, 27, 22, 9, 3, 31, 22, 41, 26, 21, 20, 5,
         3, 14, 13, 10, 11, 16, 6, 5, 1, 9, 8, 7, 8, 4, 4, 2, 0]
t11HB = [3, 4, 10, 24, 34, 33, 21, 15, 5, 3, 4, 10, 32, 17, 11, 10, 11, 7, 13, 18, 30, 31, 20,
         5, 25, 11, 19, 59, 27, 18, 12, 5, 35, 33, 31, 58, 30, 16, 7, 5, 28, 26, 32, 19, 17, 15,
         8, 14, 14, 12, 9, 13, 14, 9, 4, 1, 11, 4, 6, 6, 6, 3, 2, 0]
t12HB = [9, 6, 16, 33, 41, 39, 38, 26, 7, 5, 6, 9, 23, 16, 26, 11, 17, 7, 11, 14, 21, 30, 10, 7,
         17, 10, 15, 12, 18, 28, 14, 5, 32, 13, 22, 19, 18, 16, 9, 5, 40, 17, 31, 29, 17, 13, 4,
         2, 27, 12, 11, 15, 10, 7, 4, 1, 27, 12, 8, 12, 6, 3, 1, 0]
t13HB = [1, 5, 14, 21, 34, 51, 46, 71, 42, 52, 68, 52, 67, 44, 43, 19, 3, 4, 12, 19, 31, 26, 44,
         33, 31, 24, 32, 24, 31, 35, 22, 14, 15, 13, 23, 36, 59, 49, 77, 65, 29, 40, 30, 40, 27,
         33, 42, 16, 22,
         20, 37, 61, 56, 79, 73, 64, 43, 76, 56, 37, 26, 31, 25, 14, 35, 16, 60, 57, 97, 75,
         114, 91, 54, 73, 55, 41, 48, 53, 23, 24, 58, 27, 50, 96, 76, 70, 93, 84, 77, 58, 79,
         29, 74, 49, 41, 17, 47,
         45, 78, 74, 115, 94, 90, 79, 69, 83, 71, 50, 59, 38, 36, 15, 72, 34, 56, 95, 92, 85,
         91, 90, 86, 73, 77, 65, 51, 44, 43, 42, 43, 20, 30, 44, 55, 78, 72, 87, 78, 61, 46, 54,
         37, 30, 20, 16, 53,
         25, 41, 37, 44, 59, 54, 81, 66, 76, 57, 54, 37, 18, 39, 11, 35, 33, 31, 57, 42, 82, 72,
         80, 47, 58, 55, 21, 22, 26, 38, 22, 53, 25, 23, 38, 70, 60, 51, 36, 55, 26, 34, 23, 27,
         14, 9, 7, 34, 32,
         28, 39, 49, 75, 30, 52, 48, 40, 52, 28, 18, 17, 9, 5, 45, 21, 34, 64, 56, 50, 49, 45,
         31, 19, 12, 15, 10, 7, 6, 3, 48, 23, 20, 39, 36, 35, 53, 21, 16, 23, 13, 10, 6, 1, 4,
         2, 16, 15, 17, 27, 25,
         20, 29, 11, 17, 12, 16, 8, 1, 1, 0, 1]
t15HB = [7, 12, 18, 53, 47, 76, 124, 108, 89, 123, 108, 119, 107, 81, 122, 63, 13, 5, 16, 27,
         46, 36, 61, 51, 42, 70, 52, 83, 65, 41, 59, 36, 19, 17, 15, 24, 41, 34, 59, 48, 40, 64,
         50, 78, 62, 80, 56,
         33, 29, 28, 25, 43, 39, 63, 55, 93, 76, 59, 93, 72, 54, 75, 50, 29, 52, 22, 42, 40, 67,
         57, 95, 79, 72, 57, 89, 69, 49, 66, 46, 27, 77, 37, 35, 66, 58, 52, 91, 74, 62, 48, 79,
         63, 90, 62, 40, 38,
         125, 32, 60, 56, 50, 92, 78, 65, 55, 87, 71, 51, 73, 51, 70, 30, 109, 53, 49, 94, 88,
         75, 66, 122, 91, 73, 56, 42, 64, 44, 21, 25, 90, 43, 41, 77, 73, 63, 56, 92, 77, 66,
         47, 67, 48, 53, 36, 20,
         71, 34, 67, 60, 58, 49, 88, 76, 67, 106, 71, 54, 38, 39, 23, 15, 109, 53, 51, 47, 90,
         82, 58, 57, 48, 72, 57, 41, 23, 27, 62, 9, 86, 42, 40, 37, 70, 64, 52, 43, 70, 55, 42,
         25, 29, 18, 11, 11,
         118, 68, 30, 55, 50, 46, 74, 65, 49, 39, 24, 16, 22, 13, 14, 7, 91, 44, 39, 38, 34, 63,
         52, 45, 31, 52, 28, 19, 14, 8, 9, 3, 123, 60, 58, 53, 47, 43, 32, 22, 37, 24, 17, 12,
         15, 10, 2, 1, 71,
         37, 34, 30, 28, 20, 17, 26, 21, 16, 10, 6, 8, 6, 2, 0]
t16HB = [1, 5, 14, 44, 74, 63, 110, 93, 172, 149, 138, 242, 225, 195, 376, 17, 3, 4, 12, 20, 35,
         62, 53, 47, 83, 75, 68, 119, 201, 107, 207, 9, 15, 13, 23, 38, 67, 58, 103, 90, 161,
         72, 127, 117,
         110, 209, 206, 16, 45, 21, 39, 69, 64, 114, 99, 87, 158, 140, 252, 212, 199, 387, 365,
         26, 75, 36, 68, 65, 115, 101, 179, 164, 155, 264, 246, 226, 395, 382, 362, 9, 66, 30,
         59, 56, 102,
         185, 173, 265, 142, 253, 232, 400, 388, 378, 445, 16, 111, 54, 52, 100, 184, 178, 160,
         133, 257, 244, 228, 217, 385, 366, 715, 10, 98, 48, 91, 88, 165, 157, 148, 261, 248,
         407, 397, 372,
         380, 889, 884, 8, 85, 84, 81, 159, 156, 143, 260, 249, 427, 401, 392, 383, 727, 713,
         708, 7, 154, 76, 73, 141, 131, 256, 245, 426, 406, 394, 384, 735, 359, 710, 352, 11,
         139, 129, 67, 125,
         247, 233, 229, 219, 393, 743, 737, 720, 885, 882, 439, 4, 243, 120, 118, 115, 227, 223,
         396, 746, 742, 736, 721, 712, 706, 223, 436, 6, 202, 224, 222, 218, 216, 389, 386, 381,
         364, 888,
         443, 707, 440, 437, 1728, 4, 747, 211, 210, 208, 370, 379, 734, 723, 714, 1735, 883,
         877, 876, 3459, 865, 2, 377, 369, 102, 187, 726, 722, 358, 711, 709, 866, 1734, 871,
         3458, 870, 434,
         0, 12, 10, 7, 11, 10, 17, 11, 9, 13, 12, 10, 7, 5, 3, 1, 3]
t24HB = [15, 13, 46, 80, 146, 262, 248, 434, 426, 669, 653, 649, 621, 517, 1032, 88, 14, 12, 21,
         38, 71, 130, 122, 216, 209, 198, 327, 345, 319, 297, 279, 42, 47, 22, 41, 74, 68, 128,
         120, 221,
         207, 194, 182, 340, 315, 295, 541, 18, 81, 39, 75, 70, 134, 125, 116, 220, 204, 190,
         178, 325, 311, 293, 271, 16, 147, 72, 69, 135, 127, 118, 112, 210, 200, 188, 352, 323,
         306, 285,
         540, 14, 263, 66, 129, 126, 119, 114, 214, 202, 192, 180, 341, 317, 301, 281, 262, 12,
         249, 123, 121, 117, 113, 215, 206, 195, 185, 347, 330, 308, 291, 272, 520, 10, 435,
         115, 111,
         109, 211, 203, 196, 187, 353, 332, 313, 298, 283, 531, 381, 17, 427, 212, 208, 205,
         201, 193, 186, 177, 169, 320, 303, 286, 268, 514, 377, 16, 335, 199, 197, 191, 189,
         181, 174, 333,
         321, 305, 289, 275, 521, 379, 371, 11, 668, 184, 183, 179, 175, 344, 331, 314, 304,
         290, 277, 530, 383, 373, 366, 10, 652, 346, 171, 168, 164, 318, 309, 299, 287, 276,
         263, 513, 375,
         368, 362, 6, 648, 322, 316, 312, 307, 302, 292, 284, 269, 261, 512, 376, 370, 364, 359,
         4, 620, 300, 296, 294, 288, 282, 273, 266, 515, 380, 374, 369, 365, 361, 357, 2, 1033,
         280, 278,
         274, 267, 264, 259, 382, 378, 372, 367, 363, 360, 358, 356, 0, 43, 20, 19, 17, 15, 13,
         11, 9, 7, 6, 4, 7, 5, 3, 1, 3]
t32HB = [1, 5, 4, 5, 6, 5, 4, 4, 7, 3, 6, 0, 7, 2, 3, 1]
t33HB = [15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0]

t1l = [1, 3, 2, 3]
t2l = [1, 3, 6, 3, 3, 5, 5, 5, 6]
t3l = [2, 2, 6, 3, 2, 5, 5, 5, 6]
t5l = [1, 3, 6, 7, 3, 3, 6, 7, 6, 6, 7, 8, 7, 6, 7, 8]
t6l = [3, 3, 5, 7, 3, 2, 4, 5, 4, 4, 5, 6, 6, 5, 6, 7]
t7l = [1, 3, 6, 8, 8, 9, 3, 4, 6, 7, 7, 8, 6, 5, 7, 8, 8, 9, 7, 7, 8, 9, 9, 9, 7, 7, 8, 9,
       9, 10, 8, 8, 9, 10, 10, 10]
t8l = [2, 3, 6, 8, 8, 9, 3, 2, 4, 8, 8, 8, 6, 4, 6, 8, 8, 9, 8, 8, 8, 9, 9, 10, 8, 7, 8, 9,
       10, 10, 9, 8, 9, 9, 11, 11]
t9l = [3, 3, 5, 6, 8, 9, 3, 3, 4, 5, 6, 8, 4, 4, 5, 6, 7, 8, 6, 5, 6, 7, 7, 8, 7, 6, 7, 7,
       8, 9, 8, 7, 8, 8, 9, 9]
t10l = [1, 3, 6, 8, 9, 9, 9, 10, 3, 4, 6, 7, 8, 9, 8, 8, 6, 6, 7, 8, 9, 10, 9, 9, 7, 7, 8,
        9, 10, 10, 9, 10, 8, 8, 9, 10, 10, 10, 10, 10, 9, 9, 10, 10, 11, 11, 10, 11, 8, 8,
        9, 10, 10, 10, 11, 11, 9, 8, 9, 10, 10, 11, 11, 11]
t11l = [2, 3, 5, 7, 8, 9, 8, 9, 3, 3, 4, 6, 8, 8, 7, 8, 5, 5, 6, 7, 8, 9, 8, 8, 7, 6, 7, 9,
        8, 10, 8, 9, 8, 8, 8, 9, 9, 10, 9, 10, 8, 8, 9, 10, 10, 11, 10, 11, 8, 7, 7, 8, 9,
        10, 10, 10, 8, 7, 8, 9, 10, 10, 10, 10]
t12l = [4, 3, 5, 7, 8, 9, 9, 9, 3, 3, 4, 5, 7, 7, 8, 8, 5, 4, 5, 6, 7, 8, 7, 8, 6, 5, 6, 6,
        7, 8, 8, 8, 7, 6, 7, 7, 8, 8, 8, 9, 8, 7, 8, 8, 8, 9, 8, 9, 8, 7, 7, 8, 8, 9, 9,
        10, 9, 8, 8, 9, 9, 9, 9, 10]
t13l = [1, 4, 6, 7, 8, 9, 9, 10, 9, 10, 11, 11, 12, 12, 13, 13, 3, 4, 6, 7, 8, 8, 9, 9, 9,
        9, 10, 10, 11, 12, 12, 12, 6, 6, 7, 8, 9, 9, 10, 10, 9, 10, 10, 11, 11, 12, 13, 13,
        7, 7, 8, 9, 9, 10, 10, 10, 10, 11, 11, 11, 11, 12, 13, 13,
        8, 7, 9, 9, 10, 10, 11, 11, 10, 11, 11, 12, 12, 13, 13, 14, 9, 8, 9, 10, 10, 10,
        11, 11, 11, 11, 12, 11, 13, 13, 14, 14, 9, 9, 10, 10, 11, 11, 11, 11, 11, 12, 12,
        12, 13, 13, 14, 14, 10, 9, 10, 11, 11, 11, 12, 12, 12, 12, 13, 13, 13, 14, 16, 16,
        9, 8, 9, 10,
        10, 11, 11, 12, 12, 12, 12, 13, 13, 14, 15, 15, 10, 9, 10, 10, 11, 11, 11, 13, 12,
        13, 13, 14, 14, 14, 16, 15, 10, 10, 10, 11, 11, 12, 12, 13, 12, 13, 14, 13, 14, 15,
        16, 17, 11, 10, 10, 11, 12, 12, 12, 12, 13, 13, 13, 14, 15, 15, 15, 16, 11, 11, 11,
        12, 12,
        13, 12, 13, 14, 14, 15, 15, 15, 16, 16, 16, 12, 11, 12, 13, 13, 13, 14, 14, 14, 14,
        14, 15, 16, 15, 16, 16, 13, 12, 12, 13, 13, 13, 15, 14, 14, 17, 15, 15, 15, 17, 16,
        16, 12, 12, 13, 14, 14, 14, 15, 14, 15, 15, 16, 16, 19, 18, 19, 16]
t15l = [3, 4, 5, 7, 7, 8, 9, 9, 9, 10, 10, 11, 11, 11, 12, 13, 4, 3, 5, 6, 7, 7, 8, 8, 8,
        9, 9, 10, 10, 10, 11, 11, 5, 5, 5, 6, 7, 7, 8, 8, 8, 9, 9, 10, 10, 11, 11, 11, 6,
        6, 6, 7, 7, 8, 8, 9, 9, 9, 10, 10, 10, 11, 11, 11, 7, 6, 7,
        7, 8, 8, 9, 9, 9, 9, 10, 10, 10, 11, 11, 11, 8, 7, 7, 8, 8, 8, 9, 9, 9, 9, 10, 10,
        11, 11, 11, 12, 9, 7, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10, 11, 11, 12, 12, 9, 8, 8, 9,
        9, 9, 9, 10, 10, 10, 10, 10, 11, 11, 11, 12, 9, 8, 8, 9, 9, 9, 9, 10, 10, 10, 10,
        11, 11,
        12, 12, 12, 9, 8, 9, 9, 9, 9, 10, 10, 10, 11, 11, 11, 11, 12, 12, 12, 10, 9, 9, 9,
        10, 10, 10, 10, 10, 11, 11, 11, 11, 12, 13, 12, 10, 9, 9, 9, 10, 10, 10, 10, 11,
        11, 11, 11, 12, 12, 12, 13, 11, 10, 9, 10, 10, 10, 11, 11, 11, 11, 11, 11, 12, 12,
        13, 13,
        11, 10, 10, 10, 10, 11, 11, 11, 11, 12, 12, 12, 12, 12, 13, 13, 12, 11, 11, 11, 11,
        11, 11, 11, 12, 12, 12, 12, 13, 13, 12, 13, 12, 11, 11, 11, 11, 11, 11, 12, 12, 12,
        12, 12, 13, 13, 13, 13]
t16l = [1, 4, 6, 8, 9, 9, 10, 10, 11, 11, 11, 12, 12, 12, 13, 9, 3, 4, 6, 7, 8, 9, 9, 9,
        10, 10, 10, 11, 12, 11, 12, 8, 6, 6, 7, 8, 9, 9, 10, 10, 11, 10, 11, 11, 11, 12,
        12, 9, 8, 7, 8, 9, 9, 10, 10, 10, 11, 11, 12, 12, 12, 13, 13,
        10, 9, 8, 9, 9, 10, 10, 11, 11, 11, 12, 12, 12, 13, 13, 13, 9, 9, 8, 9, 9, 10, 11,
        11, 12, 11, 12, 12, 13, 13, 13, 14, 10, 10, 9, 9, 10, 11, 11, 11, 11, 12, 12, 12,
        12, 13, 13, 14, 10, 10, 9, 10, 10, 11, 11, 11, 12, 12, 13, 13, 13, 13, 15, 15, 10,
        10, 10,
        10, 11, 11, 11, 12, 12, 13, 13, 13, 13, 14, 14, 14, 10, 11, 10, 10, 11, 11, 12, 12,
        13, 13, 13, 13, 14, 13, 14, 13, 11, 11, 11, 10, 11, 12, 12, 12, 12, 13, 14, 14, 14,
        15, 15, 14, 10, 12, 11, 11, 11, 12, 12, 13, 14, 14, 14, 14, 14, 14, 13, 14, 11, 12,
        12,
        12, 12, 12, 13, 13, 13, 13, 15, 14, 14, 14, 14, 16, 11, 14, 12, 12, 12, 13, 13, 14,
        14, 14, 16, 15, 15, 15, 17, 15, 11, 13, 13, 11, 12, 14, 14, 13, 14, 14, 15, 16, 15,
        17, 15, 14, 11, 9, 8, 8, 9, 9, 10, 10, 10, 11, 11, 11, 11, 11, 11, 11, 8]
t24l = [4, 4, 6, 7, 8, 9, 9, 10, 10, 11, 11, 11, 11, 11, 12, 9, 4, 4, 5, 6, 7, 8, 8, 9, 9,
        9, 10, 10, 10, 10, 10, 8, 6, 5, 6, 7, 7, 8, 8, 9, 9, 9, 9, 10, 10, 10, 11, 7, 7, 6,
        7, 7, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10, 10, 7, 8, 7, 7, 8,
        8, 8, 8, 9, 9, 9, 10, 10, 10, 10, 11, 7, 9, 7, 8, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10,
        10, 10, 7, 9, 8, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10, 10, 10, 11, 7, 10, 8, 8, 8, 9, 9,
        9, 9, 10, 10, 10, 10, 10, 11, 11, 8, 10, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10,
        11, 11,
        8, 10, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 11, 11, 11, 8, 11, 9, 9, 9, 9, 10, 10,
        10, 10, 10, 10, 11, 11, 11, 11, 8, 11, 10, 9, 9, 9, 10, 10, 10, 10, 10, 10, 11, 11,
        11, 11, 8, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 8, 11, 10,
        10,
        10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 11, 8, 12, 10, 10, 10, 10, 10, 10, 11,
        11, 11, 11, 11, 11, 11, 11, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 4]
t32l = [1, 4, 4, 5, 4, 6, 5, 6, 4, 5, 5, 6, 5, 6, 6, 6]
t33l = [4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4]


@dataclass
class HuffCodeTab:
    xlen: int  # max. x-index+
    ylen: int  # max. y-index+
    linbits: int  # number of linbits
    linmax: int  # max number to be stored in linbits
    table: list  # pointer to array[xlen][ylen]
    hlen: list  # pointer to array[xlen][ylen]

    def __init__(self, xlen, ylen, linbits, linmax, table, hlen):
        self.xlen = xlen
        self.ylen = ylen
        self.linbits = linbits
        self.linmax = linmax
        self.table = table
        self.hlen = hlen


huffman_table = [HuffCodeTab(0, 0, 0, 0, None, None),
                 HuffCodeTab(2, 2, 0, 0, t1HB, t1l),
                 HuffCodeTab(3, 3, 0, 0, t2HB, t2l),
                 HuffCodeTab(3, 3, 0, 0, t3HB, t3l),
                 HuffCodeTab(0, 0, 0, 0, None, None),  # Apparently not used
                 HuffCodeTab(4, 4, 0, 0, t5HB, t5l),
                 HuffCodeTab(4, 4, 0, 0, t6HB, t6l),
                 HuffCodeTab(6, 6, 0, 0, t7HB, t7l),
                 HuffCodeTab(6, 6, 0, 0, t8HB, t8l),
                 HuffCodeTab(6, 6, 0, 0, t9HB, t9l),
                 HuffCodeTab(8, 8, 0, 0, t10HB, t10l),
                 HuffCodeTab(8, 8, 0, 0, t11HB, t11l),
                 HuffCodeTab(8, 8, 0, 0, t12HB, t12l),
                 HuffCodeTab(16, 16, 0, 0, t13HB, t13l),
                 HuffCodeTab(0, 0, 0, 0, None, None),  # Apparently not used
                 HuffCodeTab(16, 16, 0, 0, t15HB, t15l),
                 HuffCodeTab(16, 16, 1, 1, t16HB, t16l),
                 HuffCodeTab(16, 16, 2, 3, t16HB, t16l),
                 HuffCodeTab(16, 16, 3, 7, t16HB, t16l),
                 HuffCodeTab(16, 16, 4, 15, t16HB, t16l),
                 HuffCodeTab(16, 16, 6, 63, t16HB, t16l),
                 HuffCodeTab(16, 16, 8, 255, t16HB, t16l),
                 HuffCodeTab(16, 16, 10, 1023, t16HB, t16l),
                 HuffCodeTab(16, 16, 13, 8191, t16HB, t16l),
                 HuffCodeTab(16, 16, 4, 15, t24HB, t24l),
                 HuffCodeTab(16, 16, 5, 31, t24HB, t24l),
                 HuffCodeTab(16, 16, 6, 63, t24HB, t24l),
                 HuffCodeTab(16, 16, 7, 127, t24HB, t24l),
                 HuffCodeTab(16, 16, 8, 255, t24HB, t24l),
                 HuffCodeTab(16, 16, 9, 511, t24HB, t24l),
                 HuffCodeTab(16, 16, 11, 2047, t24HB, t24l),
                 HuffCodeTab(16, 16, 13, 8191, t24HB, t24l),
                 HuffCodeTab(1, 16, 0, 0, t32HB, t32l),
                 HuffCodeTab(1, 16, 0, 0, t33HB, t33l)]


# This is table B.9: coefficients for aliasing reduction
def MDCT_CA(coef):
    return np.int32(coef / math.sqrt(np.double(1.0) + (coef * coef)) * 0x7fffffff)


def MDCT_CS(coef):
    return np.int32(np.double(1.0) / math.sqrt(np.double(1.0) + (coef * coef)) * 0x7fffffff)


MDCT_CA0 = MDCT_CA(-0.6)
MDCT_CA1 = MDCT_CA(-0.535)
MDCT_CA2 = MDCT_CA(-0.33)
MDCT_CA3 = MDCT_CA(-0.185)
MDCT_CA4 = MDCT_CA(-0.095)
MDCT_CA5 = MDCT_CA(-0.041)
MDCT_CA6 = MDCT_CA(-0.0142)
MDCT_CA7 = MDCT_CA(-0.0037)

MDCT_CS0 = MDCT_CS(-0.6)
MDCT_CS1 = MDCT_CS(-0.535)
MDCT_CS2 = MDCT_CS(-0.33)
MDCT_CS3 = MDCT_CS(-0.185)
MDCT_CS4 = MDCT_CS(-0.095)
MDCT_CS5 = MDCT_CS(-0.041)
MDCT_CS6 = MDCT_CS(-0.0142)
MDCT_CS7 = MDCT_CS(-0.0037)

# 1st element = region0_count, 2nd element = region1_count
subdv_table = [
    (0, 0),  # 0 bands
    (0, 0),  # 1 bands
    (0, 0),  # 2 bands
    (0, 0),  # 3 bands
    (0, 0),  # 4 bands
    (0, 1),  # 5 bands
    (1, 1),  # 6 bands
    (1, 1),  # 7 bands
    (1, 2),  # 8 bands
    (2, 2),  # 9 bands
    (2, 3),  # 10 bands
    (2, 3),  # 11 bands
    (3, 4),  # 12 bands
    (3, 4),  # 13 bands
    (3, 4),  # 14 bands
    (4, 5),  # 15 bands
    (4, 5),  # 16 bands
    (4, 6),  # 17 bands
    (5, 6),  # 18 bands
    (5, 6),  # 19 bands
    (5, 7),  # 20 bands
    (6, 7),  # 21 bands
    (6, 7),  # 22 bands
]
