"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiDefinition = void 0;
const apigateway = require("aws-cdk-lib/aws-apigateway");
const add_1 = require("../errors/add");
const schema_1 = require("../schema");
const utils_1 = require("./utils");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const omitDeep = require('omit-deep-lodash');
/** Represents an OpenAPI v3 definition asset. */
class ApiDefinition extends apigateway.ApiDefinition {
    /** Represents an OpenAPI v3 definition asset. */
    constructor(scope, props) {
        super();
        this.scope = scope;
        this.upload = props.upload === true;
        this.schema = this.resolveSource(props.source);
        // Handle injects/rejects
        this.schema.inject(props.injections);
        this.schema.reject(props.rejections);
        this.schema.rejectDeep(props.rejectionsDeep);
        // Configurate integrations
        this.configureValidators(props.validators);
        this.configureAuthorizers(props.authorizers);
        this.configurePaths(props.paths, props.defaultCors, props.defaultIntegration);
        // Finally expose the processed OpenApi v3 document
        this.document = this.schema.toDocument();
        this.validateDocument(this.document);
    }
    /** Configures API Gateway validators (if any). */
    configureValidators(validators = {}) {
        const keys = Object.keys(validators);
        // Do not assign x-amazon-apigateway-request-validators object if none provided
        if (keys.length < 1) {
            return;
        }
        const defaults = keys.filter(k => validators[k].default === true);
        // Ensure only single validator configured as default
        if (defaults.length > 1) {
            add_1.addError(this.scope, 'You may only configure one default validator');
            return;
        }
        // Configure the default validator if provided
        if (defaults.length === 1) {
            const defaultValidator = defaults[0];
            this.schema.set('x-amazon-apigateway-request-validator', defaultValidator);
        }
        // Omit the non-standard "default" field
        const cleaned = omitDeep(validators, 'default');
        this.schema.set('x-amazon-apigateway-request-validators', cleaned);
    }
    /**
     * Configure Authorizers within OpenApi `components.securitySchemes`.
     */
    configureAuthorizers(authorizers = []) {
        authorizers.map(a => {
            const authorizerComponentSecuritySchemePath = `components.securitySchemes.${a.id}`;
            if (!this.schema.has(authorizerComponentSecuritySchemePath)) {
                add_1.addError(this.scope, `Security Scheme ${a.id} not found in OpenAPI Definition`);
                return;
            }
            /**
            * if current authorizer is defined for whole api, add security to all paths
            */
            const schemaSecurity = this.schema.get('security');
            // check if security includes authorizer
            const schemaSecurityIncludesCurrentAuthorizer = Array.isArray(schemaSecurity) && schemaSecurity.some(s => Object.keys(s).includes(a.id));
            if (schemaSecurityIncludesCurrentAuthorizer) {
                // loop schema paths
                const schemaPaths = utils_1.getSchemaPaths(this.schema);
                if (schemaPaths !== undefined) {
                    Object.keys(schemaPaths).forEach(path => {
                        // loop methods
                        const schemaPathMethods = utils_1.getMethodsFromSchemaPath(schemaPaths[path]);
                        Object.keys(schemaPathMethods).forEach(method => {
                            // check if method has security
                            const methodSecurity = this.schema.get(`paths.${path}.${method}.security`);
                            if (methodSecurity && Array.isArray(methodSecurity)) {
                                // check if security includes authorizer
                                const methodSecurityIncludesCurrentAuthorizer = methodSecurity.some(s => Object.keys(s).includes(a.id));
                                if (!methodSecurityIncludesCurrentAuthorizer) {
                                    this.schema.set(`paths.${path}.${method}.security`, [...methodSecurity, { [a.id]: [] }]);
                                }
                            }
                            else {
                                this.schema.set(`paths.${path}.${method}.security`, [{ [a.id]: [] }]);
                            }
                        });
                    });
                }
            }
            const authorizer = this.schema.get(authorizerComponentSecuritySchemePath);
            authorizer['x-amazon-apigateway-authtype'] = a.xAmazonApigatewayAuthtype;
            authorizer['x-amazon-apigateway-authorizer'] = a.xAmazonApigatewayAuthorizer;
            this.schema.set(authorizerComponentSecuritySchemePath, authorizer);
        });
    }
    /**
     * Configure all `x-amazon-apigateway-integration` values within OpenApi `paths`.
     */
    configurePaths(paths = {}, defaultCors, defaultIntegration) {
        const schemaPaths = utils_1.getSchemaPaths(this.schema);
        // Check that schema has paths object
        if (typeof schemaPaths === 'undefined') {
            add_1.addError(this.scope, 'OpenAPI Definition does not have paths object');
            return;
        }
        // Loop through paths to ensure all paths are defined in OpenAPI schema
        Object.keys(paths).forEach(path => {
            if (!schemaPaths[path]) {
                const message = `Path ${path} not found in OpenAPI Definition. Check paths-props in definition.`;
                add_1.addError(this.scope, message);
            }
        });
        // Loop through all schema paths
        Object.keys(schemaPaths).map((path) => {
            if (!defaultIntegration && !paths[path]) {
                const message = `Missing integration for path: ${path}. Check paths-props in definition, or add a default integration.`;
                add_1.addError(this.scope, message);
                return;
            }
            if (typeof defaultCors !== 'undefined') {
                this.schema.set(`paths.${path}.options`, {
                    'x-amazon-apigateway-integration': defaultCors.xAmazonApigatewayIntegration,
                });
            }
            const methods = paths[path];
            this.configurePathMethods(path, schemaPaths[path], methods, defaultIntegration);
        });
    }
    /**
     * Configure `x-amazon-apigateway-integration` for given method.
     */
    configurePathMethods(schemaPathName, schemaPath, methods = {}, defaultIntegration) {
        // Loop through given methods to ensure they are defined
        // and dont have an existing integration
        Object.keys(methods).map((method) => {
            const methodName = method.toLowerCase();
            this.ensureMethodExists(schemaPathName, methodName);
            this.ensureNoIntegrationAlready(schemaPathName, methodName);
        });
        const schemaPathMethods = utils_1.getMethodsFromSchemaPath(schemaPath);
        // Look through all schema path methods
        Object.keys(schemaPathMethods).map((schemaPathMethod) => {
            const method = methods[schemaPathMethod];
            if (!defaultIntegration && !method) {
                const message = `OpenAPI schema has an unhandled path method: ${schemaPathName}/${schemaPathMethod}`;
                add_1.addError(this.scope, message);
                return;
            }
            let integration;
            // Generate mock integration if requested
            if (defaultIntegration && !method) {
                integration = defaultIntegration;
            }
            else {
                integration = method;
            }
            const methodPath = `paths['${schemaPathName}']['${schemaPathMethod}']`;
            const validator = integration.validator;
            if (typeof validator === 'string') {
                this.schema.set(`${methodPath}['x-amazon-apigateway-request-validator']`, validator);
            }
            this.schema.set(`${methodPath}['x-amazon-apigateway-integration']`, integration.xAmazonApigatewayIntegration);
        });
    }
    /**
     * End-User can provide the OpenAPI definition either as a path to a file or
     * as an Asset. This method handles that and always returns Asset Source.
     */
    resolveSource(source) {
        if (typeof source === 'string') {
            return schema_1.Schema.fromAsset(source);
        }
        return source;
    }
    /**
     * Implement the functionality of exposing the API definition, either as `s3Location` or as `inlineDefinition`.
     *
     * Called by `apigateway.SpecRestApi`.
     *
     * @see https://github.com/aws/aws-cdk/blob/87dd2a6eb0b8208e49ff5f0cc8486ad58410d3ef/packages/%40aws-cdk/aws-apigateway/lib/restapi.ts#L636
     * @see https://github.com/aws/aws-cdk/blob/87dd2a6eb0b8208e49ff5f0cc8486ad58410d3ef/packages/%40aws-cdk/aws-apigateway/lib/api-definition.ts#L81-L88
     */
    bind(_) {
        if (this.upload === true) {
            const asset = this.schema.toAsset(this.scope, 'SchemaAsset');
            return {
                s3Location: {
                    bucket: asset.bucket.bucketArn,
                    key: asset.s3ObjectKey,
                },
            };
        }
        return {
            inlineDefinition: this.schema.toDocument(),
        };
    }
    /**
     * Ensures OpenAPI definition contains a given method for the path.
     */
    ensureMethodExists(path, method) {
        const value = this.schema.get(`paths[${path}][${method}]`);
        if (typeof value === 'undefined') {
            const message = `OpenAPI schema is missing method ${method} for path: ${path}`;
            add_1.addError(this.scope, message);
        }
    }
    /**
     * Ensures OpenAPI definition does not already have
     * `x-amazon-apigateway-integration` configuration for given method for the path.
     */
    ensureNoIntegrationAlready(path, method) {
        const value = this.schema.get(`paths[${path}][${method}]['x-amazon-apigateway-integration']`);
        if (typeof value !== 'undefined') {
            const message = `OpenAPI schema already has x-amazon-apigateway-integration configuration for method ${method} in path: ${path}`;
            add_1.addError(this.scope, message);
        }
    }
    /** Validate final OpenApi v3 document. */
    validateDocument(document) {
        if (typeof (document) !== 'object') {
            add_1.addError(this.scope, 'definition should be of type object');
        }
        if (Object.keys(document).length === 0) {
            add_1.addError(this.scope, 'JSON definition cannot be empty');
        }
    }
}
exports.ApiDefinition = ApiDefinition;
//# sourceMappingURL=data:application/json;base64,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