import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { Construct } from 'constructs';
import { IDocument } from './idocument';
import { SchemaProps } from './props';
/**
 * Represents an OpenApi v3 Schema which can be deserialized from YAML-file, modified
 * and then serialized back to YAML.
 */
export declare class Schema {
    /**
     * A string representing supported SemVer range.
     * @see https://github.com/npm/node-semver
     */
    static openApiSupportedVersions: string;
    /** Parse OpenApi v3 schema from inline YAML content. */
    static fromInline(content: string): Schema;
    /** Parse OpenApi v3 schema by loading a YAML file from given path. */
    static fromAsset(path: string): Schema;
    /**
     * Holds the actual parsed OpenApi v3 Schema Definition.
     * @see https://github.com/drwpow/openapi-typescript/blob/main/src/types.ts#L11-L22
     *
     * @todo info object?
     * @todo validate openapi string?
    */
    private document;
    /**
     * OpenApi version used by schema document.
     *
     * @example
     * '3.0.3'
     */
    openApiVersion: string;
    /** Construct a new Schema instance from OpenApi v3 JSON.  */
    constructor(props: SchemaProps);
    /** Serialize to YAML string. */
    toYaml(): string;
    /** Serialize to JSON string. */
    toJson(): string;
    /** Return the actual OpenApi v3 document. */
    toDocument(): IDocument;
    /** Return the OpenApi v3 document as an S3 Asset. */
    toAsset(scope: Construct, id: string): Asset;
    /** Check if definition has a value in given object path. */
    has(path: string): boolean;
    /** Get a value from given object path. */
    get<T>(path: string): T;
    /** Set a value to given object path. */
    set(path: string, value: any): void;
    /** Inject multiple values to given paths. */
    inject(records?: Record<string, any>): void;
    /** Reject – i.e. remove values – from given object paths. */
    reject(paths?: string[]): void;
    /** Reject deep within object – i.e. remove all nested object paths. */
    rejectDeep(paths?: string[]): void;
}
