import { IAspect, Stack } from 'aws-cdk-lib';
import { CfnTaskDefinition } from 'aws-cdk-lib/aws-ecs';
import { Policy } from 'aws-cdk-lib/aws-iam';
import { IConstruct } from 'constructs';
/**
 * Configuration properties for ECRRepositoryAspect
 */
export interface ECRRepositoryAspectProps {
    /**
     * Override the ECR repository account id of the Docker Image assets
     *
     * @default - process.env.BSS_IMAGE_ASSET_ACCOUNT_ID
     */
    readonly imageAssetAccountId?: string;
}
/**
 * Abtract aspect for ECR repository.
 *
 * You must provide the account id in props or set BSS_IMAGE_ASSET_ACCOUNT_ID in env
 */
export declare abstract class ECRRepositoryAspect implements IAspect {
    /**
     * @internal
     */
    static readonly _repoPolicies: Map<string, Policy>;
    readonly account: string;
    constructor(props?: ECRRepositoryAspectProps);
    abstract visit(construct: IConstruct): void;
    protected getRepoName(imageUri: string): string | undefined;
    protected crossAccountECRPolicy(stack: Stack, repoName: string): Policy;
}
/**
 * Process the image assets in ECS task definition
 */
export declare class ECSTaskDefinition extends ECRRepositoryAspect {
    constructor(props?: ECRRepositoryAspectProps);
    protected hasBeReplaced(prop: CfnTaskDefinition.ContainerDefinitionProperty): string | undefined;
    visit(construct: IConstruct): void;
}
/**
 * Process the image assets in SageMaker training job in Step Functions
 */
export declare class StepFunctionsSageMakerTrainingJob extends ECRRepositoryAspect {
    constructor(props?: ECRRepositoryAspectProps);
    visit(construct: IConstruct): void;
}
/**
 * Process the image assets in AWS Batch job
 */
export declare class BatchJobDefinition extends ECRRepositoryAspect {
    /**
     * @internal
     */
    readonly _repoNames: string[];
    private _executionRole?;
    private _executionRoleArn?;
    private _allRolesMap;
    constructor(props?: ECRRepositoryAspectProps);
    visit(construct: IConstruct): void;
}
/**
 * Default ECR asset aspect, support using ECR assets in below services,
 *
 * - ECS task definition
 * - SageMaker training job in Step Functions
 * - AWS Batch job
 * - AWS Lambda container image
 */
export declare class CompositeECRRepositoryAspect extends ECRRepositoryAspect {
    /**
     * @internal
     */
    readonly _aspects: ECRRepositoryAspect[];
    constructor(props?: ECRRepositoryAspectProps);
    visit(construct: IConstruct): void;
}
