"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeECRRepositoryAspect = exports.BatchJobDefinition = exports.StepFunctionsSageMakerTrainingJob = exports.ECSTaskDefinition = exports.ECRRepositoryAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const batch = require("@aws-cdk/aws-batch-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const FN_SUB = 'Fn::Sub';
/**
 * Abtract aspect for ECR repository.
 *
 * You must provide the account id in props or set BSS_IMAGE_ASSET_ACCOUNT_ID in env
 */
class ECRRepositoryAspect {
    constructor(props = {}) {
        this.account = props.imageAssetAccountId ?? process.env.BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    getRepoName(imageUri) {
        const matches = /\d{12}\.dkr\.ecr\..*\/(.*):.*/g.exec(imageUri);
        if (matches) {
            return matches[1];
        }
        return undefined;
    }
    crossAccountECRPolicy(stack, repoName) {
        const policy = ECRRepositoryAspect._repoPolicies.get(repoName);
        if (policy) {
            return policy;
        }
        const newPolicy = new aws_iam_1.Policy(stack, `CrossAccountECR-${repoName}`, {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ecr:BatchCheckLayerAvailability',
                        'ecr:GetDownloadUrlForLayer',
                        'ecr:BatchGetImage',
                    ],
                    resources: [
                        aws_cdk_lib_1.Arn.format({
                            account: this.account,
                            service: 'ecr',
                            resource: 'repository',
                            resourceName: repoName,
                        }, stack),
                    ],
                }),
            ],
        });
        ECRRepositoryAspect._repoPolicies.set(repoName, newPolicy);
        return newPolicy;
    }
}
exports.ECRRepositoryAspect = ECRRepositoryAspect;
_a = JSII_RTTI_SYMBOL_1;
ECRRepositoryAspect[_a] = { fqn: "cdk-bootstrapless-synthesizer.ECRRepositoryAspect", version: "2.2.10" };
/**
 * @internal
 */
ECRRepositoryAspect._repoPolicies = new Map();
/**
 * Process the image assets in ECS task definition
 */
class ECSTaskDefinition extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    hasBeReplaced(prop) {
        if (typeof prop.image === 'object' && FN_SUB in prop.image &&
            prop.image[FN_SUB].indexOf(this.account) > -1) {
            return prop.image[FN_SUB];
        }
        else if (prop.image && prop.image && prop.image.indexOf(this.account) > -1) {
            return prop.image;
        }
        return undefined;
    }
    visit(construct) {
        if (construct instanceof aws_ecs_1.TaskDefinition) {
            const containers = construct.stack.resolve(construct.node.defaultChild.containerDefinitions);
            let imageUri = undefined;
            if (containers instanceof Array) {
                for (const container of containers) {
                    if (container) {
                        imageUri = this.hasBeReplaced(container);
                        if (imageUri) {
                            break;
                        }
                    }
                }
            }
            else if (containers) {
                imageUri = this.hasBeReplaced(containers);
            }
            if (imageUri) {
                const repoName = this.getRepoName(imageUri);
                if (repoName) {
                    construct.executionRole.attachInlinePolicy(this.crossAccountECRPolicy(construct.stack, repoName));
                }
            }
        }
    }
}
exports.ECSTaskDefinition = ECSTaskDefinition;
_b = JSII_RTTI_SYMBOL_1;
ECSTaskDefinition[_b] = { fqn: "cdk-bootstrapless-synthesizer.ECSTaskDefinition", version: "2.2.10" };
/**
 * Process the image assets in SageMaker training job in Step Functions
 */
class StepFunctionsSageMakerTrainingJob extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    visit(construct) {
        if (construct instanceof aws_stepfunctions_tasks_1.SageMakerCreateTrainingJob) {
            const stack = aws_cdk_lib_1.Stack.of(construct);
            const state = construct.toStateJson();
            const image = stack.resolve(state.Parameters.AlgorithmSpecification.TrainingImage);
            if (FN_SUB in image) {
                const repoName = this.getRepoName(image[FN_SUB]);
                if (repoName) {
                    construct.role.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                }
            }
        }
    }
}
exports.StepFunctionsSageMakerTrainingJob = StepFunctionsSageMakerTrainingJob;
_c = JSII_RTTI_SYMBOL_1;
StepFunctionsSageMakerTrainingJob[_c] = { fqn: "cdk-bootstrapless-synthesizer.StepFunctionsSageMakerTrainingJob", version: "2.2.10" };
/**
 * Process the image assets in AWS Batch job
 */
class BatchJobDefinition extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
        this._repoNames = [];
        this._allRolesMap = new Map();
    }
    visit(construct) {
        if (construct instanceof batch.JobDefinition) {
            const stack = construct.stack;
            this._executionRoleArn = construct.node.defaultChild
                .containerProperties.executionRoleArn;
            if (this._executionRoleArn && this._allRolesMap.get(this._executionRoleArn)) {
                this._executionRole = this._allRolesMap.get(this._executionRoleArn);
            }
            const image = construct.node.defaultChild
                .containerProperties.image;
            const image_resolved = stack.resolve(image);
            if (FN_SUB in image_resolved) {
                const repoName = this.getRepoName(image_resolved[FN_SUB]);
                if (repoName) {
                    if (this._executionRole) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                    else {
                        if (this._repoNames.indexOf(repoName) < 0) {
                            this._repoNames.push(repoName);
                        }
                    }
                }
            }
        }
        if (construct instanceof aws_iam_1.Role) {
            this._allRolesMap.set(construct.roleArn, construct);
            if (construct.roleArn == this._executionRoleArn) {
                const stack = construct.stack;
                this._executionRole = construct;
                while (this._repoNames.length > 0) {
                    const repoName = this._repoNames.pop();
                    if (repoName) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                }
            }
        }
    }
}
exports.BatchJobDefinition = BatchJobDefinition;
_d = JSII_RTTI_SYMBOL_1;
BatchJobDefinition[_d] = { fqn: "cdk-bootstrapless-synthesizer.BatchJobDefinition", version: "2.2.10" };
/**
 * Default ECR asset aspect, support using ECR assets in below services,
 *
 * - ECS task definition
 * - SageMaker training job in Step Functions
 * - AWS Batch job
 * - AWS Lambda container image
 */
class CompositeECRRepositoryAspect extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
        this._aspects = [
            new ECSTaskDefinition(props),
            new StepFunctionsSageMakerTrainingJob(props),
            new BatchJobDefinition(props),
        ];
    }
    visit(construct) {
        for (const _aspect of this._aspects) {
            _aspect.visit(construct);
        }
    }
}
exports.CompositeECRRepositoryAspect = CompositeECRRepositoryAspect;
_e = JSII_RTTI_SYMBOL_1;
CompositeECRRepositoryAspect[_e] = { fqn: "cdk-bootstrapless-synthesizer.CompositeECRRepositoryAspect", version: "2.2.10" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXNwZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2FzcGVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGtEQUFrRDtBQUNsRCw2Q0FBa0Q7QUFFbEQsaURBQXdFO0FBQ3hFLGlEQUFvRTtBQUNwRSxpRkFBaUY7QUFHakYsTUFBTSxNQUFNLEdBQUcsU0FBUyxDQUFDO0FBYXpCOzs7O0dBSUc7QUFDSCxNQUFzQixtQkFBbUI7SUFRdkMsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsMEJBQTJCLENBQUM7SUFDdEYsQ0FBQztJQUlTLFdBQVcsQ0FBQyxRQUFnQjtRQUNwQyxNQUFNLE9BQU8sR0FBRyxnQ0FBZ0MsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDaEUsSUFBSSxPQUFPLEVBQUU7WUFDWCxPQUFPLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNuQjtRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFUyxxQkFBcUIsQ0FBQyxLQUFZLEVBQUUsUUFBZ0I7UUFDNUQsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUMvRCxJQUFJLE1BQU0sRUFBRTtZQUNWLE9BQU8sTUFBTSxDQUFDO1NBQ2Y7UUFFRCxNQUFNLFNBQVMsR0FBRyxJQUFJLGdCQUFNLENBQUMsS0FBSyxFQUFFLG1CQUFtQixRQUFRLEVBQUUsRUFBRTtZQUNqRSxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSx5QkFBZSxDQUFDO29CQUNsQixPQUFPLEVBQUU7d0JBQ1AsaUNBQWlDO3dCQUNqQyw0QkFBNEI7d0JBQzVCLG1CQUFtQjtxQkFDcEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULGlCQUFHLENBQUMsTUFBTSxDQUFDOzRCQUNULE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTzs0QkFDckIsT0FBTyxFQUFFLEtBQUs7NEJBQ2QsUUFBUSxFQUFFLFlBQVk7NEJBQ3RCLFlBQVksRUFBRSxRQUFRO3lCQUN2QixFQUFFLEtBQUssQ0FBQztxQkFDVjtpQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFDSCxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUMzRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDOztBQWpESCxrREFrREM7OztBQWhEQzs7R0FFRztBQUNhLGlDQUFhLEdBQUcsSUFBSSxHQUFHLEVBQWtCLENBQUM7QUErQzVEOztHQUVHO0FBQ0gsTUFBYSxpQkFBa0IsU0FBUSxtQkFBbUI7SUFFeEQsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNmLENBQUM7SUFFUyxhQUFhLENBQUMsSUFBbUQ7UUFDekUsSUFBSSxPQUFPLElBQUksQ0FBQyxLQUFLLEtBQUssUUFBUSxJQUFJLE1BQU0sSUFBSSxJQUFJLENBQUMsS0FBSztZQUN2RCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7WUFDM0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzNCO2FBQU0sSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFLLElBQUksQ0FBQyxLQUFnQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRTtZQUN4RixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7U0FDbkI7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU0sS0FBSyxDQUFDLFNBQXFCO1FBQ2hDLElBQUksU0FBUyxZQUFZLHdCQUFjLEVBQUU7WUFDdkMsTUFBTSxVQUFVLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxZQUFrQyxDQUFDLG9CQUFvQixDQUFDLENBQUM7WUFDcEgsSUFBSSxRQUFRLEdBQUcsU0FBUyxDQUFDO1lBQ3pCLElBQUksVUFBVSxZQUFZLEtBQUssRUFBRTtnQkFDL0IsS0FBSyxNQUFNLFNBQVMsSUFBSSxVQUFVLEVBQUU7b0JBQ2xDLElBQUksU0FBMEQsRUFBRTt3QkFDOUQsUUFBUSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7d0JBQ3pDLElBQUksUUFBUSxFQUFFOzRCQUNaLE1BQU07eUJBQ1A7cUJBQ0Y7aUJBQ0Y7YUFDRjtpQkFBTSxJQUFJLFVBQTJELEVBQUU7Z0JBQ3RFLFFBQVEsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO2FBQzNDO1lBRUQsSUFBSSxRQUFRLEVBQUU7Z0JBQ1osTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDNUMsSUFBSSxRQUFRLEVBQUU7b0JBQ1osU0FBUyxDQUFDLGFBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO2lCQUNwRzthQUNGO1NBQ0Y7SUFDSCxDQUFDOztBQXhDSCw4Q0F5Q0M7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxpQ0FBa0MsU0FBUSxtQkFBbUI7SUFDeEUsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNmLENBQUM7SUFFTSxLQUFLLENBQUMsU0FBcUI7UUFDaEMsSUFBSSxTQUFTLFlBQVksb0RBQTBCLEVBQUU7WUFDbkQsTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDbEMsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLFdBQVcsRUFNbEMsQ0FBQztZQUNGLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUNuRixJQUFJLE1BQU0sSUFBSSxLQUFLLEVBQUU7Z0JBQ25CLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBQ2pELElBQUksUUFBUSxFQUFFO29CQUNaLFNBQVMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO2lCQUNoRjthQUNGO1NBQ0Y7SUFDSCxDQUFDOztBQXZCSCw4RUF3QkM7OztBQUdEOztHQUVHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxtQkFBbUI7SUFTekQsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNiLElBQUksQ0FBQyxVQUFVLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxHQUFHLEVBQUUsQ0FBQztJQUNoQyxDQUFDO0lBRU0sS0FBSyxDQUFDLFNBQXFCO1FBQ2hDLElBQUksU0FBUyxZQUFZLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDNUMsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQztZQUM5QixJQUFJLENBQUMsaUJBQWlCLEdBQUssU0FBUyxDQUFDLElBQUksQ0FBQyxZQUEyQztpQkFDbEYsbUJBQThFLENBQUMsZ0JBQWdCLENBQUM7WUFFbkcsSUFBSSxJQUFJLENBQUMsaUJBQWlCLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUU7Z0JBQzNFLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDckU7WUFDRCxNQUFNLEtBQUssR0FBSyxTQUFTLENBQUMsSUFBSSxDQUFDLFlBQTJDO2lCQUN2RSxtQkFBOEUsQ0FBQyxLQUFLLENBQUM7WUFDeEYsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM1QyxJQUFJLE1BQU0sSUFBSSxjQUFjLEVBQUU7Z0JBQzVCLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBQzFELElBQUksUUFBUSxFQUFFO29CQUNaLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTt3QkFDdkIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7cUJBQ3JGO3lCQUFNO3dCQUNMLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFOzRCQUN6QyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQzt5QkFDaEM7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0QsSUFBSSxTQUFTLFlBQVksY0FBSSxFQUFFO1lBQzdCLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFDcEQsSUFBSSxTQUFTLENBQUMsT0FBTyxJQUFJLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtnQkFDL0MsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQztnQkFDOUIsSUFBSSxDQUFDLGNBQWMsR0FBRyxTQUFTLENBQUM7Z0JBQ2hDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO29CQUNqQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDO29CQUN2QyxJQUFJLFFBQVEsRUFBRTt3QkFDWixJQUFJLENBQUMsY0FBYyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztxQkFDckY7aUJBQ0Y7YUFDRjtTQUNGO0lBQ0gsQ0FBQzs7QUFyREgsZ0RBc0RDOzs7QUFHRDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSw0QkFBNkIsU0FBUSxtQkFBbUI7SUFPbkUsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNiLElBQUksQ0FBQyxRQUFRLEdBQUc7WUFDZCxJQUFJLGlCQUFpQixDQUFDLEtBQUssQ0FBQztZQUM1QixJQUFJLGlDQUFpQyxDQUFDLEtBQUssQ0FBQztZQUM1QyxJQUFJLGtCQUFrQixDQUFDLEtBQUssQ0FBQztTQUM5QixDQUFDO0lBQ0osQ0FBQztJQUVELEtBQUssQ0FBQyxTQUFxQjtRQUN6QixLQUFLLE1BQU0sT0FBTyxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDbkMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUMxQjtJQUNILENBQUM7O0FBcEJILG9FQXFCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGJhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1iYXRjaC1hbHBoYSc7XG5pbXBvcnQgeyBJQXNwZWN0LCBTdGFjaywgQXJuIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgYmF0Y2hfbGliIGZyb20gJ2F3cy1jZGstbGliL2F3cy1iYXRjaCc7XG5pbXBvcnQgeyBUYXNrRGVmaW5pdGlvbiwgQ2ZuVGFza0RlZmluaXRpb24gfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNzJztcbmltcG9ydCB7IFBvbGljeSwgUG9saWN5U3RhdGVtZW50LCBSb2xlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBTYWdlTWFrZXJDcmVhdGVUcmFpbmluZ0pvYiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zLXRhc2tzJztcbmltcG9ydCB7IElDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuY29uc3QgRk5fU1VCID0gJ0ZuOjpTdWInO1xuLyoqXG4gKiBDb25maWd1cmF0aW9uIHByb3BlcnRpZXMgZm9yIEVDUlJlcG9zaXRvcnlBc3BlY3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFQ1JSZXBvc2l0b3J5QXNwZWN0UHJvcHMge1xuICAvKipcbiAgICogT3ZlcnJpZGUgdGhlIEVDUiByZXBvc2l0b3J5IGFjY291bnQgaWQgb2YgdGhlIERvY2tlciBJbWFnZSBhc3NldHNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBwcm9jZXNzLmVudi5CU1NfSU1BR0VfQVNTRVRfQUNDT1VOVF9JRFxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2VBc3NldEFjY291bnRJZD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBYnRyYWN0IGFzcGVjdCBmb3IgRUNSIHJlcG9zaXRvcnkuXG4gKlxuICogWW91IG11c3QgcHJvdmlkZSB0aGUgYWNjb3VudCBpZCBpbiBwcm9wcyBvciBzZXQgQlNTX0lNQUdFX0FTU0VUX0FDQ09VTlRfSUQgaW4gZW52XG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBFQ1JSZXBvc2l0b3J5QXNwZWN0IGltcGxlbWVudHMgSUFzcGVjdCB7XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgc3RhdGljIHJlYWRvbmx5IF9yZXBvUG9saWNpZXMgPSBuZXcgTWFwPHN0cmluZywgUG9saWN5PigpO1xuICByZWFkb25seSBhY2NvdW50OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEVDUlJlcG9zaXRvcnlBc3BlY3RQcm9wcyA9IHt9KSB7XG4gICAgdGhpcy5hY2NvdW50ID0gcHJvcHMuaW1hZ2VBc3NldEFjY291bnRJZCA/PyBwcm9jZXNzLmVudi5CU1NfSU1BR0VfQVNTRVRfQUNDT1VOVF9JRCE7XG4gIH1cblxuICBhYnN0cmFjdCB2aXNpdChjb25zdHJ1Y3Q6IElDb25zdHJ1Y3QpOiB2b2lkO1xuXG4gIHByb3RlY3RlZCBnZXRSZXBvTmFtZShpbWFnZVVyaTogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBtYXRjaGVzID0gL1xcZHsxMn1cXC5ka3JcXC5lY3JcXC4uKlxcLyguKik6LiovZy5leGVjKGltYWdlVXJpKTtcbiAgICBpZiAobWF0Y2hlcykge1xuICAgICAgcmV0dXJuIG1hdGNoZXNbMV07XG4gICAgfVxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICBwcm90ZWN0ZWQgY3Jvc3NBY2NvdW50RUNSUG9saWN5KHN0YWNrOiBTdGFjaywgcmVwb05hbWU6IHN0cmluZyk6IFBvbGljeSB7XG4gICAgY29uc3QgcG9saWN5ID0gRUNSUmVwb3NpdG9yeUFzcGVjdC5fcmVwb1BvbGljaWVzLmdldChyZXBvTmFtZSk7XG4gICAgaWYgKHBvbGljeSkge1xuICAgICAgcmV0dXJuIHBvbGljeTtcbiAgICB9XG5cbiAgICBjb25zdCBuZXdQb2xpY3kgPSBuZXcgUG9saWN5KHN0YWNrLCBgQ3Jvc3NBY2NvdW50RUNSLSR7cmVwb05hbWV9YCwge1xuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAnZWNyOkJhdGNoQ2hlY2tMYXllckF2YWlsYWJpbGl0eScsXG4gICAgICAgICAgICAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLFxuICAgICAgICAgICAgJ2VjcjpCYXRjaEdldEltYWdlJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgQXJuLmZvcm1hdCh7XG4gICAgICAgICAgICAgIGFjY291bnQ6IHRoaXMuYWNjb3VudCxcbiAgICAgICAgICAgICAgc2VydmljZTogJ2VjcicsXG4gICAgICAgICAgICAgIHJlc291cmNlOiAncmVwb3NpdG9yeScsXG4gICAgICAgICAgICAgIHJlc291cmNlTmFtZTogcmVwb05hbWUsXG4gICAgICAgICAgICB9LCBzdGFjayksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuICAgIEVDUlJlcG9zaXRvcnlBc3BlY3QuX3JlcG9Qb2xpY2llcy5zZXQocmVwb05hbWUsIG5ld1BvbGljeSk7XG4gICAgcmV0dXJuIG5ld1BvbGljeTtcbiAgfVxufVxuXG4vKipcbiAqIFByb2Nlc3MgdGhlIGltYWdlIGFzc2V0cyBpbiBFQ1MgdGFzayBkZWZpbml0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBFQ1NUYXNrRGVmaW5pdGlvbiBleHRlbmRzIEVDUlJlcG9zaXRvcnlBc3BlY3Qge1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBFQ1JSZXBvc2l0b3J5QXNwZWN0UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBoYXNCZVJlcGxhY2VkKHByb3A6IENmblRhc2tEZWZpbml0aW9uLkNvbnRhaW5lckRlZmluaXRpb25Qcm9wZXJ0eSk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHR5cGVvZiBwcm9wLmltYWdlID09PSAnb2JqZWN0JyAmJiBGTl9TVUIgaW4gcHJvcC5pbWFnZSAmJlxuICAgICAgKHByb3AuaW1hZ2VbRk5fU1VCXSBhcyBzdHJpbmcpLmluZGV4T2YodGhpcy5hY2NvdW50KSA+IC0xKSB7XG4gICAgICByZXR1cm4gcHJvcC5pbWFnZVtGTl9TVUJdO1xuICAgIH0gZWxzZSBpZiAocHJvcC5pbWFnZSAmJiAocHJvcC5pbWFnZSBhcyBzdHJpbmcpICYmIHByb3AuaW1hZ2UuaW5kZXhPZih0aGlzLmFjY291bnQpID4gLTEpIHtcbiAgICAgIHJldHVybiBwcm9wLmltYWdlO1xuICAgIH1cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgcHVibGljIHZpc2l0KGNvbnN0cnVjdDogSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIGlmIChjb25zdHJ1Y3QgaW5zdGFuY2VvZiBUYXNrRGVmaW5pdGlvbikge1xuICAgICAgY29uc3QgY29udGFpbmVycyA9IGNvbnN0cnVjdC5zdGFjay5yZXNvbHZlKChjb25zdHJ1Y3Qubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuVGFza0RlZmluaXRpb24pLmNvbnRhaW5lckRlZmluaXRpb25zKTtcbiAgICAgIGxldCBpbWFnZVVyaSA9IHVuZGVmaW5lZDtcbiAgICAgIGlmIChjb250YWluZXJzIGluc3RhbmNlb2YgQXJyYXkpIHtcbiAgICAgICAgZm9yIChjb25zdCBjb250YWluZXIgb2YgY29udGFpbmVycykge1xuICAgICAgICAgIGlmIChjb250YWluZXIgYXMgQ2ZuVGFza0RlZmluaXRpb24uQ29udGFpbmVyRGVmaW5pdGlvblByb3BlcnR5KSB7XG4gICAgICAgICAgICBpbWFnZVVyaSA9IHRoaXMuaGFzQmVSZXBsYWNlZChjb250YWluZXIpO1xuICAgICAgICAgICAgaWYgKGltYWdlVXJpKSB7XG4gICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSBlbHNlIGlmIChjb250YWluZXJzIGFzIENmblRhc2tEZWZpbml0aW9uLkNvbnRhaW5lckRlZmluaXRpb25Qcm9wZXJ0eSkge1xuICAgICAgICBpbWFnZVVyaSA9IHRoaXMuaGFzQmVSZXBsYWNlZChjb250YWluZXJzKTtcbiAgICAgIH1cblxuICAgICAgaWYgKGltYWdlVXJpKSB7XG4gICAgICAgIGNvbnN0IHJlcG9OYW1lID0gdGhpcy5nZXRSZXBvTmFtZShpbWFnZVVyaSk7XG4gICAgICAgIGlmIChyZXBvTmFtZSkge1xuICAgICAgICAgIGNvbnN0cnVjdC5leGVjdXRpb25Sb2xlIS5hdHRhY2hJbmxpbmVQb2xpY3kodGhpcy5jcm9zc0FjY291bnRFQ1JQb2xpY3koY29uc3RydWN0LnN0YWNrLCByZXBvTmFtZSkpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogUHJvY2VzcyB0aGUgaW1hZ2UgYXNzZXRzIGluIFNhZ2VNYWtlciB0cmFpbmluZyBqb2IgaW4gU3RlcCBGdW5jdGlvbnNcbiAqL1xuZXhwb3J0IGNsYXNzIFN0ZXBGdW5jdGlvbnNTYWdlTWFrZXJUcmFpbmluZ0pvYiBleHRlbmRzIEVDUlJlcG9zaXRvcnlBc3BlY3Qge1xuICBjb25zdHJ1Y3Rvcihwcm9wczogRUNSUmVwb3NpdG9yeUFzcGVjdFByb3BzID0ge30pIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cblxuICBwdWJsaWMgdmlzaXQoY29uc3RydWN0OiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgaWYgKGNvbnN0cnVjdCBpbnN0YW5jZW9mIFNhZ2VNYWtlckNyZWF0ZVRyYWluaW5nSm9iKSB7XG4gICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKGNvbnN0cnVjdCk7XG4gICAgICBjb25zdCBzdGF0ZSA9IGNvbnN0cnVjdC50b1N0YXRlSnNvbigpIGFzIHtcbiAgICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICAgIEFsZ29yaXRobVNwZWNpZmljYXRpb246IHtcbiAgICAgICAgICAgIFRyYWluaW5nSW1hZ2U6IGFueTtcbiAgICAgICAgICB9O1xuICAgICAgICB9O1xuICAgICAgfTtcbiAgICAgIGNvbnN0IGltYWdlID0gc3RhY2sucmVzb2x2ZShzdGF0ZS5QYXJhbWV0ZXJzLkFsZ29yaXRobVNwZWNpZmljYXRpb24uVHJhaW5pbmdJbWFnZSk7XG4gICAgICBpZiAoRk5fU1VCIGluIGltYWdlKSB7XG4gICAgICAgIGNvbnN0IHJlcG9OYW1lID0gdGhpcy5nZXRSZXBvTmFtZShpbWFnZVtGTl9TVUJdKTtcbiAgICAgICAgaWYgKHJlcG9OYW1lKSB7XG4gICAgICAgICAgY29uc3RydWN0LnJvbGUuYXR0YWNoSW5saW5lUG9saWN5KHRoaXMuY3Jvc3NBY2NvdW50RUNSUG9saWN5KHN0YWNrLCByZXBvTmFtZSkpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG59XG5cblxuLyoqXG4gKiBQcm9jZXNzIHRoZSBpbWFnZSBhc3NldHMgaW4gQVdTIEJhdGNoIGpvYlxuICovXG5leHBvcnQgY2xhc3MgQmF0Y2hKb2JEZWZpbml0aW9uIGV4dGVuZHMgRUNSUmVwb3NpdG9yeUFzcGVjdCB7XG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHJlYWRvbmx5IF9yZXBvTmFtZXM6IHN0cmluZ1tdO1xuICBwcml2YXRlIF9leGVjdXRpb25Sb2xlPzogUm9sZTtcbiAgcHJpdmF0ZSBfZXhlY3V0aW9uUm9sZUFybj86IHN0cmluZztcbiAgcHJpdmF0ZSBfYWxsUm9sZXNNYXA6IE1hcDxzdHJpbmcsIFJvbGU+O1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBFQ1JSZXBvc2l0b3J5QXNwZWN0UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgICB0aGlzLl9yZXBvTmFtZXMgPSBbXTtcbiAgICB0aGlzLl9hbGxSb2xlc01hcCA9IG5ldyBNYXAoKTtcbiAgfVxuXG4gIHB1YmxpYyB2aXNpdChjb25zdHJ1Y3Q6IElDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICBpZiAoY29uc3RydWN0IGluc3RhbmNlb2YgYmF0Y2guSm9iRGVmaW5pdGlvbikge1xuICAgICAgY29uc3Qgc3RhY2sgPSBjb25zdHJ1Y3Quc3RhY2s7XG4gICAgICB0aGlzLl9leGVjdXRpb25Sb2xlQXJuID0gKChjb25zdHJ1Y3Qubm9kZS5kZWZhdWx0Q2hpbGQgYXMgYmF0Y2hfbGliLkNmbkpvYkRlZmluaXRpb24pXG4gICAgICAgIC5jb250YWluZXJQcm9wZXJ0aWVzIGFzIGJhdGNoX2xpYi5DZm5Kb2JEZWZpbml0aW9uLkNvbnRhaW5lclByb3BlcnRpZXNQcm9wZXJ0eSkuZXhlY3V0aW9uUm9sZUFybjtcblxuICAgICAgaWYgKHRoaXMuX2V4ZWN1dGlvblJvbGVBcm4gJiYgdGhpcy5fYWxsUm9sZXNNYXAuZ2V0KHRoaXMuX2V4ZWN1dGlvblJvbGVBcm4pKSB7XG4gICAgICAgIHRoaXMuX2V4ZWN1dGlvblJvbGUgPSB0aGlzLl9hbGxSb2xlc01hcC5nZXQodGhpcy5fZXhlY3V0aW9uUm9sZUFybik7XG4gICAgICB9XG4gICAgICBjb25zdCBpbWFnZSA9ICgoY29uc3RydWN0Lm5vZGUuZGVmYXVsdENoaWxkIGFzIGJhdGNoX2xpYi5DZm5Kb2JEZWZpbml0aW9uKVxuICAgICAgICAuY29udGFpbmVyUHJvcGVydGllcyBhcyBiYXRjaF9saWIuQ2ZuSm9iRGVmaW5pdGlvbi5Db250YWluZXJQcm9wZXJ0aWVzUHJvcGVydHkpLmltYWdlO1xuICAgICAgY29uc3QgaW1hZ2VfcmVzb2x2ZWQgPSBzdGFjay5yZXNvbHZlKGltYWdlKTtcbiAgICAgIGlmIChGTl9TVUIgaW4gaW1hZ2VfcmVzb2x2ZWQpIHtcbiAgICAgICAgY29uc3QgcmVwb05hbWUgPSB0aGlzLmdldFJlcG9OYW1lKGltYWdlX3Jlc29sdmVkW0ZOX1NVQl0pO1xuICAgICAgICBpZiAocmVwb05hbWUpIHtcbiAgICAgICAgICBpZiAodGhpcy5fZXhlY3V0aW9uUm9sZSkge1xuICAgICAgICAgICAgdGhpcy5fZXhlY3V0aW9uUm9sZS5hdHRhY2hJbmxpbmVQb2xpY3kodGhpcy5jcm9zc0FjY291bnRFQ1JQb2xpY3koc3RhY2ssIHJlcG9OYW1lKSk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGlmICh0aGlzLl9yZXBvTmFtZXMuaW5kZXhPZihyZXBvTmFtZSkgPCAwKSB7XG4gICAgICAgICAgICAgIHRoaXMuX3JlcG9OYW1lcy5wdXNoKHJlcG9OYW1lKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgaWYgKGNvbnN0cnVjdCBpbnN0YW5jZW9mIFJvbGUpIHtcbiAgICAgIHRoaXMuX2FsbFJvbGVzTWFwLnNldChjb25zdHJ1Y3Qucm9sZUFybiwgY29uc3RydWN0KTtcbiAgICAgIGlmIChjb25zdHJ1Y3Qucm9sZUFybiA9PSB0aGlzLl9leGVjdXRpb25Sb2xlQXJuKSB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gY29uc3RydWN0LnN0YWNrO1xuICAgICAgICB0aGlzLl9leGVjdXRpb25Sb2xlID0gY29uc3RydWN0O1xuICAgICAgICB3aGlsZSAodGhpcy5fcmVwb05hbWVzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICBjb25zdCByZXBvTmFtZSA9IHRoaXMuX3JlcG9OYW1lcy5wb3AoKTtcbiAgICAgICAgICBpZiAocmVwb05hbWUpIHtcbiAgICAgICAgICAgIHRoaXMuX2V4ZWN1dGlvblJvbGUuYXR0YWNoSW5saW5lUG9saWN5KHRoaXMuY3Jvc3NBY2NvdW50RUNSUG9saWN5KHN0YWNrLCByZXBvTmFtZSkpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfVxufVxuXG5cbi8qKlxuICogRGVmYXVsdCBFQ1IgYXNzZXQgYXNwZWN0LCBzdXBwb3J0IHVzaW5nIEVDUiBhc3NldHMgaW4gYmVsb3cgc2VydmljZXMsXG4gKlxuICogLSBFQ1MgdGFzayBkZWZpbml0aW9uXG4gKiAtIFNhZ2VNYWtlciB0cmFpbmluZyBqb2IgaW4gU3RlcCBGdW5jdGlvbnNcbiAqIC0gQVdTIEJhdGNoIGpvYlxuICogLSBBV1MgTGFtYmRhIGNvbnRhaW5lciBpbWFnZVxuICovXG5leHBvcnQgY2xhc3MgQ29tcG9zaXRlRUNSUmVwb3NpdG9yeUFzcGVjdCBleHRlbmRzIEVDUlJlcG9zaXRvcnlBc3BlY3Qge1xuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHJlYWRvbmx5IF9hc3BlY3RzOiBFQ1JSZXBvc2l0b3J5QXNwZWN0W107XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEVDUlJlcG9zaXRvcnlBc3BlY3RQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICAgIHRoaXMuX2FzcGVjdHMgPSBbXG4gICAgICBuZXcgRUNTVGFza0RlZmluaXRpb24ocHJvcHMpLFxuICAgICAgbmV3IFN0ZXBGdW5jdGlvbnNTYWdlTWFrZXJUcmFpbmluZ0pvYihwcm9wcyksXG4gICAgICBuZXcgQmF0Y2hKb2JEZWZpbml0aW9uKHByb3BzKSxcbiAgICBdO1xuICB9XG5cbiAgdmlzaXQoY29uc3RydWN0OiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgZm9yIChjb25zdCBfYXNwZWN0IG9mIHRoaXMuX2FzcGVjdHMpIHtcbiAgICAgIF9hc3BlY3QudmlzaXQoY29uc3RydWN0KTtcbiAgICB9XG4gIH1cbn0iXX0=