"""BMT basics."""
from typing import List, Optional

from .data import (
    all_classes, all_elements, all_slots,
    all_types, ancestors, descendants, children, parent, element,
)
from .util import with_formatting


class Toolkit():
    """Biolink model toolkit - lite!"""

    def __init__(self, schema=None):
        """Initialize."""
        if schema is not None:
            raise ValueError("bmt-lite does not support the `schema` argument. The biolink model version is dictated by the library flavor you installed.")

    def get_all_classes(self) -> List[str]:
        """Get all classes."""
        return all_classes

    def get_all_slots(self) -> List[str]:
        """Get all slots."""
        return all_slots

    def get_all_types(self) -> List[str]:
        """Get all types."""
        return all_types

    def get_all_elements(self) -> List[str]:
        """Get all elements."""
        return all_elements

    @with_formatting()
    def get_ancestors(
        self,
        name: str,
        reflexive: bool = True,
    ) -> List[str]:
        """Get ancestors."""
        if name not in ancestors:
            return []
        if reflexive:
            return ancestors[name] + [name]
        else:
            return ancestors[name]

    @with_formatting()
    def get_descendants(
        self,
        name: str,
        reflexive: bool = True,
    ) -> List[str]:
        """Get descendants."""
        if name not in descendants:
            return []
        if reflexive:
            return descendants[name] + [name]
        else:
            return descendants[name]

    @with_formatting()
    def get_children(
        self,
        name: str,
    ) -> List[str]:
        """Get children."""
        return children.get(name, [])

    @with_formatting()
    def get_parent(
        self,
        name: str,
    ) -> Optional[str]:
        """Get parent."""
        return parent.get(name, None)

    @with_formatting()
    def get_element(
        self,
        name: str,
    ) -> Optional["Element"]:
        """Get element."""
        if name in all_classes:
            return ClassDefinition(name, **element.get(name, dict()))
        elif name in all_slots:
            return SlotDefinition(name, **element.get(name, dict()))
        return None


class Element():
    """Biolink model element."""

    def __init__(self, name: str):
        """Initialize."""
        self.name: str = name


class SlotDefinition(Element):
    """Slot definition."""

    def __init__(
        self,
        name: str,
        symmetric: bool = False,
        inverse: Optional[str] = None,
    ):
        """Initialize."""
        super().__init__(name)
        self.symmetric: bool = symmetric
        self.inverse: Optional[str] = inverse


class ClassDefinition(Element):
    """Class definition."""

    def __init__(
        self,
        name: str,
        id_prefixes: List[str],
    ):
        """Initialize."""
        super().__init__(name)
        self.id_prefixes: List[str] = id_prefixes
