"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatFiles = void 0;
const tslib_1 = require("tslib");
const path = require("path");
const json_1 = require("nx/src/generators/utils/json");
const project_configuration_1 = require("nx/src/generators/utils/project-configuration");
const object_sort_1 = require("nx/src/utils/object-sort");
/**
 * Formats all the created or updated files using Prettier
 * @param tree - the file system tree
 */
function formatFiles(tree) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let prettier;
        try {
            prettier = yield Promise.resolve().then(() => require('prettier'));
        }
        catch (_a) { }
        sortWorkspaceJson(tree);
        sortTsConfig(tree);
        if (!prettier)
            return;
        const files = new Set(tree.listChanges().filter((file) => file.type !== 'DELETE'));
        yield Promise.all(Array.from(files).map((file) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const systemPath = path.join(tree.root, file.path);
            let options = {
                filepath: systemPath,
            };
            const resolvedOptions = yield prettier.resolveConfig(systemPath, {
                editorconfig: true,
            });
            if (!resolvedOptions) {
                return;
            }
            options = Object.assign(Object.assign({}, options), resolvedOptions);
            if (file.path.endsWith('.swcrc')) {
                options.parser = 'json';
            }
            const support = yield prettier.getFileInfo(systemPath, options);
            if (support.ignored || !support.inferredParser) {
                return;
            }
            try {
                tree.write(file.path, prettier.format(file.content.toString('utf-8'), options));
            }
            catch (e) {
                console.warn(`Could not format ${file.path}. Error: "${e.message}"`);
            }
        })));
    });
}
exports.formatFiles = formatFiles;
function sortWorkspaceJson(tree) {
    const workspaceJsonPath = (0, project_configuration_1.getWorkspacePath)(tree);
    if (!workspaceJsonPath) {
        return;
    }
    try {
        const workspaceJson = (0, json_1.readJson)(tree, workspaceJsonPath);
        if (Object.entries(workspaceJson.projects).length !== 0) {
            const sortedProjects = (0, object_sort_1.sortObjectByKeys)(workspaceJson.projects);
            (0, json_1.writeJson)(tree, workspaceJsonPath, Object.assign(Object.assign({}, workspaceJson), { projects: sortedProjects }));
        }
    }
    catch (e) {
        // catch noop
    }
}
function sortTsConfig(tree) {
    try {
        const tsConfigPath = getRootTsConfigPath(tree);
        if (!tsConfigPath) {
            return;
        }
        (0, json_1.updateJson)(tree, tsConfigPath, (tsconfig) => (Object.assign(Object.assign({}, tsconfig), { compilerOptions: Object.assign(Object.assign({}, tsconfig.compilerOptions), { paths: (0, object_sort_1.sortObjectByKeys)(tsconfig.compilerOptions.paths) }) })));
    }
    catch (e) {
        // catch noop
    }
}
function getRootTsConfigPath(tree) {
    for (const path of ['tsconfig.base.json', 'tsconfig.json']) {
        if (tree.exists(path)) {
            return path;
        }
    }
    return null;
}
//# sourceMappingURL=format-files.js.map