"""
Classical Force-field Inspired Descriptors (CFID)
Find details in:
https://journals.aps.org/prmaterials/abstract/10.1103/PhysRevMaterials.2.083801
"""
from jarvis.analysis.structure.neighbors import NeighborsAnalysis
from numpy.linalg import norm, solve
from jarvis.core.specie import Specie
import matplotlib.pyplot as plt
from jarvis.core.atoms import Atoms, VacuumPadding
from jarvis.core.specie import get_descrp_arr_name
from jarvis.io.vasp.inputs import Poscar
from collections import defaultdict
from math import pi
from operator import itemgetter
import time, itertools, collections, math, os, json, sys
import numpy as np
from math import log


class CFID(object):
    def __init__(self, atoms):
        self._atoms = atoms

    def get_comp_descp(
        self,
        jcell=True,
        jmean_chem=True,
        jmean_chg=True,
        jrdf=False,
        jrdf_adf=True,
        print_names=False,
    ):
        """
        Get chemo-structural CFID decriptors

        Args:
        struct: Structure object
        jcell: whether to use cell-size descriptors
        jmean_chem: whether to use average chemical descriptors
        jmean_chg: whether to use average charge distribution descriptors
        jmean_rdf: whether to use radial distribution descriptors
        jrdf_adf: whether to use radial as well as angle distribution descriptors
        print_names: whether to print names of descriptors
        Returns:
          cat: catenated final descriptors
        """
        cat = []
        s = self._atoms
        cell = []
        mean_chem = []
        rdf = []
        adf = []
        nn = []
        mean_chg = []
        adfa = []
        adfb = []
        ddf = []

        if jmean_chem == True:
            el_dict = s.composition._content
            # print (el_dict,type(el_dict))
            arr = []
            for k, v in el_dict.items():
                des = Specie(k).get_descrp_arr
                arr.append(des)
            mean_chem = np.mean(np.array(arr), axis=0)
            # print ('mean_chem',len(mean_chem))

        if jcell == True:
            v_pa = round(float(s.volume) / float(s.num_atoms), 5)
            logv_pa = round(log(v_pa), 5)
            pf = s.packing_fraction
            density = round(s.density, 5)
            cell = np.array([v_pa, logv_pa, pf, density])
            # print ('jcell',len(cell))

        if jrdf == True:
            Nbrs = NeighborsAnalysis(s)
            _, distrdf, nn = Nbrs.get_rdf()
            rdf = np.array(distrdf)
            print("rdf", len(rdf))

        if jrdf_adf == True:
            try:
                adfa = np.zeros(179)
                adfb = np.zeros(179)
                ddf = np.zeros(179)
                rdf = np.zeros(100)
                nn = np.zeros(100)
                distributions = NeighborsAnalysis(s).get_all_distributions
                rdf = distributions["rdf"]
                nn = distributions["nn"]
                adfa = distributions["adfa"]
                adfb = distributions["adfb"]
                ddf = distributions["ddf"]

            except:
                pass
            adfa = np.array(adfa)
            adfb = np.array(adfb)
            rdf = np.array(rdf)
            ddf = np.array(ddf)
            nn = np.array(nn)
            # print ('adfa',len(adfa))
            # print ('ddf',len(ddf))
            # print ('adfb',len(adfb))
            # print ('rdf',len(rdf))
            # print ('nn',len(nn))

        if jmean_chg == True:
            chgarr = []
            el_dict = s.composition._content
            for k, v in el_dict.items():
                chg = Specie(k).get_chgdescrp_arr
                chgarr.append(chg)
            mean_chg = np.mean(chgarr, axis=0)
            # print ('mean_chg',len(mean_chg))

        if print_names == True:
            nmes = []
            chem_nm = get_descrp_arr_name()
            for d, nm in zip(
                [mean_chem, cell, mean_chg, rdf, adfa, adfb, ddf, nn],
                ["mean_chem", "cell", "mean_chg", "rdf", "adfa", "adfb", "ddf", "nn"],
            ):
                if len(d) != 0:
                    for ff, dd in enumerate(d):
                        cat.append(dd)
                        if nm == "mean_chem":
                            tag = chem_nm[ff]
                        else:
                            tag = str(nm) + str("_") + str(ff)
                        nmes.append(str(tag))
            cat = np.array(cat).astype(float)
            # print (nmes,len(nmes))
            return nmes
        else:
            for d, nm in zip(
                [mean_chem, cell, mean_chg, rdf, adfa, adfb, ddf, nn],
                ["mean_chem", "cell", "mean_chg", "rdf", "adfa", "adfb", "ddf", "nn"],
            ):
                if len(d) != 0 :
                #if d != []:
                    for ff, dd in enumerate(d):
                        cat.append(dd)
            cat = np.array(cat).astype(float)
        return cat

"""
if __name__ == "__main__":
    box = [[2.715, 2.715, 0], [0, 2.715, 2.715], [2.715, 0, 2.715]]
    coords = [[0, 0, 0], [0.25, 0.25, 0.25]]
    elements = ["Si", "Si"]
    p = Poscar.from_file(
        "/rk2/knc6/JARVIS-DFT/2D-1L/POSCAR-mp-2815-1L.vasp_PBEBO/MAIN-RELAX-Surf-mp-2815/POSCAR"
    ).atoms
    print(p)
    v = VacuumPadding(p, vacuum=20.0).get_effective_2d_slab()
    v = VacuumPadding(p, vacuum=20.0).get_effective_molecule()
    print()
    print(v)
    import sys

    cfid_names = [
        "bp_mult_atom_rad",
        "hfus_add_bp",
        "elec_aff_mult_voro_coord",
        "mol_vol_subs_atom_mass",
        "is_halogen",
        "atom_mass_subs_first_ion_en",
        "row",
        "mol_vol_mult_atom_mass",
        "voro_coord_divi_therm_cond",
        "voro_coord_subs_mp",
        "polzbl_mult_atom_rad",
        "elec_aff_mult_X",
        "GV",
        "nfunfill",
        "voro_coord_subs_therm_cond",
        "mp_divi_therm_cond",
        "elec_aff_add_X",
        "mol_vol_add_bp",
        "C-9",
        "C-8",
        "C-7",
        "C-6",
        "C-5",
        "C-4",
        "C-3",
        "C-2",
        "C-1",
        "C-0",
        "mp_subs_elec_aff",
        "hfus_add_elec_aff",
        "elec_aff_subs_first_ion_en",
        "therm_cond_subs_mol_vol",
        "X_subs_mol_vol",
        "first_ion_en_subs_hfus",
        "first_ion_en_add_voro_coord",
        "first_ion_en_divi_atom_mass",
        "atom_rad_subs_polzbl",
        "me1",
        "me3",
        "me2",
        "elec_aff_mult_atom_mass",
        "elec_aff_add_atom_mass",
        "therm_cond_subs_elec_aff",
        "atom_rad_divi_atom_mass",
        "atom_rad_divi_voro_coord",
        "max_oxid_s",
        "polzbl_mult_first_ion_en",
        "mp_divi_voro_coord",
        "mol_vol_divi_hfus",
        "mp_divi_atom_mass",
        "therm_cond_subs_mp",
        "npvalence",
        "mol_vol_subs_atom_rad",
        "atom_rad_divi_hfus",
        "mp",
        "hfus_add_atom_rad",
        "atom_mass_add_voro_coord",
        "atom_mass_divi_atom_rad",
        "hfus_divi_atom_rad",
        "polzbl_mult_X",
        "first_ion_en",
        "bp_add_X",
        "therm_cond_subs_atom_rad",
        "first_ion_en_subs_voro_coord",
        "first_ion_en_subs_polzbl",
        "hfus_subs_X",
        "first_ion_en_mult_mol_vol",
        "first_ion_en_mult_mp",
        "bp_mult_atom_mass",
        "bp_subs_mp",
        "polzbl_add_atom_rad",
        "therm_cond_subs_hfus",
        "mol_vol_add_mp",
        "first_ion_en_subs_elec_aff",
        "atom_mass_add_therm_cond",
        "atom_mass_divi_therm_cond",
        "first_ion_en_divi_mol_vol",
        "polzbl_divi_mp",
        "elec_aff_subs_therm_cond",
        "ndunfill",
        "op_eg",
        "atom_mass_subs_mp",
        "mol_vol_subs_voro_coord",
        "jv_enp",
        "polzbl_mult_atom_mass",
        "first_ion_en_add_mp",
        "first_ion_en_add_atom_mass",
        "mol_vol_mult_bp",
        "mol_vol_add_atom_mass",
        "nfvalence",
        "hfus_subs_voro_coord",
        "voro_coord_divi_mol_vol",
        "hfus_mult_first_ion_en",
        "mp_mult_atom_mass",
        "polzbl",
        "therm_cond_subs_first_ion_en",
        "mol_vol_subs_elec_aff",
        "therm_cond_divi_atom_rad",
        "first_ion_en_subs_atom_mass",
        "atom_rad_mult_X",
        "mp_subs_mol_vol",
        "hfus_divi_bp",
        "mol_vol_subs_therm_cond",
        "mol_vol_subs_first_ion_en",
        "oq_bg",
        "therm_cond_mult_X",
        "mol_vol_subs_X",
        "elec_aff_subs_X",
        "atom_rad",
        "bp_subs_elec_aff",
        "polzbl_divi_first_ion_en",
        "mp_mult_atom_rad",
        "polzbl_mult_mp",
        "first_ion_en_divi_bp",
        "atom_mass",
        "bp_subs_therm_cond",
        "bp_add_atom_mass",
        "polzbl_divi_hfus",
        "hfus_mult_atom_rad",
        "first_ion_en_divi_voro_coord",
        "mp_add_therm_cond",
        "hfus_mult_atom_mass",
        "hfus_divi_mol_vol",
        "hfus_divi_polzbl",
        "first_ion_en_divi_hfus",
        "elec_aff_subs_atom_mass",
        "hfus_mult_voro_coord",
        "polzbl_subs_X",
        "elec_aff_mult_therm_cond",
        "atom_rad_subs_atom_mass",
        "is_alkali",
        "voro_coord_subs_atom_mass",
        "mol_vol_mult_X",
        "hfus_subs_mp",
        "mol_vol_divi_atom_rad",
        "atom_mass_subs_polzbl",
        "atom_rad_divi_mol_vol",
        "elec_aff_add_bp",
        "C-13",
        "C-12",
        "C-11",
        "C-10",
        "C-17",
        "C-16",
        "C-15",
        "C-14",
        "hfus_divi_voro_coord",
        "polzbl_subs_atom_mass",
        "C-19",
        "C-18",
        "X_subs_atom_rad",
        "X_subs_elec_aff",
        "voro_coord",
        "mp_add_atom_mass",
        "elec_aff_add_mp",
        "is_noble_gas",
        "polzbl_add_bp",
        "hfus_add_polzbl",
        "X_subs_first_ion_en",
        "mp_subs_first_ion_en",
        "atom_rad_mult_voro_coord",
        "mol_vol_subs_mp",
        "atom_mass_divi_mp",
        "voro_coord_subs_polzbl",
        "bp_divi_first_ion_en",
        "polzbl_add_mp",
        "polzbl_add_therm_cond",
        "hfus_subs_atom_rad",
        "first_ion_en_divi_therm_cond",
        "X_subs_atom_mass",
        "mol_vol_subs_bp",
        "polzbl_mult_therm_cond",
        "bp_subs_voro_coord",
        "e1",
        "e3",
        "e2",
        "atom_rad_subs_X",
        "X_add_voro_coord",
        "mp_mult_voro_coord",
        "mp_subs_atom_mass",
        "mp_add_voro_coord",
        "therm_cond_divi_hfus",
        "bp_divi_atom_mass",
        "first_ion_en_mult_X",
        "hfus_mult_mol_vol",
        "voro_coord_divi_bp",
        "mol_vol_divi_voro_coord",
        "atom_mass_divi_voro_coord",
        "is_lanthanoid",
        "mol_vol_divi_polzbl",
        "atom_mass_subs_therm_cond",
        "atom_rad_divi_therm_cond",
        "atom_mass_divi_polzbl",
        "bp_add_atom_rad",
        "voro_coord_subs_hfus",
        "polzbl_subs_bp",
        "polzbl_divi_bp",
        "elec_aff_mult_mp",
        "ndvalence",
        "atom_mass_divi_mol_vol",
        "mp_divi_mol_vol",
        "therm_cond_subs_voro_coord",
        "atom_mass_subs_atom_rad",
        "polzbl_divi_voro_coord",
        "polzbl_subs_mp",
        "polzbl_subs_therm_cond",
        "KV",
        "min_oxid_s",
        "elec_aff_mult_atom_rad",
        "therm_cond_subs_polzbl",
        "voro_coord_subs_bp",
        "bp_add_voro_coord",
        "nsunfill",
        "mp_mult_X",
        "atom_mass_mult_atom_rad",
        "first_ion_en_mult_voro_coord",
        "elec_aff_subs_voro_coord",
        "bp_divi_mp",
        "atom_rad_subs_mol_vol",
        "bp_subs_hfus",
        "atom_rad_subs_first_ion_en",
        "hfus_divi_mp",
        "first_ion_en_subs_mp",
        "mp_add_atom_rad",
        "hfus_add_atom_mass",
        "bp_subs_mol_vol",
        "hfus_add_first_ion_en",
        "bp_subs_atom_rad",
        "bp_divi_atom_rad",
        "C-26",
        "atom_mass_subs_hfus",
        "mp_subs_therm_cond",
        "polzbl_divi_atom_rad",
        "polzbl_add_mol_vol",
        "first_ion_en_add_bp",
        "hfus_subs_therm_cond",
        "mp_divi_hfus",
        "X_subs_bp",
        "bp_mult_mp",
        "atom_mass_mult_therm_cond",
        "mol_vol_divi_therm_cond",
        "voro_coord_subs_X",
        "first_ion_en_subs_mol_vol",
        "polzbl_add_elec_aff",
        "therm_cond_subs_atom_mass",
        "hfus_subs_bp",
        "atom_mass_subs_mol_vol",
        "X",
        "is_actinoid",
        "hfus_subs_first_ion_en",
        "first_ion_en_divi_mp",
        "atom_rad_add_therm_cond",
        "atom_rad_divi_polzbl",
        "hfus_divi_atom_mass",
        "mp_mult_therm_cond",
        "first_ion_en_add_X",
        "polzbl_add_voro_coord",
        "mol_vol_subs_polzbl",
        "therm_cond_divi_first_ion_en",
        "bp_mult_voro_coord",
        "atom_mass_add_atom_rad",
        "hfus_mult_mp",
        "voro_coord_divi_polzbl",
        "elec_aff_add_therm_cond",
        "C-28",
        "C-29",
        "hfus_add_voro_coord",
        "C-27",
        "C-24",
        "C-25",
        "C-22",
        "C-23",
        "C-20",
        "C-21",
        "therm_cond_divi_polzbl",
        "mol_vol_mult_mp",
        "mp_subs_X",
        "bp_subs_atom_mass",
        "first_ion_en_add_therm_cond",
        "voro_coord_divi_atom_mass",
        "atom_mass_subs_voro_coord",
        "hfus_add_therm_cond",
        "atom_rad_divi_mp",
        "mp_subs_voro_coord",
        "hfus_mult_polzbl",
        "bp_divi_hfus",
        "therm_cond_divi_voro_coord",
        "first_ion_en_subs_atom_rad",
        "atom_rad_subs_bp",
        "atom_rad_add_X",
        "mol_vol_add_voro_coord",
        "avg_ion_rad",
        "bp_divi_mol_vol",
        "atom_rad_subs_mp",
        "atom_mass_mult_voro_coord",
        "therm_cond_add_X",
        "atom_rad_subs_therm_cond",
        "polzbl_add_first_ion_en",
        "polzbl_mult_bp",
        "first_ion_en_mult_elec_aff",
        "elec_aff_subs_mol_vol",
        "nsvalence",
        "mol_vol_subs_hfus",
        "atom_mass_divi_bp",
        "bp_subs_polzbl",
        "bp_add_mp",
        "polzbl_divi_mol_vol",
        "voro_coord_subs_elec_aff",
        "elec_aff_add_atom_rad",
        "mol_vol_add_atom_rad",
        "hfus_mult_X",
        "mp_subs_hfus",
        "voro_coord_divi_first_ion_en",
        "atom_rad_divi_first_ion_en",
        "elec_aff_mult_mol_vol",
        "is_metalloid",
        "hfus_mult_therm_cond",
        "hfus_divi_therm_cond",
        "therm_cond_divi_mol_vol",
        "elec_aff",
        "mp_divi_atom_rad",
        "elec_aff_subs_polzbl",
        "polzbl_divi_atom_mass",
        "atom_rad_subs_elec_aff",
        "bp_divi_therm_cond",
        "atom_mass_mult_X",
        "mp_divi_polzbl",
        "elec_aff_subs_atom_rad",
        "voro_coord_divi_atom_rad",
        "hfus_subs_mol_vol",
        "mol_vol_add_therm_cond",
        "bp_mult_X",
        "therm_cond_divi_mp",
        "mp_subs_polzbl",
        "atom_rad_divi_bp",
        "atom_rad_subs_hfus",
        "polzbl_subs_elec_aff",
        "therm_cond_divi_atom_mass",
        "bp_divi_voro_coord",
        "bp_add_therm_cond",
        "hfus_divi_first_ion_en",
        "X_mult_voro_coord",
        "bp_subs_X",
        "polzbl_subs_voro_coord",
        "coulmn",
        "first_ion_en_divi_polzbl",
        "first_ion_en_mult_atom_mass",
        "mol_vol_mult_therm_cond",
        "bp_mult_therm_cond",
        "mol_vol",
        "mol_vol_mult_voro_coord",
        "atom_mass_subs_elec_aff",
        "atom_mass_divi_first_ion_en",
        "hfus_subs_elec_aff",
        "first_ion_en_mult_bp",
        "polzbl_mult_mol_vol",
        "polzbl_add_atom_mass",
        "elec_aff_mult_bp",
        "voro_coord_subs_atom_rad",
        "mp_add_X",
        "elec_aff_subs_bp",
        "polzbl_divi_therm_cond",
        "polzbl_mult_voro_coord",
        "bp",
        "therm_cond_mult_voro_coord",
        "elec_aff_subs_mp",
        "C-31",
        "C-30",
        "C-33",
        "C-32",
        "C-35",
        "C-34",
        "first_ion_en_subs_X",
        "is_transition_metal",
        "mp_subs_bp",
        "hfus_add_mol_vol",
        "block",
        "atom_rad_subs_voro_coord",
        "X_subs_therm_cond",
        "polzbl_subs_first_ion_en",
        "hfus_subs_atom_mass",
        "mol_vol_divi_bp",
        "first_ion_en_subs_therm_cond",
        "elec_aff_add_voro_coord",
        "voro_coord_divi_hfus",
        "therm_cond",
        "bp_subs_first_ion_en",
        "X_subs_voro_coord",
        "therm_cond_divi_bp",
        "mp_divi_bp",
        "first_ion_en_divi_atom_rad",
        "hfus_subs_polzbl",
        "mol_vol_divi_mp",
        "atom_rad_add_voro_coord",
        "mol_vol_mult_atom_rad",
        "first_ion_en_mult_atom_rad",
        "hfus_mult_bp",
        "voro_coord_subs_first_ion_en",
        "Z",
        "polzbl_subs_hfus",
        "polzbl_add_X",
        "is_alkaline",
        "npunfill",
        "first_ion_en_subs_bp",
        "voro_coord_divi_mp",
        "atom_mass_divi_hfus",
        "oq_enp",
        "elec_aff_subs_hfus",
        "atom_mass_subs_bp",
        "first_ion_en_add_elec_aff",
        "first_ion_en_add_mol_vol",
        "hfus_mult_elec_aff",
        "polzbl_mult_elec_aff",
        "hfus_add_X",
        "hfus_add_mp",
        "therm_cond_subs_bp",
        "bp_divi_polzbl",
        "X_subs_hfus",
        "atom_rad_mult_therm_cond",
        "therm_cond_add_voro_coord",
        "mop_eg",
        "X_subs_polzbl",
        "mol_vol_divi_first_ion_en",
        "mp_divi_first_ion_en",
        "voro_coord_subs_mol_vol",
        "elec_aff_add_mol_vol",
        "mol_vol_add_X",
        "atom_mass_add_X",
        "polzbl_subs_atom_rad",
        "mol_vol_divi_atom_mass",
        "polzbl_subs_mol_vol",
        "mp_subs_atom_rad",
        "X_subs_mp",
        "atom_mass_subs_X",
        "first_ion_en_add_atom_rad",
        "therm_cond_subs_X",
        "hfus",
        "first_ion_en_mult_therm_cond",
        "cell_0",
        "cell_1",
        "cell_2",
        "cell_3",
        "mean_chg_0",
        "mean_chg_1",
        "mean_chg_2",
        "mean_chg_3",
        "mean_chg_4",
        "mean_chg_5",
        "mean_chg_6",
        "mean_chg_7",
        "mean_chg_8",
        "mean_chg_9",
        "mean_chg_10",
        "mean_chg_11",
        "mean_chg_12",
        "mean_chg_13",
        "mean_chg_14",
        "mean_chg_15",
        "mean_chg_16",
        "mean_chg_17",
        "mean_chg_18",
        "mean_chg_19",
        "mean_chg_20",
        "mean_chg_21",
        "mean_chg_22",
        "mean_chg_23",
        "mean_chg_24",
        "mean_chg_25",
        "mean_chg_26",
        "mean_chg_27",
        "mean_chg_28",
        "mean_chg_29",
        "mean_chg_30",
        "mean_chg_31",
        "mean_chg_32",
        "mean_chg_33",
        "mean_chg_34",
        "mean_chg_35",
        "mean_chg_36",
        "mean_chg_37",
        "mean_chg_38",
        "mean_chg_39",
        "mean_chg_40",
        "mean_chg_41",
        "mean_chg_42",
        "mean_chg_43",
        "mean_chg_44",
        "mean_chg_45",
        "mean_chg_46",
        "mean_chg_47",
        "mean_chg_48",
        "mean_chg_49",
        "mean_chg_50",
        "mean_chg_51",
        "mean_chg_52",
        "mean_chg_53",
        "mean_chg_54",
        "mean_chg_55",
        "mean_chg_56",
        "mean_chg_57",
        "mean_chg_58",
        "mean_chg_59",
        "mean_chg_60",
        "mean_chg_61",
        "mean_chg_62",
        "mean_chg_63",
        "mean_chg_64",
        "mean_chg_65",
        "mean_chg_66",
        "mean_chg_67",
        "mean_chg_68",
        "mean_chg_69",
        "mean_chg_70",
        "mean_chg_71",
        "mean_chg_72",
        "mean_chg_73",
        "mean_chg_74",
        "mean_chg_75",
        "mean_chg_76",
        "mean_chg_77",
        "mean_chg_78",
        "mean_chg_79",
        "mean_chg_80",
        "mean_chg_81",
        "mean_chg_82",
        "mean_chg_83",
        "mean_chg_84",
        "mean_chg_85",
        "mean_chg_86",
        "mean_chg_87",
        "mean_chg_88",
        "mean_chg_89",
        "mean_chg_90",
        "mean_chg_91",
        "mean_chg_92",
        "mean_chg_93",
        "mean_chg_94",
        "mean_chg_95",
        "mean_chg_96",
        "mean_chg_97",
        "mean_chg_98",
        "mean_chg_99",
        "mean_chg_100",
        "mean_chg_101",
        "mean_chg_102",
        "mean_chg_103",
        "mean_chg_104",
        "mean_chg_105",
        "mean_chg_106",
        "mean_chg_107",
        "mean_chg_108",
        "mean_chg_109",
        "mean_chg_110",
        "mean_chg_111",
        "mean_chg_112",
        "mean_chg_113",
        "mean_chg_114",
        "mean_chg_115",
        "mean_chg_116",
        "mean_chg_117",
        "mean_chg_118",
        "mean_chg_119",
        "mean_chg_120",
        "mean_chg_121",
        "mean_chg_122",
        "mean_chg_123",
        "mean_chg_124",
        "mean_chg_125",
        "mean_chg_126",
        "mean_chg_127",
        "mean_chg_128",
        "mean_chg_129",
        "mean_chg_130",
        "mean_chg_131",
        "mean_chg_132",
        "mean_chg_133",
        "mean_chg_134",
        "mean_chg_135",
        "mean_chg_136",
        "mean_chg_137",
        "mean_chg_138",
        "mean_chg_139",
        "mean_chg_140",
        "mean_chg_141",
        "mean_chg_142",
        "mean_chg_143",
        "mean_chg_144",
        "mean_chg_145",
        "mean_chg_146",
        "mean_chg_147",
        "mean_chg_148",
        "mean_chg_149",
        "mean_chg_150",
        "mean_chg_151",
        "mean_chg_152",
        "mean_chg_153",
        "mean_chg_154",
        "mean_chg_155",
        "mean_chg_156",
        "mean_chg_157",
        "mean_chg_158",
        "mean_chg_159",
        "mean_chg_160",
        "mean_chg_161",
        "mean_chg_162",
        "mean_chg_163",
        "mean_chg_164",
        "mean_chg_165",
        "mean_chg_166",
        "mean_chg_167",
        "mean_chg_168",
        "mean_chg_169",
        "mean_chg_170",
        "mean_chg_171",
        "mean_chg_172",
        "mean_chg_173",
        "mean_chg_174",
        "mean_chg_175",
        "mean_chg_176",
        "mean_chg_177",
        "mean_chg_178",
        "mean_chg_179",
        "mean_chg_180",
        "mean_chg_181",
        "mean_chg_182",
        "mean_chg_183",
        "mean_chg_184",
        "mean_chg_185",
        "mean_chg_186",
        "mean_chg_187",
        "mean_chg_188",
        "mean_chg_189",
        "mean_chg_190",
        "mean_chg_191",
        "mean_chg_192",
        "mean_chg_193",
        "mean_chg_194",
        "mean_chg_195",
        "mean_chg_196",
        "mean_chg_197",
        "mean_chg_198",
        "mean_chg_199",
        "mean_chg_200",
        "mean_chg_201",
        "mean_chg_202",
        "mean_chg_203",
        "mean_chg_204",
        "mean_chg_205",
        "mean_chg_206",
        "mean_chg_207",
        "mean_chg_208",
        "mean_chg_209",
        "mean_chg_210",
        "mean_chg_211",
        "mean_chg_212",
        "mean_chg_213",
        "mean_chg_214",
        "mean_chg_215",
        "mean_chg_216",
        "mean_chg_217",
        "mean_chg_218",
        "mean_chg_219",
        "mean_chg_220",
        "mean_chg_221",
        "mean_chg_222",
        "mean_chg_223",
        "mean_chg_224",
        "mean_chg_225",
        "mean_chg_226",
        "mean_chg_227",
        "mean_chg_228",
        "mean_chg_229",
        "mean_chg_230",
        "mean_chg_231",
        "mean_chg_232",
        "mean_chg_233",
        "mean_chg_234",
        "mean_chg_235",
        "mean_chg_236",
        "mean_chg_237",
        "mean_chg_238",
        "mean_chg_239",
        "mean_chg_240",
        "mean_chg_241",
        "mean_chg_242",
        "mean_chg_243",
        "mean_chg_244",
        "mean_chg_245",
        "mean_chg_246",
        "mean_chg_247",
        "mean_chg_248",
        "mean_chg_249",
        "mean_chg_250",
        "mean_chg_251",
        "mean_chg_252",
        "mean_chg_253",
        "mean_chg_254",
        "mean_chg_255",
        "mean_chg_256",
        "mean_chg_257",
        "mean_chg_258",
        "mean_chg_259",
        "mean_chg_260",
        "mean_chg_261",
        "mean_chg_262",
        "mean_chg_263",
        "mean_chg_264",
        "mean_chg_265",
        "mean_chg_266",
        "mean_chg_267",
        "mean_chg_268",
        "mean_chg_269",
        "mean_chg_270",
        "mean_chg_271",
        "mean_chg_272",
        "mean_chg_273",
        "mean_chg_274",
        "mean_chg_275",
        "mean_chg_276",
        "mean_chg_277",
        "mean_chg_278",
        "mean_chg_279",
        "mean_chg_280",
        "mean_chg_281",
        "mean_chg_282",
        "mean_chg_283",
        "mean_chg_284",
        "mean_chg_285",
        "mean_chg_286",
        "mean_chg_287",
        "mean_chg_288",
        "mean_chg_289",
        "mean_chg_290",
        "mean_chg_291",
        "mean_chg_292",
        "mean_chg_293",
        "mean_chg_294",
        "mean_chg_295",
        "mean_chg_296",
        "mean_chg_297",
        "mean_chg_298",
        "mean_chg_299",
        "mean_chg_300",
        "mean_chg_301",
        "mean_chg_302",
        "mean_chg_303",
        "mean_chg_304",
        "mean_chg_305",
        "mean_chg_306",
        "mean_chg_307",
        "mean_chg_308",
        "mean_chg_309",
        "mean_chg_310",
        "mean_chg_311",
        "mean_chg_312",
        "mean_chg_313",
        "mean_chg_314",
        "mean_chg_315",
        "mean_chg_316",
        "mean_chg_317",
        "mean_chg_318",
        "mean_chg_319",
        "mean_chg_320",
        "mean_chg_321",
        "mean_chg_322",
        "mean_chg_323",
        "mean_chg_324",
        "mean_chg_325",
        "mean_chg_326",
        "mean_chg_327",
        "mean_chg_328",
        "mean_chg_329",
        "mean_chg_330",
        "mean_chg_331",
        "mean_chg_332",
        "mean_chg_333",
        "mean_chg_334",
        "mean_chg_335",
        "mean_chg_336",
        "mean_chg_337",
        "mean_chg_338",
        "mean_chg_339",
        "mean_chg_340",
        "mean_chg_341",
        "mean_chg_342",
        "mean_chg_343",
        "mean_chg_344",
        "mean_chg_345",
        "mean_chg_346",
        "mean_chg_347",
        "mean_chg_348",
        "mean_chg_349",
        "mean_chg_350",
        "mean_chg_351",
        "mean_chg_352",
        "mean_chg_353",
        "mean_chg_354",
        "mean_chg_355",
        "mean_chg_356",
        "mean_chg_357",
        "mean_chg_358",
        "mean_chg_359",
        "mean_chg_360",
        "mean_chg_361",
        "mean_chg_362",
        "mean_chg_363",
        "mean_chg_364",
        "mean_chg_365",
        "mean_chg_366",
        "mean_chg_367",
        "mean_chg_368",
        "mean_chg_369",
        "mean_chg_370",
        "mean_chg_371",
        "mean_chg_372",
        "mean_chg_373",
        "mean_chg_374",
        "mean_chg_375",
        "mean_chg_376",
        "mean_chg_377",
        "rdf_0",
        "rdf_1",
        "rdf_2",
        "rdf_3",
        "rdf_4",
        "rdf_5",
        "rdf_6",
        "rdf_7",
        "rdf_8",
        "rdf_9",
        "rdf_10",
        "rdf_11",
        "rdf_12",
        "rdf_13",
        "rdf_14",
        "rdf_15",
        "rdf_16",
        "rdf_17",
        "rdf_18",
        "rdf_19",
        "rdf_20",
        "rdf_21",
        "rdf_22",
        "rdf_23",
        "rdf_24",
        "rdf_25",
        "rdf_26",
        "rdf_27",
        "rdf_28",
        "rdf_29",
        "rdf_30",
        "rdf_31",
        "rdf_32",
        "rdf_33",
        "rdf_34",
        "rdf_35",
        "rdf_36",
        "rdf_37",
        "rdf_38",
        "rdf_39",
        "rdf_40",
        "rdf_41",
        "rdf_42",
        "rdf_43",
        "rdf_44",
        "rdf_45",
        "rdf_46",
        "rdf_47",
        "rdf_48",
        "rdf_49",
        "rdf_50",
        "rdf_51",
        "rdf_52",
        "rdf_53",
        "rdf_54",
        "rdf_55",
        "rdf_56",
        "rdf_57",
        "rdf_58",
        "rdf_59",
        "rdf_60",
        "rdf_61",
        "rdf_62",
        "rdf_63",
        "rdf_64",
        "rdf_65",
        "rdf_66",
        "rdf_67",
        "rdf_68",
        "rdf_69",
        "rdf_70",
        "rdf_71",
        "rdf_72",
        "rdf_73",
        "rdf_74",
        "rdf_75",
        "rdf_76",
        "rdf_77",
        "rdf_78",
        "rdf_79",
        "rdf_80",
        "rdf_81",
        "rdf_82",
        "rdf_83",
        "rdf_84",
        "rdf_85",
        "rdf_86",
        "rdf_87",
        "rdf_88",
        "rdf_89",
        "rdf_90",
        "rdf_91",
        "rdf_92",
        "rdf_93",
        "rdf_94",
        "rdf_95",
        "rdf_96",
        "rdf_97",
        "rdf_98",
        "rdf_99",
        "adfa_0",
        "adfa_1",
        "adfa_2",
        "adfa_3",
        "adfa_4",
        "adfa_5",
        "adfa_6",
        "adfa_7",
        "adfa_8",
        "adfa_9",
        "adfa_10",
        "adfa_11",
        "adfa_12",
        "adfa_13",
        "adfa_14",
        "adfa_15",
        "adfa_16",
        "adfa_17",
        "adfa_18",
        "adfa_19",
        "adfa_20",
        "adfa_21",
        "adfa_22",
        "adfa_23",
        "adfa_24",
        "adfa_25",
        "adfa_26",
        "adfa_27",
        "adfa_28",
        "adfa_29",
        "adfa_30",
        "adfa_31",
        "adfa_32",
        "adfa_33",
        "adfa_34",
        "adfa_35",
        "adfa_36",
        "adfa_37",
        "adfa_38",
        "adfa_39",
        "adfa_40",
        "adfa_41",
        "adfa_42",
        "adfa_43",
        "adfa_44",
        "adfa_45",
        "adfa_46",
        "adfa_47",
        "adfa_48",
        "adfa_49",
        "adfa_50",
        "adfa_51",
        "adfa_52",
        "adfa_53",
        "adfa_54",
        "adfa_55",
        "adfa_56",
        "adfa_57",
        "adfa_58",
        "adfa_59",
        "adfa_60",
        "adfa_61",
        "adfa_62",
        "adfa_63",
        "adfa_64",
        "adfa_65",
        "adfa_66",
        "adfa_67",
        "adfa_68",
        "adfa_69",
        "adfa_70",
        "adfa_71",
        "adfa_72",
        "adfa_73",
        "adfa_74",
        "adfa_75",
        "adfa_76",
        "adfa_77",
        "adfa_78",
        "adfa_79",
        "adfa_80",
        "adfa_81",
        "adfa_82",
        "adfa_83",
        "adfa_84",
        "adfa_85",
        "adfa_86",
        "adfa_87",
        "adfa_88",
        "adfa_89",
        "adfa_90",
        "adfa_91",
        "adfa_92",
        "adfa_93",
        "adfa_94",
        "adfa_95",
        "adfa_96",
        "adfa_97",
        "adfa_98",
        "adfa_99",
        "adfa_100",
        "adfa_101",
        "adfa_102",
        "adfa_103",
        "adfa_104",
        "adfa_105",
        "adfa_106",
        "adfa_107",
        "adfa_108",
        "adfa_109",
        "adfa_110",
        "adfa_111",
        "adfa_112",
        "adfa_113",
        "adfa_114",
        "adfa_115",
        "adfa_116",
        "adfa_117",
        "adfa_118",
        "adfa_119",
        "adfa_120",
        "adfa_121",
        "adfa_122",
        "adfa_123",
        "adfa_124",
        "adfa_125",
        "adfa_126",
        "adfa_127",
        "adfa_128",
        "adfa_129",
        "adfa_130",
        "adfa_131",
        "adfa_132",
        "adfa_133",
        "adfa_134",
        "adfa_135",
        "adfa_136",
        "adfa_137",
        "adfa_138",
        "adfa_139",
        "adfa_140",
        "adfa_141",
        "adfa_142",
        "adfa_143",
        "adfa_144",
        "adfa_145",
        "adfa_146",
        "adfa_147",
        "adfa_148",
        "adfa_149",
        "adfa_150",
        "adfa_151",
        "adfa_152",
        "adfa_153",
        "adfa_154",
        "adfa_155",
        "adfa_156",
        "adfa_157",
        "adfa_158",
        "adfa_159",
        "adfa_160",
        "adfa_161",
        "adfa_162",
        "adfa_163",
        "adfa_164",
        "adfa_165",
        "adfa_166",
        "adfa_167",
        "adfa_168",
        "adfa_169",
        "adfa_170",
        "adfa_171",
        "adfa_172",
        "adfa_173",
        "adfa_174",
        "adfa_175",
        "adfa_176",
        "adfa_177",
        "adfa_178",
        "adfb_0",
        "adfb_1",
        "adfb_2",
        "adfb_3",
        "adfb_4",
        "adfb_5",
        "adfb_6",
        "adfb_7",
        "adfb_8",
        "adfb_9",
        "adfb_10",
        "adfb_11",
        "adfb_12",
        "adfb_13",
        "adfb_14",
        "adfb_15",
        "adfb_16",
        "adfb_17",
        "adfb_18",
        "adfb_19",
        "adfb_20",
        "adfb_21",
        "adfb_22",
        "adfb_23",
        "adfb_24",
        "adfb_25",
        "adfb_26",
        "adfb_27",
        "adfb_28",
        "adfb_29",
        "adfb_30",
        "adfb_31",
        "adfb_32",
        "adfb_33",
        "adfb_34",
        "adfb_35",
        "adfb_36",
        "adfb_37",
        "adfb_38",
        "adfb_39",
        "adfb_40",
        "adfb_41",
        "adfb_42",
        "adfb_43",
        "adfb_44",
        "adfb_45",
        "adfb_46",
        "adfb_47",
        "adfb_48",
        "adfb_49",
        "adfb_50",
        "adfb_51",
        "adfb_52",
        "adfb_53",
        "adfb_54",
        "adfb_55",
        "adfb_56",
        "adfb_57",
        "adfb_58",
        "adfb_59",
        "adfb_60",
        "adfb_61",
        "adfb_62",
        "adfb_63",
        "adfb_64",
        "adfb_65",
        "adfb_66",
        "adfb_67",
        "adfb_68",
        "adfb_69",
        "adfb_70",
        "adfb_71",
        "adfb_72",
        "adfb_73",
        "adfb_74",
        "adfb_75",
        "adfb_76",
        "adfb_77",
        "adfb_78",
        "adfb_79",
        "adfb_80",
        "adfb_81",
        "adfb_82",
        "adfb_83",
        "adfb_84",
        "adfb_85",
        "adfb_86",
        "adfb_87",
        "adfb_88",
        "adfb_89",
        "adfb_90",
        "adfb_91",
        "adfb_92",
        "adfb_93",
        "adfb_94",
        "adfb_95",
        "adfb_96",
        "adfb_97",
        "adfb_98",
        "adfb_99",
        "adfb_100",
        "adfb_101",
        "adfb_102",
        "adfb_103",
        "adfb_104",
        "adfb_105",
        "adfb_106",
        "adfb_107",
        "adfb_108",
        "adfb_109",
        "adfb_110",
        "adfb_111",
        "adfb_112",
        "adfb_113",
        "adfb_114",
        "adfb_115",
        "adfb_116",
        "adfb_117",
        "adfb_118",
        "adfb_119",
        "adfb_120",
        "adfb_121",
        "adfb_122",
        "adfb_123",
        "adfb_124",
        "adfb_125",
        "adfb_126",
        "adfb_127",
        "adfb_128",
        "adfb_129",
        "adfb_130",
        "adfb_131",
        "adfb_132",
        "adfb_133",
        "adfb_134",
        "adfb_135",
        "adfb_136",
        "adfb_137",
        "adfb_138",
        "adfb_139",
        "adfb_140",
        "adfb_141",
        "adfb_142",
        "adfb_143",
        "adfb_144",
        "adfb_145",
        "adfb_146",
        "adfb_147",
        "adfb_148",
        "adfb_149",
        "adfb_150",
        "adfb_151",
        "adfb_152",
        "adfb_153",
        "adfb_154",
        "adfb_155",
        "adfb_156",
        "adfb_157",
        "adfb_158",
        "adfb_159",
        "adfb_160",
        "adfb_161",
        "adfb_162",
        "adfb_163",
        "adfb_164",
        "adfb_165",
        "adfb_166",
        "adfb_167",
        "adfb_168",
        "adfb_169",
        "adfb_170",
        "adfb_171",
        "adfb_172",
        "adfb_173",
        "adfb_174",
        "adfb_175",
        "adfb_176",
        "adfb_177",
        "adfb_178",
        "ddf_0",
        "ddf_1",
        "ddf_2",
        "ddf_3",
        "ddf_4",
        "ddf_5",
        "ddf_6",
        "ddf_7",
        "ddf_8",
        "ddf_9",
        "ddf_10",
        "ddf_11",
        "ddf_12",
        "ddf_13",
        "ddf_14",
        "ddf_15",
        "ddf_16",
        "ddf_17",
        "ddf_18",
        "ddf_19",
        "ddf_20",
        "ddf_21",
        "ddf_22",
        "ddf_23",
        "ddf_24",
        "ddf_25",
        "ddf_26",
        "ddf_27",
        "ddf_28",
        "ddf_29",
        "ddf_30",
        "ddf_31",
        "ddf_32",
        "ddf_33",
        "ddf_34",
        "ddf_35",
        "ddf_36",
        "ddf_37",
        "ddf_38",
        "ddf_39",
        "ddf_40",
        "ddf_41",
        "ddf_42",
        "ddf_43",
        "ddf_44",
        "ddf_45",
        "ddf_46",
        "ddf_47",
        "ddf_48",
        "ddf_49",
        "ddf_50",
        "ddf_51",
        "ddf_52",
        "ddf_53",
        "ddf_54",
        "ddf_55",
        "ddf_56",
        "ddf_57",
        "ddf_58",
        "ddf_59",
        "ddf_60",
        "ddf_61",
        "ddf_62",
        "ddf_63",
        "ddf_64",
        "ddf_65",
        "ddf_66",
        "ddf_67",
        "ddf_68",
        "ddf_69",
        "ddf_70",
        "ddf_71",
        "ddf_72",
        "ddf_73",
        "ddf_74",
        "ddf_75",
        "ddf_76",
        "ddf_77",
        "ddf_78",
        "ddf_79",
        "ddf_80",
        "ddf_81",
        "ddf_82",
        "ddf_83",
        "ddf_84",
        "ddf_85",
        "ddf_86",
        "ddf_87",
        "ddf_88",
        "ddf_89",
        "ddf_90",
        "ddf_91",
        "ddf_92",
        "ddf_93",
        "ddf_94",
        "ddf_95",
        "ddf_96",
        "ddf_97",
        "ddf_98",
        "ddf_99",
        "ddf_100",
        "ddf_101",
        "ddf_102",
        "ddf_103",
        "ddf_104",
        "ddf_105",
        "ddf_106",
        "ddf_107",
        "ddf_108",
        "ddf_109",
        "ddf_110",
        "ddf_111",
        "ddf_112",
        "ddf_113",
        "ddf_114",
        "ddf_115",
        "ddf_116",
        "ddf_117",
        "ddf_118",
        "ddf_119",
        "ddf_120",
        "ddf_121",
        "ddf_122",
        "ddf_123",
        "ddf_124",
        "ddf_125",
        "ddf_126",
        "ddf_127",
        "ddf_128",
        "ddf_129",
        "ddf_130",
        "ddf_131",
        "ddf_132",
        "ddf_133",
        "ddf_134",
        "ddf_135",
        "ddf_136",
        "ddf_137",
        "ddf_138",
        "ddf_139",
        "ddf_140",
        "ddf_141",
        "ddf_142",
        "ddf_143",
        "ddf_144",
        "ddf_145",
        "ddf_146",
        "ddf_147",
        "ddf_148",
        "ddf_149",
        "ddf_150",
        "ddf_151",
        "ddf_152",
        "ddf_153",
        "ddf_154",
        "ddf_155",
        "ddf_156",
        "ddf_157",
        "ddf_158",
        "ddf_159",
        "ddf_160",
        "ddf_161",
        "ddf_162",
        "ddf_163",
        "ddf_164",
        "ddf_165",
        "ddf_166",
        "ddf_167",
        "ddf_168",
        "ddf_169",
        "ddf_170",
        "ddf_171",
        "ddf_172",
        "ddf_173",
        "ddf_174",
        "ddf_175",
        "ddf_176",
        "ddf_177",
        "ddf_178",
        "nn_0",
        "nn_1",
        "nn_2",
        "nn_3",
        "nn_4",
        "nn_5",
        "nn_6",
        "nn_7",
        "nn_8",
        "nn_9",
        "nn_10",
        "nn_11",
        "nn_12",
        "nn_13",
        "nn_14",
        "nn_15",
        "nn_16",
        "nn_17",
        "nn_18",
        "nn_19",
        "nn_20",
        "nn_21",
        "nn_22",
        "nn_23",
        "nn_24",
        "nn_25",
        "nn_26",
        "nn_27",
        "nn_28",
        "nn_29",
        "nn_30",
        "nn_31",
        "nn_32",
        "nn_33",
        "nn_34",
        "nn_35",
        "nn_36",
        "nn_37",
        "nn_38",
        "nn_39",
        "nn_40",
        "nn_41",
        "nn_42",
        "nn_43",
        "nn_44",
        "nn_45",
        "nn_46",
        "nn_47",
        "nn_48",
        "nn_49",
        "nn_50",
        "nn_51",
        "nn_52",
        "nn_53",
        "nn_54",
        "nn_55",
        "nn_56",
        "nn_57",
        "nn_58",
        "nn_59",
        "nn_60",
        "nn_61",
        "nn_62",
        "nn_63",
        "nn_64",
        "nn_65",
        "nn_66",
        "nn_67",
        "nn_68",
        "nn_69",
        "nn_70",
        "nn_71",
        "nn_72",
        "nn_73",
        "nn_74",
        "nn_75",
        "nn_76",
        "nn_77",
        "nn_78",
        "nn_79",
        "nn_80",
        "nn_81",
        "nn_82",
        "nn_83",
        "nn_84",
        "nn_85",
        "nn_86",
        "nn_87",
        "nn_88",
        "nn_89",
        "nn_90",
        "nn_91",
        "nn_92",
        "nn_93",
        "nn_94",
        "nn_95",
        "nn_96",
        "nn_97",
        "nn_98",
        "nn_99",
    ]
    print(cfid_names)

    sys.exit()
    # jvasp54_desc = [3920.1349999999998, 2815.1310049999997, 7.595000000000001, -51.5475, 0.0, 55.27627999999999, 4.0, 698.943325, 9.792068783997193, -1635.18, 10.729999999999999, 3.35652, 65.8, 0.0, -62.6025, 957.6367783821725, 3.7295000000000003, 2827.39, 0.0, 84.89999999999999, 240.15, 84.94999999999999, 0.0, -1.5, 0.0, 84.89999999999999, 84.94999999999999, 240.15, 1640.8205, 1.5555050000000001, -7.366720000000001, 57.1475, -10.085, 8.530215, 15.726220000000001, 0.19850985523724304, -6.625000000000001, 29.24425, 31.3939, 29.24425, 66.55543, 65.362, 68.24300000000001, 0.023150132393741984, 0.1975, 6.0, 60.4135665, 193.34500000000003, 463.70450837386375, 21.14859041956059, -1572.5775, 2.0, 11.23, 30.98724122941106, 1642.18, 1.421005, 71.0025, 49.11525862068965, 0.13784637931034482, 17.565, 8.726220000000001, 2817.305, 68.37750000000001, 1.7262200000000005, 0.87622, -2.173995, 113.70897435, 12281.5453818, 247137.89077499998, 1172.755, 9.075, 69.406495, 1654.635, 7.366720000000001, 133.605, 78.5524249868398, 0.7116095942649003, 0.005943593942853046, -68.24300000000001, 2.5, 0.9793499999999997, -1578.1775, 5.455, -5.237106005555556, 660.5102499999999, 1650.90622, 72.72872000000001, 28611.54055, 76.4575, 0.0, -6.803995, 0.6618320411556421, 1.4182692680499998, 145147.5017, 7.8500000000000005, 60.876279999999994, 11.095500000000001, 48.03353448275862, -55.27627999999999, 2.856, 1629.725, 5.0138315784812e-05, -57.1475, 3.7287799999999995, 1.101, 150.38445000000002, 10.085, -1.0105, 1.225, 2813.5755, 1.0423318105124229, 2293.7799999999997, 19097.522, 0.007937747432826298, 64.0025, 2745.3325, 2878.9375, 101.32911482676487, 0.28033499999999995, 1.6496227500000002, 1711.7825, 18.255075325, 0.02053276689864997, 0.017607866379310343, 310.44970785116, -62.643, 1.9084199999999998, 5.48, 50.17550000000001, -62.777499999999996, 0.0, -57.0025, 30.1641, -1641.983995, 10.99948275862069, 56.1525, 0.10948786103517917, 2816.2945, 84.89999999999999, 84.89999999999999, 0.0, 1.5, 0.0, 0.0, 0.0, 293.3, 0.02089125, -56.1525, 0.0, 0.0, 1.145, 1.0105, 7.0, 1706.1825000000001, 1643.5395, 0.0, 2822.785, 8.046005000000001, -6.35622, 1633.4537799999998, 9.25, -1629.725, 0.0578467993898695, -0.8500000000000003, 380.9309028091707, 1650.0300000000002, 77.4525, -1.028995, 25.29382941130023, -61.6325, -2802.48, 889.89725, 2807.935, 29.24425, 31.3939, 29.24425, -1.145, 9.370000000000001, 15256.72, 1578.1775, 1649.18, 191.38807015106877, 36.7933138925483, 21.024237300000003, 1.89144765, 0.003803935062326302, 2.41025, 8.805125, 0.0, 3.0439924568965515, -5.600000000000001, 2.4442402175820317, 9.276104525862069, 2816.16, 6.803995, -2807.085, 0.00332279591692924, 1429.4720000000002, 2.5, 6.1464292235812925, 166.87454366471385, 62.6025, 62.777499999999996, 1.0025, -1634.33, -61.752500000000005, 142.56699999999998, -2.0, 1.5212750000000002, 61.752500000000005, -2807.935, 2821.935, 0.5, 3628.6644000000006, 85.589, 56.18217, -5.6405, 1.772298969060432, -11.23, 2814.738995, -7.50122, 8.686722150490886e-05, -1633.4537799999998, 1643.405, 64.198505, 2802.48, 8.922225000000001, 2813.71, 2052.7281034482758, 0.0, 63.806495, 1572.5775, 5.863793103448276, 20.305, 2823.66122, -69.406495, 15146.368824558456, -2812.565, 7251971.9966, 6671.1166625, 37.911789787682046, 4.63, -3.7287799999999995, 9.2095, 5.600000000000001, -2814.738995, 51.5475, 2.37, 0.0, -8.530215, 0.014562674499019535, 70.8275, 0.2290544181034483, 0.0022216652898575666, 201311.8069, 11.09622, 14.850000000000001, 4.6049999999999995, 9.809074065592162, 25995.74, 65.22749999999999, 546.0522378, 1.0802801724137931, 70.962, 52.05, 0.0, 7.196005, 0.0, -1.5, 1.5, 0.0, -1.5, 0.0, 51.949999999999996, 5.465032327586207, 16597.8554, 1639.8100000000002, 2750.9325000000003, 78.32872, 0.11448921000023374, 57.0025, 69.79850499999999, 0.0015378105423059655, 1635.18, 2.4236745, 27409.016437664686, 6.975625, 7.50122, -2813.71, 3.595, 19.455000000000002, 0.8275, 284.9460518674221, -1640.9550000000002, 543.8299999999999, 71.9725, -68.37750000000001, 16.57622, 32477.7115, 12.909738585000001, -11.095500000000001, 1.5, 12.258994999999999, 0.03209931032285204, 2807.085, 4457.115, 0.775670447321849, 5.6405, 2.5845000000000002, 13.68, 0.4269849, 1641.983995, 0.8980270033090886, 0.15048416618420907, 18.90211, 0.0, 26.050364024999997, 0.043323811194946486, 7.415981791896007, 1.3595000000000002, 1192.8006896551724, -6.490500000000001, 0.11192900495498495, -0.13450000000000006, 1768.5715037725918, 144.97905, 180.08362068965516, 0.13450000000000006, 5.448275862068966, -12.258994999999999, 82.0575, 6231.0123, 0.024262549158410293, 1634.33, 0.0008441026580897188, 1.028995, 6.490500000000001, 0.7276077221063206, 335.33375, 2884.5375, 0.027253135486453788, 15.96, 2812.565, 0.8500000000000003, 11.0, 2.0632566675646555, 506.320727425, 653.501825, 341457.581675, 12.455, 77.96000000000001, 62.643, 8.311086421663079, -1.1634950000000002, 21137.57826935, 82.55050000000001, 71.85249999999999, 2483.7340000000004, 5.775, 1644.55, -2813.5755, 7.1192138971749435, 69.8, 2814.935, 695.41, -1640.8205, 0.0, 0.0, -1.5, 0.0, 52.05, 0.0, 6.35622, 0.5, -1172.755, 12.651005000000001, 2.5, -5.775, -67.2325, -0.87622, -63.806495, 0.011771526250817371, -60.876279999999994, 8.3595, 129.5519531780498, 69.6025, 2806.20878, -4.63, 0.014291806313638033, 0.565282931422759, 7.62567051724138, -7.653995, 0.021613810291072417, 8.225, 14.5655, 10.32201675, 926.6860713499999, -1.7262200000000005, 29.0, 7.653995, 10.22, 0.0, 1.0, -2806.20878, 0.006876380290770823, 1059.56873687272, -7.4716622443750005, 1.1634950000000002, -2750.9325000000003, 10.08572, 21.181220000000003, 0.15195060000000002, 7.5016, 2.566005, 1642.376005, -2745.3325, 315.6456896551724, 2.173995, 100.8775, 76.6025, 0.9793499999999997, -5.48, 1.4107850738570673, 222.90456336686108, -5.455, 13.8145, 14.825, 66.3725, 6.625000000000001, 0.2910490732730619, -4.6049999999999995, 1640.9550000000002, -1639.8100000000002, 61.6325, 9.95122, 67.2325, 0.19600499999999998, 493.98578602500004, 18.34729, 2.90948, 0.38421, 4.8291, 0.00036702036250000004, 31.107971364572855, 40.35850015006895, 33.81278230773782, 27.442348093347576, 29.031053402467077, 32.70423593411773, 33.8212868034083, 31.63413511465427, 27.34772625367516, 22.406762196517683, 17.827100996032556, 14.109735211718409, 11.377663034658609, 9.532974489245941, 8.38977773545681, 7.750782923400996, 7.435226491086258, 7.3047951314886, 7.259457981134694, 7.234593991465978, 7.191171952220209, 7.10936969980216, 6.983234136283518, 6.813491998115835, 6.606556156138735, 6.3699476298321995, 6.111949737406357, 5.840554641721345, 5.562753280442042, 5.284270559621162, 5.009269019089501, 4.741723473307095, 4.4835049355539685, 4.236989007675872, 4.002355446326318, 3.780457889181457, 3.5718765363365748, 3.3757079184528775, 3.1917319200432406, 3.0194294197555944, 2.8581734698575954, 2.7072723520934634, 2.566004647232629, 2.4336771166856868, 2.3098427111424993, 2.1937911158566186, 2.084665819109823, 1.9819603316423313, 1.8855983598802877, 1.7945757063265728, 1.708704716149688, 1.6276776462894125, 1.5508449177171562, 1.4782961398078918, 1.4093313728584902, 1.3439912644693361, 1.2818982785551971, 1.2227288969875043, 1.1666505362887252, 1.113080909719887, 1.062005563331943, 1.013481499760343, 0.9670772174457517, 0.922722984599889, 0.880456147152044, 0.8401344202859035, 0.8015538293342989, 0.7646409893040879, 0.7293550952325507, 0.6956466025401706, 0.663482368156402, 0.6327046327788776, 0.6032571499835175, 0.5750860563378818, 0.5481397010336146, 0.5223685058920683, 0.4977248276476003, 0.474162840855875, 0.45163844017849186, 0.43010914185349375, 0.40953400222281444, 0.38987353725425994, 0.37108966323350956, 0.3531456300654753, 0.3360059690849185, 0.31963645241605765, 0.3040040423870636, 0.2890768613039794, 0.2748241639862195, 0.26121630807388135, 0.24822473253075236, 0.23585686482597876, 0.22407187532953016, 0.2128684898009347, 0.20217864020184506, 0.19197821724743824, 0.1822441412258842, 0.17296745469156216, 0.16415776101471805, 0.1558137081214372, 0.1478508019111657, 0.14025033407082216, 0.13300842141395006, 0.12615658962871196, 0.11966496724011598, 0.11346494767752893, 0.10754604916204527, 0.10195074410238891, 0.09666704262230648, 0.09161522374809389, 0.08679529257184965, 0.08226120819542074, 0.07795928244605843, 0.07384033899028437, 0.06993753043919859, 0.06627728480553671, 0.06276754189334725, 0.05941897432140746, 0.05630206731408097, 0.053310897942728835, 0.0504519626164782, 0.04779576445559402, 0.04524430909089633, 0.042809403948514964, 0.04054863597879217, 0.0383689905126379, 0.036309019693677454, 0.034377393839252944, 0.03251489216949586, 0.030777728765184174, 0.02912316486462411, 0.02753545542043823, 0.026070444640786148, 0.024650133906597527, 0.023323249765785294, 0.0220629559417752, 0.0208536245074859, 0.019737217806024616, 0.01865215549356973, 0.017647753964261804, 0.016682107751752794, 0.015772254975187975, 0.014915422638474401, 0.014090245495657149, 0.013330260542572116, 0.012587835770550908, 0.011909031722660841, 0.01124706176038807, 0.010635665320433954, 0.010045565926137007, 0.009495504352136298, 0.008970311014997725, 0.008475204864645081, 0.008007579941361621, 0.007562639225265496, 0.007146015581144592, 0.0067468041091765085, 0.006375326179504394, 0.006017733031870805, 0.005686192183507038, 0.005366414055006513, 0.005070187814280387, 0.004784711395215762, 0.00451970689641956, 0.004265292203889679, 0.004027893582612218, 0.0038015577528401926, 0.0035885773300258555, 0.0033875789506288743, 0.0031962120737140803, 0.0030180360503216756, 0.0028460340296794294, 0.0026881627849834715, 0.002533397458423383, 0.0023936942042938523, 0.002254451409828794, 0.0021308184550148353, 0.0020080690764488016, 0.0018961323068868324, 0.0017881276280563385, 0.001686600169845146, 0.0015917511238168482, 0.001499711711187711, 0.001416381971762298, 0.0013332474911687626, 0.0012597470745477149, 0.0011870230100647192, 0.00111982703171598, 0.0010563356464552566, 0.0009949511746915054, 0.0009395081429675019, 0.0008841881113496589, 0.0008350542708804784, 0.0007869512530249146, 0.0007416683649549252, 0.0006999143304323217, 0.0006583176428067129, 0.0006219985627023457, 0.000585850490213948, 0.0005522485091562958, 0.0005210099361214151, 0.0004899388934777886, 0.00046287754700801955, 0.00043593825729376174, 0.0004107741764474845, 0.00038758575993360364, 0.00036440320428762535, 0.00034416660863176876, 0.00032425946206446734, 0.00030522713415900366, 0.00028813660912777804, 0.00027107892079968667, 0.0002556506210964693, 0.00024106626305867016, 0.00022651490803607724, 0.0002140157760625324, 0.0002015734396287449, 0.00018968130926941303, 0.00017906791237051614, 0.00016847647417314522, 0.00015875926910019985, 0.00014976349292729867, 0.0001407677167543978, 0.00013283683911714095, 0.0001252135161788115, 0.0001175901932404818, 0.0001111081550736258, 0.00010466256547664451, 9.836296773992605e-05, 9.290755906423003e-05, 8.747022637627909e-05, 8.225825927338249e-05, 7.767105668484217e-05, 7.309486547348693e-05, 6.877117178495834e-05, 6.49218403753317e-05, 6.107938713253205e-05, 5.748106972367381e-05, 5.425760239852221e-05, 5.1038832970593706e-05, 4.8033214386552326e-05, 4.533953405600096e-05, 4.264961567693681e-05, 4.0128942841079264e-05, 3.788280433686523e-05, 3.564023390759905e-05, 3.351720672915424e-05, 3.164834152226392e-05, 2.9783246235490307e-05, 2.7987262910623205e-05, 2.6435754798841647e-05, 2.4888383756674333e-05, 2.3362412708832256e-05, 2.2077283207877924e-05, 2.0796677496067265e-05, 1.9516071784256556e-05, 1.843262767932901e-05, 1.7375435919263294e-05, 1.6318244159197545e-05, 1.538454528394548e-05, 1.4513983134432776e-05, 1.3643420984920074e-05, 1.28361831528833e-05, 1.2120062731754525e-05, 1.1405005654192966e-05, 1.070447707376758e-05, 1.0116791026221343e-05, 9.530965502256163e-06, 8.945139978290965e-06, 8.440094889165827e-06, 7.961390500751554e-06, 7.482686112337279e-06, 7.037236969387659e-06, 6.646434153360268e-06, 6.256287984717674e-06, 5.866141816075067e-06, 5.5445951498717844e-06, 5.227464952996865e-06, 4.910334756121934e-06, 4.6215952362304455e-06, 4.364199436284249e-06, 4.107108655069971e-06, 3.850017873855703e-06, 3.6398172767107617e-06, 3.431961502218667e-06, 3.224105727726565e-06, 3.0323239514352495e-06, 2.864471583241075e-06, 2.6968801970351222e-06, 2.5292888108291753e-06, 2.38758280001076e-06, 2.252828576080556e-06, 2.118074352150357e-06, 1.9873070218973102e-06, 1.878969926545736e-06, 1.7709208568746438e-06, 1.6628717872035514e-06, 1.5645272528732965e-06, 1.4780199632616626e-06, 1.3916276861358355e-06, 1.305235409010005e-06, 1.231130879268041e-06, 1.1622328697765435e-06, 1.093353410220341e-06, 1.024473950664141e-06, 9.684626379750235e-07, 9.137040476882475e-07, 8.589454574014737e-07, 8.051497246177948e-07, 7.616246602912056e-07, 7.182188208101003e-07, 6.748129813289951e-07, 6.332054366658655e-07, 5.988222137752532e-07, 5.645167709690686e-07, 5.302113281628828e-07, 4.978138467476237e-07, 4.707277553295596e-07, 4.4369551711494483e-07, 4.166632789003309e-07, 3.912436560886878e-07, 3.6996618092075196e-07, 3.487291595443731e-07, 3.2749213816799575e-07, 3.073818860942234e-07, 2.907161645854013e-07, 2.74083530461919e-07, 2.5745089633843666e-07, 2.4140490222194203e-07, 2.2838788615490362e-07, 2.1539982649571353e-07, 2.0241176683652344e-07, 1.8949942612503252e-07, 1.7936491367735353e-07, 1.6925679294089357e-07, 1.591486722044336e-07, 1.490405514679736e-07, 1.4081432026836638e-07, 1.3297399655034483e-07, 1.2513445432872395e-07, 1.172949121071031e-07, 1.1050755934001388e-07, 1.0444214439351162e-07, 9.838175675993679e-08, 9.232136912636154e-08, 8.667828880956381e-08, 8.200050068206831e-08, 7.733044546939724e-08, 7.26603902567265e-08, 6.799033504405544e-08, 6.434543368891127e-08, 6.07583861349865e-08, 5.717133858106173e-08, 5.358429102713721e-08, 5.045716984670532e-08, 4.77083965223514e-08, 4.4962223329280006e-08, 4.22160501362086e-08, 3.9530057442393886e-08, 3.743024406297451e-08, 3.5334799641333653e-08, 3.32393552196928e-08, 3.114391079805195e-08, 2.9336722627406347e-08, 2.77418017457707e-08, 2.614826038724578e-08, 2.455471902872086e-08, 2.2964036559008173e-08, 2.1753676266138182e-08, 2.0545944720042237e-08, 1.9338213173946288e-08, 1.813048162785034e-08, 1.7033908734575327e-08, 1.6120553155464416e-08, 1.5208377643049375e-08, 1.4296202130634339e-08, 1.3384026618219303e-08, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.728136747862155, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.808741620703023, 0.0, 0.0, 0.0, 0.0, 2.1916905532774282, 0.0, 0.0, 0.0, 3.560328039605738, 0.0, 0.0, 0.0, 2.9490614850866304, 0.0, 0.9002812950464878, 0.0, 1.2413471930636002, 0.0, 0.0, 4.403763801156143, 0.0, 0.0, 0.0, 2.8439088220268114, 3.6588439559151436, 1.7663448213428563, 0.0, 0.0, 0.0, 0.0, 0.0, 3.62080976335034, 0.0, 2.041843013312849, 0.0, 0.0, 2.489219902379715, 0.0, 0.0, 1.1423599323791904, 3.88872534657552, 1.0810374704078305, 0.0, 1.0245233777498288, 0.0, 0.0, 0.3159044063918733, 5.082119010387053, 0.0, 1.7584557919305308, 0.0, 0.0, 4.089532300756224, 0.0, 0.7805218640753709, 0.0, 0.0, 0.0, 1.4261018503845853, 2.092651305836659, 0.0, 0.779366174681937, 0.0, 2.5613646113665203, 3.4492629328828106, 0.30717020728110783, 1.805789770457263, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.0, 0.0, 4.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.0, 3.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.0, 0.0, 4.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 8.0, 0.0, 0.0, 0.0, 6.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.636363636363633, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.6213561246436925, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.6347902701171704, 0.0, 0.0, 0.0, 0.0, 0.36528175887957137, 0.0, 0.0, 0.0, 0.5933880066009564, 0.0, 0.0, 0.0, 0.4915102475144384, 0.0, 0.15004688250774798, 0.0, 0.20689119884393337, 0.0, 0.0, 0.7339606335260238, 0.0, 0.0, 0.0, 0.4739848036711352, 0.6098073259858573, 0.2943908035571427, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6034682938917234, 0.0, 0.34030716888547485, 0.0, 0.0, 0.4148699837299525, 0.0, 0.0, 0.1903933220631984, 0.64812089109592, 0.18017291173463842, 0.0, 0.17075389629163815, 0.0, 0.0, 0.052650734398645545, 0.8470198350645087, 0.0, 0.2930759653217551, 0.0, 0.0, 0.6815887167927039, 0.0, 0.13008697734589517, 0.0, 0.0, 0.0, 0.2376836417307642, 0.3487752176394432, 0.0, 0.12989436244698951, 0.0, 0.42689410189442006, 0.5748771554804685, 0.05119503454685131, 0.3009649617428772, 0.0, 0.0]
    jvasp54_desc = [
        3920.135,
        2815.131005,
        7.5950000000000015,
        -51.5475,
        0.0,
        55.27628000000001,
        4.0,
        698.943325,
        9.7920687840216,
        -1635.18,
        10.73,
        3.3565199999999997,
        65.8,
        0.0,
        -62.6025,
        957.63677838205,
        3.7295,
        2827.39,
        0.0,
        84.89999999999998,
        240.15,
        84.94999999999997,
        0.0,
        -1.5,
        0.0,
        84.89999999999998,
        84.94999999999997,
        240.15,
        1640.8205,
        1.555505,
        -7.366720000000001,
        57.1475,
        -10.085,
        8.530215,
        15.72622,
        61.6325,
        0.1985098552372,
        -6.625,
        29.24425,
        31.3939,
        29.24425,
        66.55543,
        65.362,
        68.24300000000001,
        0.023150132393749998,
        0.1975,
        6.0,
        60.4135665,
        463.70450837375,
        21.1485904196,
        -1572.5775,
        2.0,
        51.5475,
        30.987241229415,
        1642.18,
        1.4210049999999999,
        71.0025,
        49.1152586207,
        0.1378463793105,
        17.565,
        8.726220000000001,
        2817.305,
        68.37750000000001,
        1.72622,
        0.8762200000000001,
        -2.173995,
        113.70897435,
        12281.545381799999,
        247137.890775,
        1172.755,
        9.075,
        69.40649499999999,
        1654.635,
        7.366720000000001,
        133.605,
        78.552424986669,
        0.7116095942645,
        0.005943593942850001,
        -68.24300000000001,
        0.5652829314225,
        2.5,
        0.97935,
        -1578.1775,
        5.455,
        -5.237106005555,
        660.5102499999998,
        1650.9062199999998,
        72.72872000000002,
        28611.54055,
        76.4575,
        0.0,
        -6.803995,
        1.41826926805,
        145147.5017,
        7.85,
        60.87628,
        11.0955,
        48.03353448275,
        -55.27628000000001,
        2.8560000000000003,
        1629.725,
        5.01383157848e-05,
        -57.1475,
        3.72878,
        1.101,
        150.38445000000004,
        10.085,
        -1.0105,
        5.600000000000001,
        2813.5755,
        6.146429223559999,
        2293.78,
        19097.522,
        0.007937747432819999,
        1.7722989690650002,
        2745.3325,
        2878.9375,
        101.32911482700001,
        0.280335,
        1.64962275,
        1711.7825,
        18.255075325,
        0.020532766898669997,
        0.017607866379310003,
        310.4497078509001,
        -62.643,
        1.90842,
        5.48,
        50.1755,
        -62.7775,
        0.0,
        -57.0025,
        30.1641,
        -1641.9839949999998,
        10.99948275862,
        56.1525,
        0.109487861035,
        2816.2945,
        84.89999999999998,
        84.89999999999998,
        0.0,
        1.5,
        0.0,
        0.0,
        0.0,
        293.3,
        0.02089125,
        -56.1525,
        0.0,
        0.0,
        1.145,
        1.0105,
        7.0,
        1706.1825,
        1643.5395,
        0.0,
        2822.785,
        8.046005000000001,
        -6.3562199999999995,
        1633.4537799999996,
        9.25,
        -1629.725,
        0.057846799389899985,
        -0.8499999999999999,
        380.93090280894995,
        1650.0300000000002,
        77.4525,
        0.0022216652898565,
        -1.028995,
        25.293829411279503,
        -61.6325,
        -2802.48,
        889.89725,
        2807.935,
        29.24425,
        31.3939,
        29.24425,
        -1.145,
        9.37,
        15256.72,
        1578.1775,
        1649.18,
        191.3880701511,
        36.79331389255,
        21.024237300000003,
        1.89144765,
        0.003803935062325,
        2.41025,
        8.805125,
        0.0,
        3.043992456895,
        -5.600000000000001,
        2.4442402175831504,
        2816.16,
        61.75250000000001,
        -2807.085,
        0.00332279591693,
        1429.4720000000002,
        2.5,
        1.0423318105135,
        7.625670517240001,
        62.6025,
        62.7775,
        1.0025,
        -1634.33,
        -61.75250000000001,
        142.56699999999995,
        -2.0,
        1.5212750000000002,
        6.803995,
        -2807.935,
        2821.935,
        0.5,
        3628.6644,
        85.589,
        56.18216999999999,
        -5.6405,
        64.0025,
        -11.23,
        2814.7389949999997,
        -7.50122,
        8.686722150505e-05,
        -1633.4537799999996,
        1643.405,
        64.19850500000001,
        10.32201675,
        8.922225,
        2813.71,
        2052.72810345,
        7.196005,
        63.806495,
        1572.5775,
        5.86379310345,
        20.305,
        2823.66122,
        -69.40649499999999,
        15146.36882458,
        -2812.565,
        7251971.9966,
        82.0575,
        7.653995,
        4.63,
        -3.72878,
        10.22,
        1.225,
        -2814.7389949999997,
        11.23,
        2.37,
        0.0,
        -8.530215,
        2750.932500000001,
        70.8275,
        1.0,
        193.34500000000003,
        201311.8069,
        11.09622,
        14.85,
        4.605,
        9.809074065593801,
        25995.74,
        65.22749999999999,
        546.0522378000002,
        1.080280172415,
        70.962,
        52.05,
        0.0,
        0.0,
        0.0,
        -1.5,
        1.5,
        0.0,
        -1.5,
        0.0,
        51.95,
        5.4650323275862,
        16597.8554,
        1639.81,
        0.014562674499045,
        78.32871999999999,
        57.0025,
        69.79850499999999,
        0.0015378105423075,
        1635.18,
        2.4236744999999997,
        27409.016437700004,
        6.975625,
        7.50122,
        -2813.71,
        3.595,
        19.455,
        0.8275,
        284.94605186755,
        -1640.9550000000004,
        543.8299999999998,
        71.9725,
        -68.37750000000001,
        16.576220000000003,
        32477.7115,
        12.909738585,
        -11.0955,
        1.5,
        12.258995,
        0.032099310322849996,
        2807.085,
        4457.115,
        -2745.3325,
        5.6405,
        2.5845,
        13.68,
        0.4269849,
        1641.9839949999998,
        0.8980270033085,
        0.15048416618410002,
        18.90211,
        0.0,
        26.050364025,
        0.04332381119495,
        7.415981791882999,
        1.3595,
        1192.800689655,
        -6.4905,
        0.1119290049552,
        -0.1345,
        1768.5715037732,
        144.97905,
        180.08362068949998,
        0.1345,
        5.44827586207,
        -12.258995,
        6671.1166625,
        6231.0123,
        0.024262549158430003,
        1634.33,
        0.0008441026580895,
        1.028995,
        66.3725,
        6.4905,
        0.72760772210657,
        335.33375,
        2884.5375,
        0.027253135486465,
        15.96,
        2812.565,
        0.8499999999999999,
        11.0,
        506.320727425,
        2.063256667565,
        653.5018249999998,
        341457.581675,
        12.455,
        77.96,
        62.643,
        8.311086421675,
        -1.1634950000000002,
        21137.57826935,
        71.85249999999998,
        9.27610452585,
        5.775,
        1644.55,
        -2813.5755,
        7.1192138971676515,
        69.8,
        2814.935,
        695.41,
        -1640.8205,
        0.0,
        0.0,
        -1.5,
        0.0,
        52.05,
        0.0,
        6.3562199999999995,
        0.5,
        -1172.755,
        12.651005,
        2.5,
        -5.775,
        -67.2325,
        -0.8762200000000001,
        -63.806495,
        0.011771526250809999,
        -60.87628,
        8.3595,
        129.55195317804998,
        69.6025,
        2806.20878,
        -4.63,
        0.014291806313635,
        0.661832041154,
        166.8745436645,
        -7.653995,
        0.02161381029107,
        8.225,
        14.5655,
        2802.48,
        926.68607135,
        -1.72622,
        29.0,
        37.91178978769426,
        9.2095,
        0.0,
        0.2290544181035,
        -2806.20878,
        0.006876380290764999,
        1059.5687368704998,
        -7.471662244375,
        1.1634950000000002,
        -2750.932500000001,
        10.08572,
        21.18122,
        0.15195060000000002,
        7.5016,
        2.566005,
        1642.376005,
        0.775670447322,
        315.645689655,
        2.173995,
        100.8775,
        76.6025,
        0.97935,
        -5.48,
        1.410785073855,
        222.90456336685003,
        -5.455,
        13.8145,
        14.825,
        82.5505,
        6.625,
        0.29104907327295004,
        -4.605,
        1640.9550000000004,
        -1639.81,
        0.11448921000050001,
        9.95122,
        2483.734,
        67.2325,
        0.19600499999999998,
        493.985786025,
        18.34729,
        2.90948,
        0.38421,
        4.8291,
        0.0003670203625,
        31.107971364572848,
        40.358500150068956,
        33.81278230773782,
        27.442348093347572,
        29.03105340246708,
        32.70423593411773,
        33.8212868034083,
        31.63413511465427,
        27.34772625367516,
        22.406762196517683,
        17.827100996032556,
        14.109735211718409,
        11.377663034658609,
        9.53297448924594,
        8.38977773545681,
        7.750782923400996,
        7.435226491086258,
        7.3047951314886,
        7.259457981134694,
        7.234593991465977,
        7.191171952220208,
        7.10936969980216,
        6.983234136283518,
        6.813491998115835,
        6.606556156138735,
        6.3699476298322,
        6.111949737406357,
        5.840554641721345,
        5.562753280442043,
        5.284270559621162,
        5.009269019089501,
        4.741723473307094,
        4.4835049355539685,
        4.236989007675872,
        4.002355446326319,
        3.780457889181457,
        3.5718765363365748,
        3.375707918452877,
        3.191731920043241,
        3.0194294197555944,
        2.858173469857596,
        2.7072723520934634,
        2.5660046472326288,
        2.4336771166856868,
        2.3098427111424997,
        2.1937911158566186,
        2.0846658191098233,
        1.9819603316423315,
        1.885598359880288,
        1.7945757063265728,
        1.7087047161496884,
        1.6276776462894125,
        1.5508449177171562,
        1.4782961398078918,
        1.4093313728584902,
        1.343991264469336,
        1.2818982785551971,
        1.2227288969875043,
        1.1666505362887252,
        1.1130809097198868,
        1.062005563331943,
        1.013481499760343,
        0.9670772174457516,
        0.9227229845998891,
        0.880456147152044,
        0.8401344202859035,
        0.8015538293342989,
        0.7646409893040879,
        0.7293550952325507,
        0.6956466025401706,
        0.6634823681564019,
        0.6327046327788776,
        0.6032571499835175,
        0.5750860563378818,
        0.5481397010336146,
        0.5223685058920683,
        0.4977248276476003,
        0.474162840855875,
        0.4516384401784919,
        0.43010914185349375,
        0.4095340022228144,
        0.38987353725426,
        0.3710896632335096,
        0.3531456300654753,
        0.3360059690849185,
        0.31963645241605765,
        0.3040040423870636,
        0.2890768613039794,
        0.2748241639862195,
        0.2612163080738813,
        0.2482247325307524,
        0.2358568648259788,
        0.22407187532953016,
        0.2128684898009347,
        0.20217864020184506,
        0.19197821724743824,
        0.1822441412258842,
        0.17296745469156216,
        0.16415776101471805,
        0.1558137081214372,
        0.1478508019111657,
        0.14025033407082216,
        0.13300842141395006,
        0.12615658962871196,
        0.11966496724011598,
        0.11346494767752892,
        0.10754604916204527,
        0.10195074410238893,
        0.09666704262230648,
        0.09161522374809387,
        0.08679529257184965,
        0.08226120819542074,
        0.07795928244605843,
        0.07384033899028437,
        0.06993753043919859,
        0.06627728480553671,
        0.06276754189334725,
        0.05941897432140746,
        0.056302067314080974,
        0.053310897942728835,
        0.0504519626164782,
        0.04779576445559402,
        0.04524430909089633,
        0.042809403948514964,
        0.04054863597879217,
        0.0383689905126379,
        0.036309019693677454,
        0.034377393839252944,
        0.032514892169495865,
        0.030777728765184174,
        0.029123164864624112,
        0.02753545542043823,
        0.026070444640786148,
        0.024650133906597527,
        0.02332324976578529,
        0.0220629559417752,
        0.0208536245074859,
        0.019737217806024616,
        0.01865215549356973,
        0.017647753964261804,
        0.016682107751752794,
        0.015772254975187975,
        0.0149154226384744,
        0.014090245495657147,
        0.013330260542572116,
        0.012587835770550908,
        0.011909031722660841,
        0.011247061760388071,
        0.010635665320433954,
        0.010045565926137008,
        0.009495504352136298,
        0.008970311014997725,
        0.008475204864645081,
        0.00800757994136162,
        0.007562639225265495,
        0.007146015581144592,
        0.006746804109176509,
        0.006375326179504394,
        0.0060177330318708045,
        0.005686192183507038,
        0.005366414055006513,
        0.0050701878142803865,
        0.004784711395215762,
        0.00451970689641956,
        0.004265292203889679,
        0.004027893582612218,
        0.003801557752840193,
        0.0035885773300258555,
        0.0033875789506288747,
        0.0031962120737140803,
        0.003018036050321676,
        0.002846034029679429,
        0.0026881627849834715,
        0.0025333974584233828,
        0.0023936942042938523,
        0.002254451409828794,
        0.0021308184550148357,
        0.0020080690764488016,
        0.0018961323068868324,
        0.0017881276280563387,
        0.001686600169845146,
        0.0015917511238168482,
        0.0014997117111877109,
        0.001416381971762298,
        0.0013332474911687624,
        0.0012597470745477149,
        0.0011870230100647194,
        0.00111982703171598,
        0.0010563356464552566,
        0.0009949511746915054,
        0.0009395081429675019,
        0.0008841881113496588,
        0.0008350542708804784,
        0.0007869512530249146,
        0.0007416683649549253,
        0.0006999143304323217,
        0.0006583176428067129,
        0.0006219985627023457,
        0.0005858504902139481,
        0.0005522485091562958,
        0.0005210099361214151,
        0.0004899388934777887,
        0.0004628775470080196,
        0.0004359382572937618,
        0.00041077417644748453,
        0.00038758575993360364,
        0.00036440320428762535,
        0.0003441666086317688,
        0.00032425946206446734,
        0.00030522713415900366,
        0.00028813660912777804,
        0.00027107892079968667,
        0.00025565062109646935,
        0.00024106626305867016,
        0.00022651490803607724,
        0.0002140157760625324,
        0.0002015734396287449,
        0.00018968130926941303,
        0.00017906791237051614,
        0.00016847647417314522,
        0.00015875926910019982,
        0.0001497634929272987,
        0.0001407677167543978,
        0.00013283683911714098,
        0.0001252135161788115,
        0.0001175901932404818,
        0.00011110815507362581,
        0.00010466256547664452,
        9.836296773992604e-05,
        9.290755906423005e-05,
        8.747022637627909e-05,
        8.225825927338249e-05,
        7.767105668484217e-05,
        7.309486547348693e-05,
        6.877117178495834e-05,
        6.49218403753317e-05,
        6.107938713253206e-05,
        5.7481069723673816e-05,
        5.425760239852221e-05,
        5.10388329705937e-05,
        4.803321438655233e-05,
        4.533953405600096e-05,
        4.264961567693681e-05,
        4.0128942841079264e-05,
        3.788280433686523e-05,
        3.564023390759905e-05,
        3.351720672915424e-05,
        3.164834152226392e-05,
        2.9783246235490307e-05,
        2.7987262910623205e-05,
        2.643575479884165e-05,
        2.488838375667433e-05,
        2.3362412708832256e-05,
        2.2077283207877924e-05,
        2.0796677496067265e-05,
        1.9516071784256562e-05,
        1.843262767932901e-05,
        1.7375435919263298e-05,
        1.6318244159197548e-05,
        1.538454528394548e-05,
        1.4513983134432776e-05,
        1.3643420984920074e-05,
        1.28361831528833e-05,
        1.2120062731754527e-05,
        1.1405005654192966e-05,
        1.070447707376758e-05,
        1.0116791026221345e-05,
        9.530965502256163e-06,
        8.945139978290965e-06,
        8.440094889165827e-06,
        7.961390500751554e-06,
        7.482686112337278e-06,
        7.037236969387659e-06,
        6.646434153360268e-06,
        6.256287984717673e-06,
        5.866141816075067e-06,
        5.544595149871784e-06,
        5.227464952996865e-06,
        4.9103347561219335e-06,
        4.6215952362304455e-06,
        4.364199436284249e-06,
        4.1071086550699715e-06,
        3.850017873855703e-06,
        3.639817276710762e-06,
        3.431961502218667e-06,
        3.2241057277265653e-06,
        3.0323239514352495e-06,
        2.864471583241075e-06,
        2.6968801970351227e-06,
        2.5292888108291757e-06,
        2.38758280001076e-06,
        2.252828576080556e-06,
        2.118074352150357e-06,
        1.9873070218973102e-06,
        1.878969926545736e-06,
        1.7709208568746438e-06,
        1.6628717872035514e-06,
        1.5645272528732962e-06,
        1.4780199632616624e-06,
        1.3916276861358357e-06,
        1.305235409010005e-06,
        1.2311308792680409e-06,
        1.1622328697765437e-06,
        1.093353410220341e-06,
        1.0244739506641408e-06,
        9.684626379750237e-07,
        9.137040476882477e-07,
        8.589454574014737e-07,
        8.051497246177948e-07,
        7.616246602912056e-07,
        7.182188208101002e-07,
        6.748129813289951e-07,
        6.332054366658655e-07,
        5.988222137752532e-07,
        5.645167709690686e-07,
        5.302113281628828e-07,
        4.978138467476236e-07,
        4.707277553295596e-07,
        4.436955171149447e-07,
        4.166632789003309e-07,
        3.912436560886878e-07,
        3.6996618092075206e-07,
        3.487291595443731e-07,
        3.274921381679957e-07,
        3.073818860942234e-07,
        2.907161645854013e-07,
        2.7408353046191896e-07,
        2.5745089633843666e-07,
        2.4140490222194203e-07,
        2.2838788615490362e-07,
        2.1539982649571358e-07,
        2.0241176683652344e-07,
        1.8949942612503247e-07,
        1.7936491367735353e-07,
        1.6925679294089355e-07,
        1.591486722044336e-07,
        1.4904055146797358e-07,
        1.4081432026836635e-07,
        1.3297399655034486e-07,
        1.2513445432872395e-07,
        1.1729491210710311e-07,
        1.1050755934001386e-07,
        1.0444214439351161e-07,
        9.83817567599368e-08,
        9.232136912636155e-08,
        8.66782888095638e-08,
        8.200050068206831e-08,
        7.733044546939724e-08,
        7.26603902567265e-08,
        6.799033504405544e-08,
        6.434543368891128e-08,
        6.075838613498651e-08,
        5.7171338581061735e-08,
        5.358429102713722e-08,
        5.045716984670532e-08,
        4.7708396522351395e-08,
        4.4962223329280006e-08,
        4.22160501362086e-08,
        3.9530057442393886e-08,
        3.7430244062974514e-08,
        3.5334799641333653e-08,
        3.32393552196928e-08,
        3.1143910798051945e-08,
        2.9336722627406347e-08,
        2.77418017457707e-08,
        2.614826038724578e-08,
        2.455471902872086e-08,
        2.2964036559008173e-08,
        2.1753676266138186e-08,
        2.054594472004224e-08,
        1.9338213173946288e-08,
        1.8130481627850343e-08,
        1.7033908734575327e-08,
        1.6120553155464416e-08,
        1.520837764304938e-08,
        1.429620213063434e-08,
        1.3384026618219304e-08,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        9.7281,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        9.8087,
        0.0,
        0.0,
        0.0,
        0.0,
        2.1917,
        0.0,
        0.0,
        0.0,
        3.5603,
        0.0,
        0.0,
        0.0,
        2.9491,
        0.0,
        0.9003,
        0.0,
        1.2413,
        0.0,
        0.0,
        4.4038,
        0.0,
        0.0,
        0.0,
        2.8439,
        3.6588,
        1.7663,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.6208,
        0.0,
        2.0418,
        0.0,
        0.0,
        2.4892,
        0.0,
        0.0,
        1.1424,
        3.8887,
        1.081,
        0.0,
        1.0245,
        0.0,
        0.0,
        0.3159,
        5.0821,
        0.0,
        1.7585,
        0.0,
        0.0,
        4.0895,
        0.0,
        0.7805,
        0.0,
        0.0,
        0.0,
        1.4261,
        2.0927,
        0.0,
        0.7794,
        0.0,
        2.5614,
        3.4493,
        0.3072,
        1.8058,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        8.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        8.0,
        3.0,
        0.0,
        0.0,
        11.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        4.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        24.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        4.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        10.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        8.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        8.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        8.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        4.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        8.0,
        0.0,
        0.0,
        0.0,
        6.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        3.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        12.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        4.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        6.666666666666668,
        0.0,
        0.0,
        0.0,
        0.0,
        2.0,
        0.0,
        0.0,
        0.0,
        4.0,
        0.0,
        0.0,
        0.0,
        4.0,
        0.0,
        1.3333333333333333,
        0.0,
        2.0,
        0.0,
        0.0,
        8.0,
        0.0,
        0.0,
        0.0,
        6.0,
        8.0,
        4.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        10.0,
        0.0,
        6.0,
        0.0,
        0.0,
        8.0,
        0.0,
        0.0,
        4.0,
        14.0,
        4.0,
        0.0,
        4.0,
        0.0,
        0.0,
        1.3333333333333333,
        22.0,
        0.0,
        8.0,
        0.0,
        0.0,
        20.0,
        0.0,
        4.0,
        0.0,
        0.0,
        0.0,
        8.0,
        12.0,
        0.0,
        4.666666666666667,
        0.0,
        16.0,
        22.0,
        2.0,
        12.0,
        0.0,
    ]

    Si = Atoms(lattice_mat=box, coords=coords, elements=elements)
    p = Poscar.from_file(
        "/rk2/knc6/JARVIS-DFT/2D-bulk/mp-2815_bulk_PBEBO/MAIN-RELAX-bulk@mp_2815/CONTCAR"
    )
    Time = time.time()
    cfid = CFID(p.atoms).get_comp_descp()
    tot_time = time.time() - Time
    print("tot_time", tot_time)

    s = p.atoms.pymatgen_converter()
    Time = time.time()
    neighbors_list = s.get_all_neighbors(10.0)
    all_distances = np.concatenate(
        tuple(map(lambda x: [itemgetter(1)(e) for e in x], neighbors_list))
    )
    rdf_dict = {}
    cutoff = 10.0
    intvl = 0.1
    dist_hist, dist_bins = np.histogram(
        all_distances, bins=np.arange(0, cutoff + intvl, intvl), density=False
    )  # equivalent to bon-order
    shell_vol = (
        4.0 / 3.0 * np.pi * (np.power(dist_bins[1:], 3) - np.power(dist_bins[:-1], 3))
    )
    print("pmg", dist_hist)
    number_density = s.num_sites / s.volume
    rdf = dist_hist / shell_vol / number_density / len(neighbors_list)
    tot_time = time.time() - Time
    print("tot_time_pmg", tot_time)
    plt.plot(dist_bins[:-1], rdf)
    plt.savefig("pmgrdf.png")
    plt.close()

    count = 0
    for i, j, k in zip(cfid, jvasp54_desc, cfid_names):
        count = count + 1
        if round(i, 3) != round(j, 3):
            print(count, i, j, k)
    print(len(cfid))
"""
