"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Karpenter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
class Karpenter extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.cluster = props.cluster;
        this.namespace = (_b = props.namespace) !== null && _b !== void 0 ? _b : 'karpenter';
        this.version = props.version;
        /*
         * We create a node role for Karpenter managed nodes, alongside an instance profile for the EC2
         * instances that will be managed by karpenter.
         *
         * We will also create a role mapping in the `aws-auth` ConfigMap so that the nodes can authenticate
         * with the Kubernetes API using IAM.
         */
        this.nodeRole = new aws_iam_1.Role(this, 'NodeRole', {
            assumedBy: new aws_iam_1.ServicePrincipal(`ec2.${aws_cdk_lib_1.Aws.URL_SUFFIX}`),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.nodeRole.roleName],
            instanceProfileName: `${this.cluster.clusterName}-${id}`,
        });
        this.cluster.awsAuth.addRoleMapping(this.nodeRole, {
            username: 'system:node:{{EC2PrivateDNSName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
            ],
        });
        /**
         * For the Karpenter controller to be able to talk to the AWS APIs, we need to set up a few
         * resources which will allow the Karpenter controller to use IAM Roles for Service Accounts
         */
        const namespace = this.cluster.addManifest('namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: this.namespace,
            },
        });
        const serviceAccount = this.cluster.addServiceAccount('karpenter', {
            namespace: this.namespace,
        });
        serviceAccount.node.addDependency(namespace);
        new aws_iam_1.Policy(this, 'ControllerPolicy', {
            roles: [serviceAccount.role],
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ec2:CreateLaunchTemplate',
                        'ec2:DeleteLaunchTemplate',
                        'ec2:CreateFleet',
                        'ec2:RunInstances',
                        'ec2:CreateTags',
                        'iam:PassRole',
                        'ec2:TerminateInstances',
                        'ec2:DescribeLaunchTemplates',
                        'ec2:DescribeInstances',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeInstanceTypes',
                        'ec2:DescribeInstanceTypeOfferings',
                        'ec2:DescribeAvailabilityZones',
                        'ssm:GetParameter',
                    ],
                    resources: ['*'],
                }),
            ],
        });
        /**
         * Finally, we can go ahead and install the Helm chart provided for Karpenter with the inputs
         * we desire.
         */
        this.chart = this.cluster.addHelmChart('karpenter', {
            // This one is important, if we don't ask helm to wait for resources to become available, the
            // subsequent creation of karpenter resources will fail.
            wait: true,
            chart: 'karpenter',
            release: 'karpenter',
            repository: 'https://charts.karpenter.sh',
            namespace: this.namespace,
            version: (_c = this.version) !== null && _c !== void 0 ? _c : undefined,
            createNamespace: false,
            values: {
                serviceAccount: {
                    create: false,
                    name: serviceAccount.serviceAccountName,
                    annotations: {
                        'eks.amazonaws.com/role-arn': serviceAccount.role.roleArn,
                    },
                },
                clusterName: this.cluster.clusterName,
                clusterEndpoint: this.cluster.clusterEndpoint,
                aws: {
                    defaultInstanceProfile: instanceProfile.ref,
                },
            },
        });
        this.chart.node.addDependency(namespace);
    }
    /**
     * addProvisioner adds a provisioner manifest to the cluster. Currently the provisioner spec
     * parameter is relatively free form.
     *
     * @param id - must consist of lower case alphanumeric characters, \'-\' or \'.\', and must start and end with an alphanumeric character
     * @param provisionerSpec - spec of Karpenters Provisioner object.
     */
    addProvisioner(id, provisionerSpec) {
        let m = {
            apiVersion: 'karpenter.sh/v1alpha5',
            kind: 'Provisioner',
            metadata: {
                name: id,
                namespace: this.namespace,
            },
            spec: {},
        };
        m.spec = provisionerSpec;
        let provisioner = this.cluster.addManifest(id, m);
        provisioner.node.addDependency(this.chart);
    }
}
exports.Karpenter = Karpenter;
_a = JSII_RTTI_SYMBOL_1;
Karpenter[_a] = { fqn: "cdk-eks-karpenter.Karpenter", version: "0.0.6" };
//# sourceMappingURL=data:application/json;base64,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