# Pony stubs

Python type hint stubs for [Pony ORM](https://github.com/ponyorm/pony)

## Goals
1. Provide type hints for Pony ORM that support both MyPy and Pyright on their strictest modes
2. Integrate the contents of this package into the official Pony ORM repository (self-deprecation)
3. Focus primarily on the aspects that users of Pony ORM most often run into (defining models, querying them)

## Usage
Install the package with your preferred Python dependency manager:
```sh
pip install pony-stubs

pipenv install pony-stubs

poetry add -D pony-stubs
```

Then define your models:
```python
# We need this to avoid `TypeError: 'type' object is not subscriptable`
# later on when forward declaring relation types with strings
from __future__ import annotations

from pony.orm import Database, Required, select, max, desc

db = Database("sqlite", "store.sqlite", create_db=True)

# Using `db.Entity` directly won't work, as both MyPy and Pyright won't
# allow inheriting a class from a variable. For Pyright this declaration
# is enough misdirection for it not to complain, but MyPy needs an extra
# `type: ignore` comment above each model declaration to work.
DbEntity = db.Entity


class Customer(DbEntity):  # type: ignore
    # If we want the type checkers to know about the autogenerated ID
    # field, we need to annotate it
    id: int
    # Otherwise, using `Required` allows `Customer.email` to be inferred
    # as `str` later on
    email = Required(str, unique=True)
    password = Required(str)
    country = Required(str)
    # Using `Optional` marks the field attribute as `str | None`
    address = Optional(str)
    # When we forward declare a relation by using the class name as a
    # string, we also need to manually annotate the field so that the
    # type checkers can infer it correctly
    orders: Set["Order"] = Set("Order")

class Order(DbEntity):  # type: ignore
    # We can also declare the primary key with Pony constructors and
    # infer the type that way
    id = PrimaryKey(int, auto=True)
    state = Required(str)
    total_price = Required(Decimal)
    # When declaring relationships after a class has been declared,
    # there's no need for annotations
    customer = Required(Customer)

class Product(DbEntity):  # type: ignore
    id: int
    name = Required(str)
    price = Required(Decimal)
```

And use them in your code:
```python
# Here result infers as `QueryResult[Customer]` and all fields in the
# generator expression inside `select` have proper types inferred
result = select(c for c in Customer if c.country != "USA")[:]
result = select(c for c in Customer if count(c.orders) > 1)[:]

# Here result infers as `Decimal`
result = max(p.price for p in Product)

# And here as `Customer | None` (as `.first()` might not find an object)
# Here is also the first time we can't properly infer something:
# `c.orders` is declared as `Set[Order]`, but Pony allows us to access
# `c.orders.total_price` as though it was typed as a plain `Order`.
# As Python doesn't yet support type intersections, we have yet to come
# up with no choice other than to type each extra field of a `Set` as
# `Any`
result = (
    select(c for c in Customer)
    .order_by(lambda c: desc(sum(c.orders.total_price)))
    .first()
)

```

## Limitations
1. We need misdirection with `db.Entity` for `pyright`, and `# type: ignore` for `mypy`
2. When forward declaring relations (relation to a model defined later in the file) we an additional manual annotation (`field: Set["RelatedObject"]`)
3. "Attribute lifting" of related fields is typed as `Any`. Pony would allow us to access attributes of `Set[Order]` as though it's type was `Order`, but python doesn't yet support type intersections so statically typing this seems to be impossible without a plugin (which would only fix the issue for MyPy but not Pyright)
4. `Query.where()` ([docs](https://docs.ponyorm.org/api_reference.html#Query.filter)) loses type information and it's results are typed as `Any`, as python doesn't keep track of generator expressions' initial iterables: `(o.customer for o in Order)` is inferred as `Generator[Customer, None, None]`, without any native way of storing the `Order` type in a generic for inferring.

## Contributing
Contributions are always most welcome! Please run the pre-commit hooks before setting up a pull request, and in case the Github actions fail, please try to fix those issues so the review itself can go as smoothly as possible

The development environment for this package requires `poetry` (https://python-poetry.org/docs/master/#installing-with-the-official-installer)

Using VSCode as the editor is recommended!

### Setting up the repo
1. Clone the repo
    - `git clone git@github.com:Jonesus/pony-stubs.git`
2. Install dependencies
    - `poetry install`
3. Install commit hooks
    - `poetry run pre-commit install --install-hooks`
4. Type ahead!

## License
This project is licensed under the MIT license (see LICENSE.md)
