
import ipaddress
import sys
import argparse
import json

## installed
import pandas as pd
import networkx as nx
from pyvis.network import Network
import numpy as np

from src.logChecker import logChecker

D_TRUE_FALSE = {
    'no':False,
    'yes':True,
}

MY_OPTIONS = """
    const options = {
    "nodes": {
        "borderWidth": null,
        "borderWidthSelected": null,
        "opacity": 0.7,
        "font": {
        "size": 14
        },
        "size": null
    },
    "edges": {
        "color": {
        "inherit": true
        },
        "font": {
        "size": 10,
        "align": "middle"
        },
        "selfReferenceSize": null,
        "selfReference": {
        "angle": 0.7853981633974483
        },
        "smooth": {
        "forceDirection": "none"
        }
    },
    "physics": {
        "minVelocity": 0.5
    }
    }
    """

def addNodes(dfSystem, G, size=5):

    for node in dfSystem.itertuples():

        system = str(node.IP)
        name   = str(node.NAME)
        
        G.add_node(
                    system,
                    label = name+"\n"+system, 
                    system =system, 
                    size  =size
                    )

    return G

def addEdges(dfEdges, G, weight=1):
    """
    Generate Edges from a table with point-to-point interfaces
    The relationship between p2p is generated by the /30 o /31 subnet

    Args:
        dfEdges (_type_): _description_
        G (_type_): _description_
        weight (int, optional): _description_. Defaults to 1.

    Returns:
        _type_: _description_
    """

    if isinstance(G, nx.Graph):
        nodeList = list(G.nodes)
    else:
        nodeList = [x['id'] for x in G.nodes]

    for edge in dfEdges.itertuples():

        name_0   = edge.NAME_0
        name_1   = edge.NAME_1
        port_0   = edge.INTERFACE_PORT_0
        port_1   = edge.INTERFACE_PORT_1
        system_0 = edge.IP_x
        system_1 = edge.IP_y
        network  = edge.network
        label    = f'{port_0}--{port_1}' 

        if system_0 not in nodeList:
            # We dont have name_0, probably an unknown end, We add it as general sink
            G.add_node(system_0, label='na', system='na', size=3, color='red')
        elif system_1 not in nodeList:
            # We dont have name_1, probably an unknown end, We add it as general sink
            G.add_node(system_1, label='na', system='na', size=3, color='red')
        else:
            pass

        G.add_edge(system_0,system_1, port_0=port_0, port_1=port_1, network=network, weight=weight, label=label)

    return G

def getDfSystem(xlsFile, xlsSheetName='sh_rtr_iface'):

    df = pd.read_excel(xlsFile, sheet_name=xlsSheetName)

    # To obataion all the differente system address
    # logChecker must be invoked with paramters -ri both
    dfSystem = df[['NAME','IP']].drop_duplicates().dropna()

    return dfSystem

def getDfEdges(xlsFile, xlsSheetName='sh_rtr_iface'):

    df = pd.read_excel(xlsFile, sheet_name=xlsSheetName)

    dfSystem = df[['NAME','IP']].drop_duplicates().dropna()

    df1 = df.drop(columns='Index')
    df1 = df1.dropna(axis=0, how='all')
    df1 = df1.dropna(axis=1, how='all')
    df1['network'] = df1['INTERFACE_IP'].apply(lambda x: ipaddress.IPv4Interface(x).network).astype(str)
    df1 = df1[['network','INTERFACE_PORT','NAME']]

    dfEdges = (df1.assign(count=df1.groupby(['network']).cumcount()).pivot(index='network', columns='count'))
    dfEdges.columns = ["_".join(str(x) for x in i) for i in dfEdges.columns]
    dfEdges = dfEdges.reset_index()
    dfEdges = dfEdges[~dfEdges.network.str.contains('/32')]
    
    dfEdges = pd.merge(dfEdges,dfSystem,left_on='NAME_0', right_on='NAME', how='left')
    dfEdges = pd.merge(dfEdges,dfSystem,left_on='NAME_1', right_on='NAME', how='left')
    
    dfEdges = dfEdges.replace(np.nan,'na')

    return dfEdges

def getOamHops(xlsFile, G, xlsSheetName='oam_lsp_trc_bgp_lbl'):

    df = pd.read_excel(xlsFile, sheet_name=xlsSheetName)

    dfSystem = df[['NAME','IP']].drop_duplicates().dropna()

    dfHops = df[['NAME','IP','to','ind','ip_hop']].dropna()
    dfHops = pd.merge(dfHops,dfSystem,left_on='ip_hop',right_on='IP', how='left')
    dfHops = dfHops.replace(np.nan,'na')

    nodes = list(dfHops.NAME_x.unique())

    for node in nodes:

        for i,row in enumerate(dfHops[dfHops.NAME_x==node].itertuples()):
            
            if i == 0:

                node_x = row.NAME_x
                ip_x   = row.IP_x 
                node_y = row.NAME_y
                ip_y   = row.ip_hop

                G.add_edge(ip_x, ip_y)

                node_x = node_y
                ip_x   = ip_y

            else:

                node_y = row.NAME_y
                ip_y   = row.ip_hop

                G.add_edge(ip_x, ip_y)

                node_x = node_y
                ip_x   = ip_y    

    return G,dfHops

def main():

    # Buliding the parser ...

    command_parser = argparse.ArgumentParser(prog='logToTopo')

    command_parser.add_argument('-xs'  ,'--xlsSheetName', required=True, type=str, default='sh_rtr_iface', choices=['sh_rtr_iface','oam_lsp_trc_bgp_lbl'], help='Name of excel sheet, where data resides. In general this matches the name of the parsing template.')
    command_parser.add_argument('-lf'  ,'--logsFolder',   required=True, type=str,  help='Name of folder where logs from taskAutom reside.')
    command_parser.add_argument('-tf'  ,'--templateFolder', type=str,  help='Name of folder where parsing templates reside. Default=templ/', default='templ/')

    args = command_parser.parse_args()

    xlsSheetName  = args.xlsSheetName
    folderName    = args.logsFolder
    xlsFile       = folderName.replace("/",".xlsx")
    tempFolder    = args.templateFolder

    ### We start the proces ...
    dicLC = dict(
        preFolder          = folderName,
        postFolder         = '',
        csvTemplate        = '',
        formatJson         = 'yes',
        templateFolder     = tempFolder,
        templateEngine     = 'textFSM',
        templateFolderPost = '',
        routerId           = 'both',
    )

    logChecker.fncRun(dicLC)

    dfSystem = getDfSystem(xlsFile, xlsSheetName)
    G  = nx.Graph()
    G  = addNodes(dfSystem,G)

    if xlsSheetName == 'sh_rtr_iface':
        dfEdges  = getDfEdges(xlsFile, xlsSheetName)
        G        = addEdges(dfEdges, G)
    elif xlsSheetName == 'oam_lsp_trc_bgp_lbl':
        G,dfHops = getOamHops(xlsFile, G)

    nt = Network()
    nt.set_options(MY_OPTIONS)
    nt.from_nx(G)

    nt.show(xlsFile + '.html')

if __name__ == '__main__':

    main()