from .pbt_logger import PBT_Logger


class SelectionStrategy(object):
    def __init__(self, strategy: str):
        """Exploitation Strategies for PBT (Jaderberg et al. 17)."""
        assert strategy in ["t_test", "truncation", "binary_tournament"]
        self.strategy = strategy
        self.truncation_percent = 0.2

    def t_test_selection(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Uniformly sample other network in population. Compare performance using
        Welch’s t-test. If sampled network is better/satisfies t-test, copy it.
        """
        raise NotImplementedError
        copy_info = False
        hyperparams = {}
        ckpt_path = "temp.pt"
        return copy_info, hyperparams, ckpt_path

    def truncation_selection(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Rank networks in population by performance. If network is in bottom 20%
        sample another uniformly from top 20% of population, and copy it.
        """
        most_recent_df = pbt_log.get_most_recent_data()
        top_df, bottom_df = pbt_log.get_truncation_population(
            most_recent_df, self.truncation_percent
        )
        # Check if worker is in top_df - if not sample from top
        copy_info = {}
        bottom_performer = worker_id in bottom_df["worker_id"].tolist()
        if not bottom_performer:
            copy_info["copy_bool"] = False
            copy_info["copy_from_worker_id"] = None
            copy_info["copy_from_pbt_steps"] = None
            copy_info["copy_from_num_updates"] = None
            copy_info["copy_from_performance"] = None
            # Keep same parameters and continue training last checkpoint
            hyperparams = most_recent_df[most_recent_df["worker_id"] == worker_id][
                "hyperparams"
            ].values[0]
            ckpt_path = most_recent_df[most_recent_df["worker_id"] == worker_id][
                "model_ckpt"
            ].values[0]
            copy_info["copy_from_ckpt"] = ckpt_path
            copy_info["copy_from_hyperparams"] = hyperparams
        else:
            # Sample worker to copy from top dataframe (20%)
            copy_from_id = top_df["worker_id"].sample(n=1, random_state=1).values[0]
            hyperparams = top_df[top_df["worker_id"] == copy_from_id][
                "hyperparams"
            ].values[0]
            ckpt_path = top_df[top_df["worker_id"] == copy_from_id][
                "model_ckpt"
            ].values[0]
            pbt_step_id = top_df[top_df["worker_id"] == copy_from_id][
                "pbt_step_id"
            ].values[0]
            num_updates = top_df[top_df["worker_id"] == copy_from_id][
                "num_updates"
            ].values[0]
            performance = top_df[top_df["worker_id"] == copy_from_id][
                pbt_log.eval_metric
            ].values[0]
            copy_info["copy_bool"] = True
            copy_info["copy_from_worker_id"] = copy_from_id
            copy_info["copy_from_pbt_steps"] = pbt_step_id
            copy_info["copy_from_num_updates"] = num_updates
            copy_info["copy_from_ckpt"] = ckpt_path
            copy_info["copy_from_hyperparams"] = hyperparams
            copy_info["copy_from_performance"] = performance
        return copy_info, hyperparams, ckpt_path

    def binary_tournament(self, worker_id: int, pbt_log: PBT_Logger):
        """
        Each member of population randomly selects another member and copies
        all its parameters (hyper & weights) if other member’s score is better.
        """
        raise NotImplementedError
        copy_info = False
        hyperparams = {}
        ckpt_path = "temp.pt"
        return copy_info, hyperparams, ckpt_path

    def select(self, worker_id: int, pbt_log: PBT_Logger):
        if self.strategy == "t_test":
            copy_info, hyperparams, ckpt = self.t_test_selection(worker_id, pbt_log)
        elif self.strategy == "truncation":
            copy_info, hyperparams, ckpt = self.truncation_selection(worker_id, pbt_log)
        elif self.strategy == "binary_tournament":
            copy_info, hyperparams, ckpt = self.binary_tournament(worker_id, pbt_log)
        return copy_info, hyperparams, ckpt
